package net.aihelp.ui.adapter.cs;

import android.content.Context;
import android.graphics.Color;
import android.graphics.Typeface;
import android.os.Build;
import android.text.Html;
import android.text.Spannable;
import android.text.SpannableStringBuilder;
import android.text.Spanned;
import android.text.TextPaint;
import android.text.TextUtils;
import android.text.method.LinkMovementMethod;
import android.text.style.ClickableSpan;
import android.text.style.UnderlineSpan;
import android.view.Gravity;
import android.view.View;
import android.view.ViewGroup;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.TextView;

import net.aihelp.R;
import net.aihelp.common.Const;
import net.aihelp.common.UserProfile;
import net.aihelp.core.ui.adapter.ItemViewDelegate;
import net.aihelp.core.ui.image.Picasso;
import net.aihelp.core.util.permission.PermissionHelper;
import net.aihelp.core.util.viewer.ImageViewer;
import net.aihelp.data.model.cs.BotTag;
import net.aihelp.data.model.cs.BotUrl;
import net.aihelp.data.model.cs.ConversationMsg;
import net.aihelp.data.model.cs.ElvaBotMsg;
import net.aihelp.db.AIHelpDBHelper;
import net.aihelp.db.faq.pojo.RealFaq;
import net.aihelp.ui.adapter.MessageListAdapter;
import net.aihelp.ui.helper.StatisticHelper;
import net.aihelp.utils.DeviceInfoUtil;
import net.aihelp.utils.FastClickValidator;
import net.aihelp.utils.TLog;
import net.aihelp.utils.URLEncodeUtil;

import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;


public abstract class BaseMsgAdapter implements ItemViewDelegate<ConversationMsg> {

    private static final String REGEX_IMAGE = "(http:|https:)(//)((?!\").)*?.(PNG|png|JPG|jpg|JPEG|jpeg)";

    protected final Context mContext;
    protected MessageListAdapter.OnClickedListenerWrapper mWrapper;
    protected List<ConversationMsg> dataList = new ArrayList<>();

    protected boolean isCurrentRtl;

    public BaseMsgAdapter(Context context) {
        this.mContext = context;
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR1) {
            isCurrentRtl = context.getResources()
                    .getConfiguration().getLayoutDirection() == View.LAYOUT_DIRECTION_RTL;
        }
    }

    public void setOnClickedListenerWrapper(MessageListAdapter.OnClickedListenerWrapper mWrapper) {
        this.mWrapper = mWrapper;
    }

    protected TextView getMsg(String botMsg) {
        if (!TextUtils.isEmpty(botMsg) && botMsg.contains("&formUrlTitle=")) {
            Matcher matcher = Pattern.compile("(http:|https:)(//)((?!\").)*?(&formUrlTitle=)*?").matcher(botMsg);
            if (matcher.matches()) {
                botMsg = botMsg + "&fromSdk=1";
                Matcher subMatcher = Pattern.compile("(?<=(formUrlTitle=)).*(?=(&fromSdk=1))").matcher(botMsg);
                String formTitle = botMsg;
                if (subMatcher.find()) {
                    formTitle = URLEncodeUtil.decode(subMatcher.group());
                }
                return getUrl(new BotUrl(formTitle, botMsg), mWrapper);
            }
        }

        TextView textView = new TextView(mContext);
        textView.setText(getUrlSupportedText(botMsg));
        textView.setTextSize(14);
        textView.setLineSpacing(dip2px(mContext, 3), 1.0f);
        textView.setTextIsSelectable(true);
        textView.setTextColor(mContext.getResources().getColor(R.color.aihelp_color_text_title));
        textView.setMovementMethod(LinkMovementMethod.getInstance());
        return textView;
    }

    protected View getRichTextMsg(final String msg) {

        LinearLayout layout = new LinearLayout(mContext);
        layout.setOrientation(LinearLayout.VERTICAL);
        layout.setGravity(Gravity.START);

        try {
            LinkedList<Integer> urlIndexList = new LinkedList<>();
            urlIndexList.add(0);
            Matcher matcher = Pattern.compile(REGEX_IMAGE).matcher(msg);
            while (matcher.find()) {
                urlIndexList.add(matcher.start());
                urlIndexList.add(Math.min(matcher.end(), msg.length()));
            }
            if (urlIndexList.getLast() != msg.length()) {
                urlIndexList.add(msg.length());
            }

            for (int i = 0; i < urlIndexList.size() - 1; i++) {
                final String urlOrRawText = msg.substring(urlIndexList.get(i), urlIndexList.get(i + 1));
                if (Pattern.compile(REGEX_IMAGE).matcher(urlOrRawText).matches()) {
                    final ImageView imageView = new ImageView(mContext);
                    layout.addView(imageView);
                    Picasso.get().load(urlOrRawText).into(imageView);
                    imageView.setOnClickListener(new View.OnClickListener() {
                        @Override
                        public void onClick(View v) {
                            ImageViewer imageViewer = ImageViewer.getDefault(mContext);
                            imageViewer.updateImageResource(urlOrRawText);
                            imageViewer.show();
                        }
                    });

                } else {
                    layout.addView(getMsg(urlOrRawText));
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return layout;
    }

    protected TextView getUrl(final BotUrl botUrl, final MessageListAdapter.OnClickedListenerWrapper wrapper) {
        TextView textView = new TextView(mContext);
        textView.setText(String.format("%s", botUrl.getUrlTitle()));
        textView.setTextSize(14);
        textView.setClickable(true);
        textView.setTextColor(mContext.getResources().getColorStateList(R.color.aihelp_action_text_color));
        textView.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                if (wrapper != null && FastClickValidator.validate()) {
                    wrapper.onUrlClicked(botUrl.getUrlAddress());
                }
            }
        });
        textView.setPadding(0, dip2px(mContext, 5), 0, dip2px(mContext, 5));
        return textView;
    }

    protected TextView getFormUrl(final ElvaBotMsg botMsg, final MessageListAdapter.OnClickedListenerWrapper wrapper) {
        TextView textView = new TextView(mContext);
        textView.setText(String.format("%s", botMsg.getBotFormUrl().getFormTitle()));
        textView.setTextSize(14);
        textView.setClickable(true);
        textView.setTextColor(mContext.getResources().getColorStateList(R.color.aihelp_action_text_color));
        textView.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                if (wrapper != null && FastClickValidator.validate()) {
                    wrapper.onFormUrlClicked(botMsg.getTimeStamp(), getFormattedFormUrl(botMsg));
                }
            }
        });
        textView.setPadding(0, dip2px(mContext, 5), 0, dip2px(mContext, 5));
        return textView;
    }

    protected TextView getAction(int index, final String actionName, final MessageListAdapter.OnClickedListenerWrapper wrapper) {
        TextView textView = new TextView(mContext);
        textView.setText(String.format("%s. %s", index, actionName));
        textView.setTextSize(14);
        textView.setClickable(true);
        textView.setTextColor(mContext.getResources().getColorStateList(R.color.aihelp_action_text_color));
        textView.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                if (wrapper != null && FastClickValidator.validate()
                        && !TextUtils.isEmpty(actionName) && !TextUtils.isEmpty(actionName.trim())) {
                    wrapper.onActionClicked(actionName);
                }
            }
        });
        textView.setPadding(0, dip2px(mContext, 5), 0, dip2px(mContext, 5));
        return textView;
    }

    protected View getFAQLayout(final ElvaBotMsg botMsg, final MessageListAdapter.OnClickedListenerWrapper wrapper) {
        LinearLayout layout = new LinearLayout(mContext);
        layout.setOrientation(LinearLayout.VERTICAL);

        if (botMsg.isHasUrl()) {

            int index = TextUtils.indexOf(botMsg.getBotUrl().getUrlAddress(), "?id=");
            if (index != -1) {
                final String faqContentId = botMsg.getBotUrl().getUrlAddress().substring(index + "?id=".length());
                final RealFaq realFaq = AIHelpDBHelper.getInstance().getFaqByContentId(faqContentId);

                View.OnClickListener onClickListener = new View.OnClickListener() {
                    @Override
                    public void onClick(View v) {
                        if (wrapper != null && FastClickValidator.validate()
                                && !TextUtils.isEmpty(botMsg.getBotUrl().getUrlAddress())) {
                            wrapper.onFaqClicked(botMsg);
                            StatisticHelper.whenBotFAQClicked(botMsg);
                        }
                    }
                };

                TextView botUrl = getUrl(botMsg.getBotUrl(), null);
                botUrl.setOnClickListener(onClickListener);
                botUrl.setPadding(0, dip2px(mContext, 10), 0, dip2px(mContext, 10));
                botUrl.setTypeface(Typeface.defaultFromStyle(Typeface.BOLD));
                layout.addView(botUrl);

                TextView faqContent = new TextView(mContext);
                String str = "<font color='#333333'><u>" + mContext.getResources().getString(R.string.aihelp_view_details) + "</u></font>";
                faqContent.setText(Html.fromHtml(String.format("%s...%s", botMsg.getTemplate().replace("\n", "<br/>"), str)));
                faqContent.setTextSize(14);
                faqContent.setLineSpacing(dip2px(mContext, 3), 1.0f);
                faqContent.setTextColor(Color.parseColor("#333333"));
                faqContent.setPadding(0, dip2px(mContext, 5), 0, dip2px(mContext, 12));
                layout.addView(faqContent);

                faqContent.setOnClickListener(onClickListener);
            }

        }

        if (botMsg.isHasFormUrl()) {

            View divider = new View(mContext);
            divider.setBackgroundColor(Color.parseColor("#EEEEEE"));
            divider.setLayoutParams(new ViewGroup.MarginLayoutParams(ViewGroup.LayoutParams.MATCH_PARENT, dip2px(mContext, 1)));
            divider.setAlpha(0.5F);
            divider.setPadding(0, dip2px(mContext, 5), 0, dip2px(mContext, 5));
            layout.addView(divider);

            TextView formUrl = getFormUrl(botMsg, mWrapper);
            formUrl.setTypeface(Typeface.defaultFromStyle(Typeface.BOLD));
            formUrl.setPadding(0, dip2px(mContext, 12), 0, dip2px(mContext, 5));
            layout.addView(formUrl);
        }

        return layout;
    }

    private String getFormattedFormUrl(ElvaBotMsg botMsg) {
        return String.format("%s&appId=%s&userId=%s&platform=%s&sdkVersion=%s&isTicket=1&%s&hasPermission=%s&fromSdk=1",
                botMsg.getBotFormUrl().getFormAddress(),
                Const.APP_ID,
                UserProfile.USER_ID,
                Const.SDK_PLATFORM,
                Const.SDK_VERSION,
                getTagParams(botMsg),
                PermissionHelper.hasReadPermission(mContext) ? 0 : 1);
    }

    private String getTagParams(ElvaBotMsg botMsg) {
        StringBuilder tagId = new StringBuilder();
        StringBuilder tagName = new StringBuilder();
        if (botMsg.isHasTag()) {
            List<BotTag> tagList = botMsg.getBotTagList();
            for (int i = 0; i < tagList.size(); i++) {
                tagId.append(tagList.get(i).getTagId());
                if (i != tagList.size() - 1) {
                    tagId.append(",");
                }

                tagName.append(tagList.get(i).getTagName());
                if (i != tagList.size() - 1) {
                    tagName.append(",");
                }
            }
        }
        return String.format("tagId=%s&tagName=%s", tagId.toString(), tagName.toString());
    }

    @Override
    public void onDataSourceUpdated(List<ConversationMsg> data) {
        dataList.clear();
        dataList.addAll(data);
    }

    protected SpannableStringBuilder getUrlSupportedText(String msg) {
        if (TextUtils.isEmpty(msg)) return new SpannableStringBuilder("");

        msg = msg.replace("<", "&lt;").replace(">", "&gt;");
        msg = "<span style=\"word-wrap:break-word\">" + msg.replace("\n", "<br/>") + "</span>";
        Spanned htmlMsg = Html.fromHtml(msg);
        SpannableStringBuilder ssb = new SpannableStringBuilder(htmlMsg);

        // 匹配多个 https 连接
        Pattern pattern = Pattern.compile("(http|https):\\/\\/[\\w\\-_]+(\\.[\\w\\-_]+)+([\\w\\-\\.\\|,@?^=%&amp;:/~\\+#]*[\\w\\-\\@?^=%&amp;/~\\+\\|#])?");
        Matcher matcher = pattern.matcher(ssb);
        while (matcher.find()) {
            int start = matcher.start();
            int end = matcher.end();
            final String urlStr = matcher.group();

            ClickableSpan clickableSpan = new ClickableSpan() {
                @Override
                public void onClick(View view) {
                    if (mWrapper != null && FastClickValidator.validate()) {
                        mWrapper.onUrlClicked(urlStr);
                    }
                }

                @Override
                public void updateDrawState(TextPaint dp) {
                    dp.setColor(Color.parseColor("#5C7EFF"));
                }
            };
            ssb.setSpan(clickableSpan, start, end, Spannable.SPAN_EXCLUSIVE_INCLUSIVE);
            ssb.setSpan(new UnderlineSpan(), start, end, Spanned.SPAN_EXCLUSIVE_INCLUSIVE);
        }

        // 匹配 a 标签
//        URLSpan[] urls = ssb.getSpans(0, htmlMsg.length(), URLSpan.class);
//        for (final URLSpan span : urls) {
//            int start = ssb.getSpanStart(span);
//            int end = ssb.getSpanEnd(span);
//            ClickableSpan clickableSpan = new ClickableSpan() {
//                @Override
//                public void onClick(View view) {
//                    if (mWrapper != null) {
//                        mWrapper.onUrlClicked(span.getURL());
//                    }
//                }
//
//                @Override
//                public void updateDrawState(TextPaint dp) {
//                    dp.setColor(Color.parseColor("#5C7EFF"));
//                }
//            };
//            ssb.setSpan(clickableSpan, start, end, Spanned.SPAN_EXCLUSIVE_INCLUSIVE);
//            ssb.setSpan(new UnderlineSpan(), start, end, Spanned.SPAN_EXCLUSIVE_INCLUSIVE);
//        }

        return ssb;
    }

    private static class CustomUrlSpan extends ClickableSpan {

        private MessageListAdapter.OnClickedListenerWrapper wrapper;
        private String url;

        public CustomUrlSpan(MessageListAdapter.OnClickedListenerWrapper wrapper, String url) {
            this.wrapper = wrapper;
            this.url = url;
        }

        @Override
        public void onClick(View widget) {
            if (wrapper != null) {
                wrapper.onUrlClicked(url);
            }
        }
    }

    protected int dip2px(Context context, double dpValue) {
        float density = context.getResources().getDisplayMetrics().density;
        return (int) (dpValue * density + 0.5);
    }

}


