package net.aihelp.init;

import android.content.Context;
import android.text.TextUtils;

import net.aihelp.BuildConfig;
import net.aihelp.common.Const;
import net.aihelp.common.SpKeys;
import net.aihelp.config.AIHelpContext;
import net.aihelp.config.FaqConfig;
import net.aihelp.common.IntentValues;
import net.aihelp.config.OperationConfig;
import net.aihelp.config.ConversationConfig;
import net.aihelp.config.UserConfig;
import net.aihelp.config.enums.PushPlatform;
import net.aihelp.core.net.monitor.NetworkMonitorManager;
import net.aihelp.core.net.mqtt.config.MqttConfig;
import net.aihelp.core.ui.image.Picasso;
import net.aihelp.data.logic.InitPresenter;
import net.aihelp.ui.listener.OnAIHelpInitializedCallback;
import net.aihelp.ui.listener.OnAIHelpSessionCloseCallback;
import net.aihelp.ui.listener.OnAIHelpSessionOpenCallback;
import net.aihelp.ui.listener.OnMessageCountArrivedCallback;
import net.aihelp.ui.listener.OnNetworkCheckResultCallback;
import net.aihelp.ui.listener.OnSpecificFormSubmittedCallback;
import net.aihelp.utils.SchemaUtil;
import net.aihelp.utils.TLog;
import net.aihelp.utils.SpUtil;
import net.aihelp.utils.ToastUtil;

class AIHelpCore {

    private Context mAppContext;
    private InitPresenter mInitPresenter;
    private OnAIHelpInitializedCallback initListener;

    public void init(Context context, String appKey, String domain, String appId, String language) {
        try {
            final String trimApiKey = !TextUtils.isEmpty(appKey) ? appKey.trim() : appKey;
            final String trimDomain = !TextUtils.isEmpty(domain) ? domain.trim() : domain;
            final String trimAppId = !TextUtils.isEmpty(appId) ? appId.trim() : appId;
            SchemaUtil.validateInitializeCredentials(context, trimApiKey, trimDomain, trimAppId);

            mAppContext = context.getApplicationContext();
            AIHelpContext.getInstance().setContext(mAppContext);

            Picasso.setSingletonInstance(new Picasso.Builder(mAppContext).build());
            mInitPresenter = new InitPresenter(mAppContext, trimApiKey, trimDomain, trimAppId, language);

            if (SpUtil.getInstance().getBoolean(SpKeys.TOGGLE_LOG)) TLog.initLog(true);

            NetworkMonitorManager.getInstance().init(mAppContext);
            mInitPresenter.loadUpListeners(initListener);
            mInitPresenter.doInitForAIHelp();
        } catch (IllegalArgumentException e) {
            throw e;
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    void updateSDKLanguage(String sdkLanguage) {
        try {
            if (isInitStillInProgress()) return;
            mInitPresenter.updateSDKLanguage(sdkLanguage);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    void showConversation(ConversationConfig conversationConfig) {
        try {
            if (isInitStillInProgress() || MqttConfig.getInstance().isConnected()) return;
            mInitPresenter.updateConversationFields(conversationConfig);
            mAppContext.startActivity(IntentValues.getShowSupportIntent(mAppContext, conversationConfig));
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    void showAllFAQSections(FaqConfig faqConfig) {
        try {
            if (isInitStillInProgress()) return;
            mInitPresenter.updateConversationFields(faqConfig.getConversationConfig());
            mAppContext.startActivity(IntentValues.getShowFAQIntent(mAppContext, faqConfig));
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    void showFAQSection(String sectionId, FaqConfig faqConfig) {
        try {
            if (isInitStillInProgress()) return;
            mInitPresenter.updateConversationFields(faqConfig.getConversationConfig());
            mAppContext.startActivity(IntentValues.getShowFAQSectionIntent(mAppContext, sectionId, faqConfig));
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    void showSingleFAQ(String mainOrDisplayId, FaqConfig faqConfig) {
        try {
            if (isInitStillInProgress()) return;
            mInitPresenter.updateConversationFields(faqConfig.getConversationConfig());
            mAppContext.startActivity(IntentValues.getShowSingleFAQIntent(mAppContext, mainOrDisplayId, faqConfig));
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    void showOperation(OperationConfig operationConfig) {
        try {
            if (isInitStillInProgress() || MqttConfig.getInstance().isConnected()) return;
            mInitPresenter.updateConversationFields(operationConfig.getConversationConfig());
            mAppContext.startActivity(IntentValues.getShowOperateIntent(mAppContext, operationConfig));
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    void showUrl(String url) {
        try {
            if (mAppContext != null && !TextUtils.isEmpty(url)) {
                mAppContext.startActivity(IntentValues.getUrlIntent(mAppContext, url));
            } else {
                TLog.e("AIHelp", "AIHelpSupport showUrl illegal argument");
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    void updateUserInfo(UserConfig config) {
        try {
            if (isPresenterStillNull()) return;
            mInitPresenter.updateUserProfile(config);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    void setUploadLogPath(String logPath) {
        try {
            if (isPresenterStillNull()) return;
            mInitPresenter.setUploadLogPath(logPath);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    void setNetworkCheckHostAddress(String hostAddress, OnNetworkCheckResultCallback listener) {
        try {
            if (isPresenterStillNull()) return;
            mInitPresenter.setNetworkCheckHostAddress(hostAddress, listener);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    void setPushTokenAndPlatform(String pushToken, PushPlatform pushPlatform) {
        try {
            if (isPresenterStillNull() || pushPlatform == null) return;
            mInitPresenter.postCrmPushTokenInfo(pushToken, pushPlatform.getValue());
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    void startUnreadMessageCountPolling(OnMessageCountArrivedCallback listener) {
        try {
            if (isPresenterStillNull()) return;
            mInitPresenter.startUnreadMessagePolling(listener);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    void setOnSpecificFormSubmittedCallback(OnSpecificFormSubmittedCallback callback) {
        try {
            if (isPresenterStillNull()) return;
            mInitPresenter.setOnSpecificFormSubmittedCallback(callback);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    void setOnAIHelpSessionOpenCallback(OnAIHelpSessionOpenCallback callback) {
        try {
            if (isPresenterStillNull()) return;
            mInitPresenter.setOnAIHelpSessionOpenCallback(callback);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    void setOnAIHelpSessionCloseCallback(OnAIHelpSessionCloseCallback callback) {
        try {
            if (isPresenterStillNull()) return;
            mInitPresenter.setOnAIHelpSessionCloseCallback(callback);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    void setOnAIHelpInitializedCallback(OnAIHelpInitializedCallback listener) {
        try {
            if (mInitPresenter == null) {
                initListener = listener;
                return;
            }
            mInitPresenter.loadUpListeners(listener);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    void runAccelerationForChina() {
        Const.isRunAccelerationForChina = true;
    }

    void enableLogging(boolean enable) {
        TLog.initLog(enable);
    }

    private boolean isInitStillInProgress() {
        try {
            if (!AIHelpContext.successfullyInit.get()) {
                TLog.e("AIHelp init is still in progress!");
                if (BuildConfig.DEBUG) {
                    ToastUtil.makeText(AIHelpContext.getInstance().getContext(),
                            "Initializing...", true);
                }
                return true;
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return false;
    }

    private boolean isPresenterStillNull() {
        try {
            if (mInitPresenter == null) {
                TLog.e("AIHelp method called too early, init presenter is still NULL!");
                return true;
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return false;
    }

    // ===================== getInstance =======================

    public static AIHelpCore getInstance() {
        return Holder.INSTANCE;
    }

    private AIHelpCore() {
    }

    private static class Holder {
        private static final AIHelpCore INSTANCE = new AIHelpCore();
    }

}
