package net.aihelp.db.faq.controller;

import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import android.text.TextUtils;

import net.aihelp.common.Const;
import net.aihelp.db.faq.pojo.DisplayFaq;
import net.aihelp.db.faq.pojo.Section;
import net.aihelp.db.faq.FaqDBHelper;
import net.aihelp.db.faq.tables.SectionTable;
import net.aihelp.db.faq.tables.SubSectionTable;
import net.aihelp.db.util.ContentValuesUtil;
import net.aihelp.core.net.json.JsonHelper;
import net.aihelp.utils.TLog;

import org.json.JSONArray;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.List;

public class SecDBController {

    private final FaqDBHelper dbHelper;
    private final FaqDBController faqController;

    private SecDBController() {
        this.dbHelper = FaqDBHelper.getInstance();
        faqController = FaqDBController.getInstance();
    }

    public boolean isFaqStoredSuccessfully() {
        int amount = 0;
        try {
            SQLiteDatabase database = dbHelper.getReadableDatabase();
            String sql = String.format("select * from %s where %s = '%s'",
                    SectionTable.TABLE_NAME, SectionTable.Columns.FAQ_FILE_NAME, Const.FAQ_FILE);
            Cursor c = database.rawQuery(sql, null);
            amount = c.getCount();
            c.close();
        } catch (Exception e) {
            TLog.e("SecDBController check FAQ store status failed.");
        }
        return amount > 0;
    }

    public List<DisplayFaq> getAllSections() {
        ArrayList<DisplayFaq> secList = new ArrayList<>();
        Cursor cursor = null;
        try {
            SQLiteDatabase database = dbHelper.getReadableDatabase();
            cursor = database.query(SectionTable.TABLE_NAME, null, null,
                    null, null, null, SectionTable.Columns.SECTION_ORDER);
            if (cursor != null && cursor.getCount() > 0 && cursor.moveToFirst()) {
                while (!cursor.isAfterLast()) {
                    secList.add(getFromCursor(false, cursor));
                    cursor.moveToNext();
                }
            }
        }  catch (Exception e) {
            e.printStackTrace();
        } finally {
            if (cursor != null) {
                cursor.close();
            }
        }
        return secList;
    }

    public List<DisplayFaq> getSubSectionsById(String sectionId) {
        ArrayList<DisplayFaq> subSectionList = new ArrayList<>();
        Cursor cursor = null;
        try {
            SQLiteDatabase database = dbHelper.getReadableDatabase();
            cursor = database.query(SubSectionTable.TABLE_NAME, null, "section_id = ?",
                    new String[]{sectionId}, null, null, null);
            if (cursor != null && cursor.getCount() > 0 && cursor.moveToFirst()) {
                while (!cursor.isAfterLast()) {
                    subSectionList.add(getFromCursor(true, cursor));
                    cursor.moveToNext();
                }
            }
        }  catch (Exception e) {
            e.printStackTrace();
        } finally {
            if (cursor != null) {
                cursor.close();
            }
        }
        return subSectionList;
    }

    public void storeSections(JSONArray faqList) {
        dbHelper.clearDatabase();
        SQLiteDatabase database = null;
        try {
            database = dbHelper.getWritableDatabase();
            database.beginTransaction();
            for (int i = 0; i < faqList.length(); i++) {
                JSONObject section = JsonHelper.getJsonObject(faqList, i);
                JSONArray faqs = JsonHelper.getJsonArray(section, "faqs");
                if (faqs.length() > 0) {
                    splitSections(database, section);
                    String sectionId = section.optString("sectionId");
                    faqController.storeFaqs(sectionId, faqs);
                }
            }
            database.setTransactionSuccessful();
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            if (database != null && database.inTransaction()) {
                try {
                    database.endTransaction();
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        }
    }

    private void splitSections(SQLiteDatabase database, JSONObject section) {
        boolean isHasSubSection = !TextUtils.isEmpty(section.optString("sectionBName"));
        database.insertWithOnConflict(SectionTable.TABLE_NAME, null,
                ContentValuesUtil.getSectionValue(section, isHasSubSection), SQLiteDatabase.CONFLICT_IGNORE);
        if (isHasSubSection) {
            database.insertWithOnConflict(SubSectionTable.TABLE_NAME, null,
                    ContentValuesUtil.getSubSectionValue(section), SQLiteDatabase.CONFLICT_IGNORE);
        }

    }

    public synchronized Section getSection(String sectionId) {
        Section section = null;
        Cursor cursor = null;
        try {
            SQLiteDatabase database = this.dbHelper.getReadableDatabase();
            cursor = database.query(SectionTable.TABLE_NAME, null,
                    "section_id = ?", new String[]{sectionId}, null, null, SectionTable.Columns.SECTION_ORDER);
            if (cursor != null && cursor.getCount() > 0 && cursor.moveToFirst()) {
                section = new Section();
                section.setSecId(cursor.getString(cursor.getColumnIndex(SectionTable.Columns.SECTION_ID)));
                section.setSecTitle(cursor.getString(cursor.getColumnIndex(SectionTable.Columns.SECTION_TITLE)));
                section.setHasSubSection("YES".equals(cursor.getString(cursor.getColumnIndex(SectionTable.Columns.HAS_SUB_SECTION))));
            }
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            if (cursor != null) {
                cursor.close();
            }
        }
        return section;
    }

    public Section getSubSection(String subSectionId) {
        Section section = null;
        Cursor cursor = null;
        try {
            SQLiteDatabase database = this.dbHelper.getReadableDatabase();
            cursor = database.query(SubSectionTable.TABLE_NAME, null,
                    "sub_section_id = ?", new String[]{subSectionId}, null, null, null);
            if (cursor != null && cursor.getCount() > 0 && cursor.moveToFirst()) {
                section = new Section();
                section.setSecId(cursor.getString(cursor.getColumnIndex(SubSectionTable.Columns.SUB_ID)));
                section.setSecTitle(cursor.getString(cursor.getColumnIndex(SubSectionTable.Columns.SUB_TITLE)));
                section.setHasSubSection(false);
            }
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            if (cursor != null) {
                cursor.close();
            }
        }
        return section;
    }

    private DisplayFaq getFromCursor(boolean isSubSection, Cursor cursor) {
        DisplayFaq adapterBean = new DisplayFaq();
        if (isSubSection) {
            adapterBean.setId(cursor.getString(cursor.getColumnIndex(SubSectionTable.Columns.SUB_ID)));
            adapterBean.setTitle(cursor.getString(cursor.getColumnIndex(SubSectionTable.Columns.SUB_TITLE)));
            adapterBean.setHasSubSection(false);
            adapterBean.setFaqType(DisplayFaq.FAQ_DISPLAY_SUB_SECTION);
        } else {
            adapterBean.setId(cursor.getString(cursor.getColumnIndex(SectionTable.Columns.SECTION_ID)));
            adapterBean.setTitle(cursor.getString(cursor.getColumnIndex(SectionTable.Columns.SECTION_TITLE)));
            adapterBean.setHasSubSection("YES".equals(cursor.getString(cursor.getColumnIndex(SectionTable.Columns.HAS_SUB_SECTION))));
            adapterBean.setFaqType(DisplayFaq.FAQ_DISPLAY_SECTION);
        }
        return adapterBean;
    }

//    private Section getSectionFromCursor(Cursor cursor) {
//        Section sectionEntity = new Section();
//        sectionEntity.setSecId(cursor.getString(cursor.getColumnIndex(SectionTable.Columns.SECTION_ID)));
//        sectionEntity.setSecTitle(cursor.getString(cursor.getColumnIndex(SectionTable.Columns.SECTION_TITLE)));
//        sectionEntity.setHasSubSection("YES".equals(cursor.getString(cursor.getColumnIndex(SectionTable.Columns.HAS_SUB_SECTION))));
//        return sectionEntity;
//    }
//
//    private SubSection getSubSectionFromCursor(Cursor cursor) {
//        SubSection sectionEntity = new SubSection();
//        sectionEntity.setSubSecId(cursor.getString(cursor.getColumnIndex(SubSectionTable.Columns.SUB_ID)));
//        sectionEntity.setSubSecTitle(cursor.getString(cursor.getColumnIndex(SubSectionTable.Columns.SUB_TITLE)));
//        return sectionEntity;
//    }

    private static final class LazyHolder {
        static final SecDBController INSTANCE = new SecDBController();

        private LazyHolder() {
        }
    }

    public static SecDBController getInstance() {
        return SecDBController.LazyHolder.INSTANCE;
    }

}
