package net.aihelp.db.faq.controller;

import android.content.ContentValues;
import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import android.text.TextUtils;

import net.aihelp.core.util.logger.AIHelpLogger;
import net.aihelp.db.faq.pojo.DisplayFaq;
import net.aihelp.db.faq.pojo.RealFaq;
import net.aihelp.db.faq.FaqDBHelper;
import net.aihelp.db.faq.tables.FaqTable;
import net.aihelp.db.util.ContentValuesUtil;
import net.aihelp.core.net.json.JsonHelper;

import org.json.JSONArray;
import org.json.JSONObject;

import java.util.ArrayList;

public class FaqDBController {

    private final FaqDBHelper dbHelper;

    private FaqDBController() {
        this.dbHelper = FaqDBHelper.getInstance();
    }

    public void storeFaqs(String sectionId, JSONArray faqs) {
        try {
            SQLiteDatabase database = dbHelper.getWritableDatabase();
            for (int i = 0; i < faqs.length(); i++) {
                JSONObject faq = JsonHelper.getJsonObject(faqs, i);
                database.insert(FaqTable.TABLE_NAME, null, ContentValuesUtil.getFaqContentValues(sectionId, faq));
            }
        } catch (Exception e) {
            e.printStackTrace();
            AIHelpLogger.error("FaqDBController#storeFaqs", e);
        }
    }

    private RealFaq cursorToFaq(Cursor cursor) {
        RealFaq faqEntity = new RealFaq();
        faqEntity.setFaqMainId(cursor.getString(cursor.getColumnIndex(FaqTable.Columns.FAQ_MAIN_ID)));
        faqEntity.setFaqDisplayId(cursor.getString(cursor.getColumnIndex(FaqTable.Columns.FAQ_DISPLAY_ID)));
        faqEntity.setFaqContentId(cursor.getString(cursor.getColumnIndex(FaqTable.Columns.FAQ_CONTENT_ID)));
        faqEntity.setFaqTitle(cursor.getString(cursor.getColumnIndex(FaqTable.Columns.FAQ_TITLE)));
        faqEntity.setFaqContent(cursor.getString(cursor.getColumnIndex(FaqTable.Columns.FAQ_CONTENT)));
        faqEntity.setFaqNoHtmlContent(cursor.getString(cursor.getColumnIndex(FaqTable.Columns.FAQ_CONTENT_NO_HTML)));
        faqEntity.setSecId(cursor.getString(cursor.getColumnIndex(FaqTable.Columns.SECTION_ID)));
        faqEntity.setHelpful(cursor.getInt(cursor.getColumnIndex(FaqTable.Columns.IS_HELPFUL)));
        return faqEntity;
    }

    public synchronized void removeFaq(String faqId) {
        if (!TextUtils.isEmpty(faqId)) {
            try {
                String whereClause = "faqId=?";
                String[] whereArgs = new String[]{faqId};
                SQLiteDatabase database = this.dbHelper.getWritableDatabase();
                database.delete(FaqTable.TABLE_NAME, whereClause, whereArgs);
            } catch (Exception e) {
                e.printStackTrace();
            }
        }

    }

    public synchronized RealFaq getFaqById(int idType, String id) {
        if (TextUtils.isEmpty(id)) return null;

        RealFaq faq = null;
        Cursor cursor = null;
        try {
            String selection = idType == RealFaq.TYPE_MAIN_ID ? "faq_main_id = ?" :
                    idType == RealFaq.TYPE_CONTENT_ID ? "faq_content_id = ?" : "faq_display_id = ?";
            SQLiteDatabase database = this.dbHelper.getReadableDatabase();
            cursor = database.query(FaqTable.TABLE_NAME, null,
                    selection, new String[]{id}, null, null, null);
            if (cursor.moveToFirst()) {
                faq = cursorToFaq(cursor);
            }
        } catch (Exception e) {
            e.printStackTrace();
            AIHelpLogger.error("FaqDBController#getFaqById", e);
        } finally {
            if (cursor != null) {
                cursor.close();
            }
        }
        return faq;
    }

    public synchronized int updateHelpfulStatus(String questionId, Boolean state) {
        int returnVal = 0;
        if (TextUtils.isEmpty(questionId)) {
            return returnVal;
        } else {
            ContentValues values = new ContentValues();
            values.put(FaqTable.Columns.IS_HELPFUL, state ? 1 : -1);

            try {
                SQLiteDatabase database = this.dbHelper.getWritableDatabase();
                returnVal = database.update(FaqTable.TABLE_NAME, values, "faq_main_id = ?", new String[]{questionId});
            } catch (Exception e) {
                e.printStackTrace();
                AIHelpLogger.error("FaqDBController#updateHelpfulStatus", e);
            }
            return returnVal;
        }
    }

    public ArrayList<RealFaq> getRealFaqList(String secId) {
        ArrayList<RealFaq> faqList = new ArrayList<>();
        Cursor cursor = null;

        try {
            SQLiteDatabase database = this.dbHelper.getReadableDatabase();
            cursor = database.query(FaqTable.TABLE_NAME, null,
                    "section_id = ?", new String[]{secId}, null, null, null);
            if (cursor != null && cursor.getCount() > 0 && cursor.moveToFirst()) {
                while (!cursor.isAfterLast()) {
                    RealFaq faq = cursorToFaq(cursor);
                    faqList.add(0, faq);
                    cursor.moveToNext();
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
            AIHelpLogger.error("FaqDBController#getRealFaqList", e);
        } finally {
            if (cursor != null) {
                cursor.close();
            }
        }
        return faqList;
    }

    public ArrayList<DisplayFaq> getDisplayFaqList(String secId) {
        ArrayList<DisplayFaq> faqList = new ArrayList<>();
        Cursor cursor = null;

        try {
            SQLiteDatabase database = this.dbHelper.getReadableDatabase();
            cursor = database.query(FaqTable.TABLE_NAME, null,
                    "section_id = ?", new String[]{secId}, null, null, null);
            if (cursor != null && cursor.getCount() > 0 &&cursor.moveToFirst()) {
                while (!cursor.isAfterLast()) {

                    DisplayFaq displayFaq = new DisplayFaq();
                    displayFaq.setId(cursor.getString(cursor.getColumnIndex(FaqTable.Columns.FAQ_MAIN_ID)));
                    displayFaq.setContentId(cursor.getString(cursor.getColumnIndex(FaqTable.Columns.FAQ_CONTENT_ID)));
                    displayFaq.setTitle(cursor.getString(cursor.getColumnIndex(FaqTable.Columns.FAQ_TITLE)));
                    displayFaq.setFaqType(DisplayFaq.FAQ_DISPLAY_QUESTION_LIST);

                    faqList.add(displayFaq);
                    cursor.moveToNext();
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
            AIHelpLogger.error("FaqDBController#getDisplayFaqList", e);
        } finally {
            if (cursor != null) {
                cursor.close();
            }
        }
        return faqList;
    }

    private static final class LazyHolder {
        static final FaqDBController INSTANCE = new FaqDBController();

        private LazyHolder() {
        }

    }

    public static FaqDBController getInstance() {
        return FaqDBController.LazyHolder.INSTANCE;
    }

}
