package net.aihelp.db;

import net.aihelp.core.util.bus.EventBus;
import net.aihelp.data.event.OperateDataPreparedEvent;
import net.aihelp.db.bot.controller.ElvaDBController;
import net.aihelp.db.faq.pojo.DisplayFaq;
import net.aihelp.db.faq.pojo.RealFaq;
import net.aihelp.db.faq.pojo.Section;
import net.aihelp.db.faq.controller.FaqDBController;
import net.aihelp.db.faq.controller.SecDBController;
import net.aihelp.db.op.controller.OperateFaqDBController;
import net.aihelp.db.op.controller.OperateSecDBController;
import net.aihelp.db.op.pojo.OperateFaq;
import net.aihelp.db.op.pojo.OperateSection;

import org.json.JSONArray;

import java.util.ArrayList;
import java.util.List;

public class AIHelpDBHelper {

    private ElvaDBController mElvaDBController;
    private SecDBController mSecDBController;
    private FaqDBController mFaqDBController;
    private OperateSecDBController mOperateSectionController;
    private OperateFaqDBController mOperateFaqController;

    private ArrayList<RealFaq> flatRealFaqList;

    private AIHelpDBHelper() {
        mElvaDBController = ElvaDBController.getInstance();
        mSecDBController = SecDBController.getInstance();
        mFaqDBController = FaqDBController.getInstance();
        mOperateSectionController = OperateSecDBController.getInstance();
        mOperateFaqController = OperateFaqDBController.getInstance();
    }

    public static AIHelpDBHelper getInstance() {
        return LazyHolder.INSTANCE;
    }

    public void updateFlatFaqList() {
        ArrayList<RealFaq> faqs = new ArrayList<>();
        List<DisplayFaq> sections = getAllSections();
        for (DisplayFaq section : sections) {
            if (section.isHasSubSection()) {
                List<DisplayFaq> subSections = getSubSections(section.getId());
                for (DisplayFaq subSection : subSections) {
                    faqs.addAll(getRealFaqList(subSection.getId()));
                }
                continue;
            }
            faqs.addAll(getRealFaqList(section.getId()));
        }
        flatRealFaqList = new ArrayList<>(faqs);
    }

    private ArrayList<RealFaq> getAllFaqs() {
        if (flatRealFaqList == null || flatRealFaqList.size() == 0) {
            updateFlatFaqList();
        } else {
            for (RealFaq faq : flatRealFaqList) {
                faq.clearSearchTerms();
            }
        }
        return flatRealFaqList;
    }

    private ArrayList<RealFaq> getRealFaqList(String sectionId) {
        return mFaqDBController.getRealFaqList(sectionId);
    }

    public ArrayList<DisplayFaq> getDisplayFaqList(String sectionId) {
        return mFaqDBController.getDisplayFaqList(sectionId);
    }

    public ArrayList<DisplayFaq> getMatchedFaqList(String query) {

        ArrayList<RealFaq> sourceFaqs = getAllFaqs();

        ArrayList<DisplayFaq> titleMatchedList = new ArrayList<>();
        ArrayList<DisplayFaq> contentMatchedList = new ArrayList<>();

        for (RealFaq faq : sourceFaqs) {

            if (faq.getFaqTitle().toLowerCase().contains(query.toLowerCase())) {
                faq.updateSearchTerm(query);
                titleMatchedList.add(new DisplayFaq(DisplayFaq.FAQ_DISPLAY_SEARCH, faq.getFaqMainId(), faq.getFaqContentId(), faq.getFaqTitle(), query));
                continue;
            }

            if (faq.getFaqNoHtmlContent().toLowerCase().contains(query.toLowerCase())) {
                faq.updateSearchTerm(query);
                contentMatchedList.add(new DisplayFaq(DisplayFaq.FAQ_DISPLAY_SEARCH, faq.getFaqMainId(), faq.getFaqContentId(), faq.getFaqTitle(), query));
            }
        }

        ArrayList<DisplayFaq> matchedFaqList = new ArrayList<>();
        matchedFaqList.addAll(contentMatchedList);
        matchedFaqList.addAll(0, titleMatchedList);
        return matchedFaqList;
    }

    public ArrayList<DisplayFaq> getMatchedFaqListForAlert(String query) {
        ArrayList<DisplayFaq> titleMatchedList = new ArrayList<>();
        for (RealFaq faq : getAllFaqs()) {
            if (faq.getFaqTitle().toLowerCase().contains(query.toLowerCase())) {
                faq.updateSearchTerm(query);
                titleMatchedList.add(new DisplayFaq(DisplayFaq.FAQ_DISPLAY_SEARCH, faq.getFaqMainId(), faq.getFaqContentId(), faq.getFaqTitle(), query));
                if (titleMatchedList.size() >= 3) {
                    break;
                }
            }
        }
        return titleMatchedList;
    }

    public void afterFaqEvaluated(String faqId, boolean isHelpful) {
        mFaqDBController.updateHelpfulStatus(faqId, isHelpful);
    }

    public boolean isFaqStoredSuccessfully() {
        return mSecDBController.isFaqStoredSuccessfully();
    }

    public boolean isOperateStoredSuccessfully() {
        return mOperateSectionController.isOperateStoredSuccessfully();
    }

    private static class LazyHolder {
        static final AIHelpDBHelper INSTANCE = new AIHelpDBHelper();
    }

    public List<DisplayFaq> getAllSections() {
        return mSecDBController.getAllSections();
    }

    public List<DisplayFaq> getSubSections(String sectionId) {
        return mSecDBController.getSubSectionsById(sectionId);
    }

    public Section getSection(String sectionId) {
        return mSecDBController.getSection(sectionId);
    }

    public Section getSubSection(String sectionId) {
        return mSecDBController.getSubSection(sectionId);
    }

    public RealFaq getFaqByMainId(String faqId) {
        return mFaqDBController.getFaqById(RealFaq.TYPE_MAIN_ID, faqId);
    }

    public RealFaq getFaqByContentId(String faqContentId) {
        return mFaqDBController.getFaqById(RealFaq.TYPE_CONTENT_ID, faqContentId);
    }

    public RealFaq getFaqByDisplayId(String faqDisplayId) {
        return mFaqDBController.getFaqById(RealFaq.TYPE_DISPLAY_ID, faqDisplayId);
    }

    public synchronized void storeFaqList(JSONArray faqList) {
        mSecDBController.storeSections(faqList);
        flatRealFaqList = null;
    }

    public void storeElvaMsg(long timeStamp, String rawResponse) {
        mElvaDBController.storeElvaBotMsg(timeStamp, rawResponse);
    }

    public JSONArray getElvaMsgArray() {
        return mElvaDBController.getElvaMsgArray();
    }

    public void clearElvaMsg() {
        mElvaDBController.clearElvaMsg();
    }

    public void storeOperateFaqs(JSONArray faqs) {
        mOperateSectionController.storeOperateFaqs(faqs);
        EventBus.getDefault().post(new OperateDataPreparedEvent());
    }

    public List<OperateSection> getOperateSectionList() {
        return mOperateSectionController.getOperateSections();
    }

    public List<OperateFaq> getOperateFaqsById(String sectionId) {
        return mOperateFaqController.getOperateFaqsById(sectionId);
    }

    public OperateFaq getOperateFaq(String faqId) {
        return mOperateFaqController.getFaq(faqId);
    }

}
