package net.aihelp.data.logic;

import net.aihelp.common.API;
import net.aihelp.common.Const;
import net.aihelp.common.IntentValues;
import net.aihelp.core.net.mqtt.AIHelpMqtt;
import net.aihelp.core.net.mqtt.callback.IMqttCallback;
import net.aihelp.core.util.bus.EventBus;
import net.aihelp.core.util.logger.AIHelpLogger;
import net.aihelp.data.event.LoadingElvaEvent;
import net.aihelp.data.event.PrepareMessageTimeStampEvent;
import net.aihelp.data.model.MsgStatusEntity;
import net.aihelp.data.model.cs.ConversationMsg;
import net.aihelp.ui.cs.BaseCSFragment;
import net.aihelp.ui.cs.ConversationFragment;
import net.aihelp.ui.cs.ElvaBotFragment;
import net.aihelp.ui.faq.BaseFaqFragment;
import net.aihelp.ui.helper.ConversationHelper;
import net.aihelp.ui.helper.ElvaBotHelper;
import net.aihelp.ui.helper.LoginMqttHelper;
import net.aihelp.ui.helper.LogoutMqttHelper;
import net.aihelp.ui.helper.ResponseMqttHelper;
import net.aihelp.ui.helper.StatisticHelper;
import net.aihelp.core.net.json.JsonHelper;
import net.aihelp.utils.TLog;

import org.json.JSONObject;

import java.util.List;

import androidx.fragment.app.Fragment;


public class MqttCallbackImpl implements IMqttCallback {

    private BaseFaqFragment faqFragment;
    private BaseCSFragment csFragment;
    private ElvaBotFragment csElvaBot;
    private ConversationFragment csConversation;

    public void updateHostView(Fragment fragment) {

        if (fragment instanceof BaseCSFragment) {
            csFragment = (BaseCSFragment) fragment;
            if (fragment instanceof ElvaBotFragment) {
                csElvaBot = (ElvaBotFragment) fragment;
            } else if (fragment instanceof ConversationFragment) {
                csConversation = (ConversationFragment) fragment;
            }
        } else if (fragment instanceof BaseFaqFragment) {
            faqFragment = (BaseFaqFragment) fragment;
        }

    }

    @Override
    public void onMqttConnected() {

    }

    @Override
    public void onMqttResponse(int loginType, String topic, String response) {
        try {

            JSONObject responseObject = new JSONObject(response);
            if (responseObject.has("code") && responseObject.optInt("code") != 200) return;

            switch (topic) {
                // login
                case API.TOPIC_LOGIN:
                    dismissMqttLoading();
                    List<ConversationMsg> mqttReplyMsg = LoginMqttHelper.getLoginResponse(response);
                    csFragment.onMqttLogin(mqttReplyMsg);
                    break;
                case API.TOPIC_LOGOUT:
                    AIHelpMqtt.getInstance().logoutMqttConnection();
                    break;

                // 机器人客诉相关
                case API.TOPIC_BOT_CHAT:
                    if (csElvaBot == null) return;
                    csElvaBot.updateChatList(ElvaBotHelper.getMqttReply(response));
                    break;
                case API.TOPIC_BOT_FAQ:
                    if (csElvaBot == null) return;
                    String timeMillis = responseObject.optString("timeMillis");
                    String isLike = responseObject.optString("isLike");
                    String ticketId = responseObject.optString("ticketId");
                    csElvaBot.updateFAQFeedback(timeMillis, "1".equals(isLike), ticketId);
                    break;

                // 人工客诉相关
                case API.TOPIC_CONVERSATION_SEND:
                    if (csConversation == null) return;
                    MsgStatusEntity statusEntity = JsonHelper.toJavaObject(response, MsgStatusEntity.class);
                    if (statusEntity != null) {
                        boolean postMsgSuccess = "ok".equals(statusEntity.getState());
                        csConversation.updateMsgStatus(postMsgSuccess, statusEntity.getTimeStamp());
                        if (postMsgSuccess) {
                            ResponseMqttHelper.tryUploadLog(responseObject.optBoolean("isUploadLog"));
                        }
                    }
                    break;
                case API.TOPIC_WITHDRAW:
                    if (csConversation == null) return;
                    csConversation.withdrawMsg(ConversationHelper.getWithdrawTimeStamp(response));
                    break;
                case API.TOPIC_CONVERSATION_RECEIVE:

                    if (csConversation != null) {
                        ConversationMsg replyMsg = ConversationHelper.getSupportReplyMsg(response);
                        EventBus.getDefault().post(new PrepareMessageTimeStampEvent(replyMsg));
                        csConversation.updateChatList(replyMsg);
                        return;
                    }

                    if (csElvaBot != null) {
                        csElvaBot.markSupportActionUnread();
                    }
                    break;
                case API.TOPIC_CONVERSATION_FINISHED:
                    if (csConversation != null) {
                        csConversation.onConversationFinished("yes".equals(responseObject.optString("storeReview")));
                        return;
                    }
                    if (csElvaBot != null) {
                        csElvaBot.markSupportActionUnread();
                    }
                    break;
                case API.TOPIC_CONVERSATION_EVALUATE:
                    break;
                case API.TOPIC_SUBMIT_FORM:
                    if (csConversation != null) {
                        String formContent = responseObject.optString("msg");
                        csConversation.onFormSubmitted(formContent);
                    }
                    if (csElvaBot != null) csElvaBot.syncLogoutTypeToServer();

                    ResponseMqttHelper.setFormSubmitStatus(true);
                    LogoutMqttHelper.updateType(LogoutMqttHelper.LOGOUT_TYPE_FORM_SUBMIT);
                    StatisticHelper.whenFormEventHappened(StatisticHelper.getClickedFormTimeStamp(), StatisticHelper.FORM_ACTION_SUBMITTED);
                    ResponseMqttHelper.tryUploadLog(responseObject.getBoolean("isUploadLog"));

                    if (Const.sSpecificFormSubmittedListener != null && responseObject.getBoolean("isSpecificForm")) {
                        Const.sSpecificFormSubmittedListener.onFormSubmitted();
                    }
                    break;

                // FAQ 未读提醒
                case API.TOPIC_FAQ_NOTIFICATION:
                    if (faqFragment != null) {
                        faqFragment.showSupportActionUnread();
                    }
                    break;
            }
        } catch (Exception e) {
            TLog.e("onMqttResponse, Exception " + e.toString());
            AIHelpLogger.error(response, e);
        }
    }

    @Override
    public void onMqttFailure(String message) {
        showMqttLoading();
    }

    @Override
    public void onMqttException() {

    }

    @Override
    public void showMqttLoading() {
        EventBus.getDefault().post(new LoadingElvaEvent(IntentValues.SHOW_CS_LOADING));
    }

    @Override
    public void dismissMqttLoading() {
        EventBus.getDefault().post(new LoadingElvaEvent(IntentValues.HIDE_CS_LOADING));
    }

    private static final class LazyHolder {
        static final MqttCallbackImpl INSTANCE = new MqttCallbackImpl();

        private LazyHolder() {
        }

    }

    private MqttCallbackImpl() {

    }

    public static IMqttCallback getInstance() {
        return MqttCallbackImpl.LazyHolder.INSTANCE;
    }

}
