package net.aihelp.data.logic;

import android.content.Context;
import android.text.TextUtils;
import android.util.Log;

import net.aihelp.common.API;
import net.aihelp.common.ConfigValues;
import net.aihelp.common.Const;
import net.aihelp.common.SpKeys;
import net.aihelp.common.UserProfile;
import net.aihelp.config.AIHelpContext;
import net.aihelp.config.ConversationConfig;
import net.aihelp.config.UserConfig;
import net.aihelp.core.mvp.AbsPresenter;
import net.aihelp.core.mvp.IView;
import net.aihelp.core.net.http.callback.ReqCallback;
import net.aihelp.core.net.json.JsonHelper;
import net.aihelp.core.util.logger.AIHelpLogger;
import net.aihelp.data.local.InitRepository;
import net.aihelp.data.model.InitEntity;
import net.aihelp.data.model.SDKConfigEntity;
import net.aihelp.data.model.UpdateCDNFileEntity;
import net.aihelp.ui.helper.MessageSyncHelper;
import net.aihelp.ui.helper.ResponseMqttHelper;
import net.aihelp.ui.helper.SkinHelper;
import net.aihelp.ui.listener.OnAIHelpInitializedCallback;
import net.aihelp.ui.listener.OnAIHelpSessionCloseCallback;
import net.aihelp.ui.listener.OnAIHelpSessionOpenCallback;
import net.aihelp.ui.listener.OnMessageCountArrivedCallback;
import net.aihelp.ui.listener.OnNetworkCheckResultCallback;
import net.aihelp.ui.listener.OnSpecificFormSubmittedCallback;
import net.aihelp.utils.DateFormatUtil;
import net.aihelp.utils.DeviceUuidFactory;
import net.aihelp.utils.LocalizeHelper;
import net.aihelp.utils.SpUtil;
import net.aihelp.utils.TLog;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;


public class InitPresenter extends AbsPresenter<IView, InitRepository> {

    private OnAIHelpInitializedCallback mInitListener;

    public InitPresenter(Context context, String appKey, String domain, String appId, String language) {
        super(context);
        mRepo.saveInitConfig(appKey, domain, appId, language);
    }

    public void updateUserProfile(UserConfig config) {
        if (config == null) return;
        mRepo.saveUserProfileConfig(config);
        Const.TOGGLE_FETCH_MESSAGE = true;
        ResponseMqttHelper.resetFlagsWhenUserInfoUpdated();
        if (config.isSyncCrmInfo()) {
            JSONObject json = new JSONObject();
            try {
                json.put("userName", config.getUserName());
                json.put("uId", config.getUserId());
                json.put("tags", config.getUserTags());
                post(API.SYNC_VIP_INFO, json, null);
            } catch (JSONException e) {
                e.printStackTrace();
            }
        }
        if (config.getPushToken() != null) {
            postCrmPushTokenInfo(config.getPushToken(), config.getPushPlatform().getValue());
        }
    }

    public void updateConversationFields(ConversationConfig config) {
        mRepo.updateConversationFields(config.getWelcomeMessage(), config.getStoryNode());
    }

    public void doInitForAIHelp() {
        goLogDauStatus();
        requestInit();
        MessageSyncHelper.syncLogMessage();
    }

    private void goLogDauStatus() {
        long initTime = mSp.getLong(SpKeys.LOG_DAU_TIME);
        if (!DateFormatUtil.isToday(initTime)) {
            try {
                JSONObject dauParams = new JSONObject();
                // DO NOT try to change this deviceId EVER, or the DAU data will be messed, which will
                // results in pricing-related issues.
                dauParams.put("deviceid", DeviceUuidFactory.id(mContext));
                post(API.LOG_DAU_URL, dauParams, new ReqCallback<String>() {
                    @Override
                    public void onReqSuccess(String result) {
                        mRepo.saveDAULogTime(System.currentTimeMillis());
                    }
                });
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
    }

    private void requestInit() {
        long lastInitTime = mSp.getLong(SpKeys.LAST_INIT_TIME, System.currentTimeMillis());
        int requestLimit = mSp.getInt(SpKeys.INIT_REQUEST_LIMIT);
        String cachedInitData = mSp.getString(SpKeys.CACHE_INIT_RESPONSE);

        if (TextUtils.isEmpty(cachedInitData) || requestLimit < 0 || System.currentTimeMillis() - lastInitTime > requestLimit) {
            get(API.INIT_URL, null, new ReqCallback<String>() {
                @Override
                public void onReqSuccess(String result) {
                    cacheInitResponse(result, System.currentTimeMillis());
                }
            });
        } else {
            cacheInitResponse(cachedInitData, lastInitTime);
        }
    }

    private void cacheInitResponse(String initResponse, long lastInitTime) {
        InitEntity initEntity = JsonHelper.toJavaObject(initResponse, InitEntity.class);
        if (initEntity != null && initEntity.isFlag()) {
            mRepo.prepareInitData(initEntity, initResponse, lastInitTime);
            if (!TextUtils.isEmpty(API.CONFIG_FILE_URL)) {
                get(API.CONFIG_FILE_URL, null, new ReqCallback<SDKConfigEntity>() {
                    @Override
                    public void onReqSuccess(SDKConfigEntity configEntity) {
                        if (configEntity != null) {
                            mRepo.saveSDKConfig(configEntity);
                            SkinHelper.prepareSkinResources(mContext, configEntity.getSdkConfigImg(), configEntity.getSdkConfigTopic());
                        }
                    }
                });
            }
            LocalizeHelper.goFetchLocalizeData(mContext);
            AIHelpContext.successfullyInit.set(true);
            if (mInitListener != null) {
                mInitListener.onAIHelpInitialized();
            }
        } else {
            AIHelpLogger.error("init error", new UnknownError(initResponse));
            Log.e("AIHelp", String.format("AIHelp SDK init failed, %s", initResponse));
        }
    }

    public void postCrmPushTokenInfo(final String pushToken, final int pushPlatform) {
        if (Const.TOGGLE_CRM_TOKEN && !TextUtils.isEmpty(UserProfile.USER_ID) &&
                !TextUtils.isEmpty(pushToken) && isValidPushPlatform(pushPlatform)) {
            mRepo.saveMqttPushInfo(pushToken, pushPlatform);
            final String latelyCrmPushInfo = String.format("%s|%s|%s", UserProfile.USER_ID, pushToken, pushPlatform);
            String cachedCrmPushInfo = mSp.getString(SpKeys.PUSH_INFO_CRM);
            if (!latelyCrmPushInfo.equals(cachedCrmPushInfo)) {
                try {
                    JSONObject params = new JSONObject();
                    params.put("token", pushToken);
                    params.put("pushTypeId", pushPlatform);
                    params.put("playerId", UserProfile.USER_ID);
                    params.put("playerName", UserProfile.USER_NAME);
                    params.put("language", Const.TARGET_LAN);
                    params.put("deviceId", DeviceUuidFactory.id(mContext));
                    post(API.CRM_TOKEN_URL, params, new ReqCallback<String>() {
                        @Override
                        public void onReqSuccess(String result) {
                            mSp.put(SpKeys.PUSH_INFO_CRM, latelyCrmPushInfo);
                        }
                    });
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        }
    }

    private boolean isValidPushPlatform(int pushPlatform) {
        int[] platformArray = new int[]{ConfigValues.PUSH_PLATFORM_APNS, ConfigValues.PUSH_PLATFORM_FIREBASE,
                ConfigValues.PUSH_PLATFORM_JPUSH, ConfigValues.PUSH_PLATFORM_GETUI
        };
        for (int p : platformArray) {
            if (p == pushPlatform) return true;
        }
        return false;
    }

    public void updateSDKLanguage(final String sdkLanguage) {
        if (!TextUtils.isEmpty(sdkLanguage)) {
            mRepo.updateSDKLanguage(sdkLanguage);
            get(API.UPDATE_FILE_NAME_URL, null, new ReqCallback<UpdateCDNFileEntity>() {
                @Override
                public void onReqSuccess(UpdateCDNFileEntity result) {
                    if (result == null) return;

                    mRepo.updateFileNameWhenLanguageUpdated(sdkLanguage, result);
                    LocalizeHelper.goFetchLocalizeData(mContext);

                    if (!TextUtils.isEmpty(result.getConfigFileName())) {
                        get(result.getConfigFileName(), null, new ReqCallback<SDKConfigEntity>() {
                            @Override
                            public void onReqSuccess(SDKConfigEntity result) {
                                mRepo.updateNameAndWelcome(result.getSdkText());
                            }
                        });
                    }

                }
            });
        }
    }

    public void setUploadLogPath(String logPath) {
        if (TextUtils.isEmpty(logPath)) return;
        if (logPath.endsWith(".txt") || logPath.endsWith(".log") || logPath.endsWith(".bytes")) {
            mRepo.setUploadLogPath(logPath);
        } else {
            TLog.e("Unsupported type, expected .txt, .log or .bytes file");
        }
    }

    public void setNetworkCheckHostAddress(String hostAddress, OnNetworkCheckResultCallback listener) {
        if (TextUtils.isEmpty(hostAddress)) {
            hostAddress = "aihelp.net";
        } else {
            hostAddress = hostAddress.replace("http://", "")
                    .replace("https://", "");
        }
        mRepo.setNetworkCheckHostAddress(hostAddress);
        Const.sCheckResultListener = listener;
    }

    private void fetchUnreadMessageCount() {
        try {
            TLog.e("fetchUnreadMessageCount()");
            // Request fetch api ONLY WHEN there is an active ticket
            if (Const.IS_SDK_SHOWING || !Const.TOGGLE_FETCH_MESSAGE) return;
            JSONObject params = new JSONObject();
            params.put("appid", Const.APP_ID);
            params.put("uid", UserProfile.USER_ID);
            get(API.FETCH_NEW_MSG, params, new ReqCallback<String>() {
                @Override
                public void onReqSuccess(String result) {
                    try {
                        if (!TextUtils.isEmpty(result)) {
                            JSONObject jsonObject = new JSONObject(result);
                            Const.TOGGLE_FETCH_MESSAGE = jsonObject.optBoolean("isHaveChat");
                            int newCount = jsonObject.optInt("cs_message_count");
                            int cachedCount = Math.max(0, mSp.getInt(UserProfile.USER_ID));
                            if (Const.sMessageListener != null) {
                                Const.sMessageListener.onMessageCountArrived(Math.max(0, newCount - cachedCount));
                            }
                        }
                    } catch (Exception e) {
                        TLog.e("Unread msg count polling failed, because " + e.toString());
                    }
                }
            });
        } catch (JSONException e) {
            e.printStackTrace();
        }
    }

    private ScheduledFuture<?> unreadMsgFuture;

    public void startUnreadMessagePolling(OnMessageCountArrivedCallback listener) {
        if (SpUtil.getInstance().getBoolean(SpKeys.TOGGLE_LOG)) Const.LIMIT_CHECKING_UNREAD = 10;
        if (listener != null && unreadMsgFuture == null) {
            if (Const.TOGGLE_OPEN_UNREAD_MSG && Const.LIMIT_CHECKING_UNREAD > 0 && !TextUtils.isEmpty(UserProfile.USER_ID)) {
                Const.sMessageListener = listener;
                unreadMsgFuture = Executors.newScheduledThreadPool(1).scheduleAtFixedRate(new Runnable() {
                    @Override
                    public void run() {
                        fetchUnreadMessageCount();
                    }
                }, 0, Const.LIMIT_CHECKING_UNREAD, TimeUnit.SECONDS);
            }
        }
    }

    public void loadUpListeners(OnAIHelpInitializedCallback listener) {
        if (listener != null && mInitListener == null) {
            mInitListener = listener;
        }
    }

    public void setOnSpecificFormSubmittedCallback(OnSpecificFormSubmittedCallback callback) {
        if (callback != null) {
            Const.sSpecificFormSubmittedListener = callback;
        }
    }

    public void setOnAIHelpSessionOpenCallback(OnAIHelpSessionOpenCallback callback) {
        if (callback != null) {
            Const.sSessionOpenListener = callback;
        }
    }

    public void setOnAIHelpSessionCloseCallback(OnAIHelpSessionCloseCallback callback) {
        if (callback != null) {
            Const.sSessionCloseListener = callback;
        }
    }

}
