package net.aihelp.data.logic;

import android.content.Context;
import android.text.TextUtils;

import net.aihelp.common.API;
import net.aihelp.common.Const;
import net.aihelp.common.UserProfile;
import net.aihelp.core.mvp.AbsPresenter;
import net.aihelp.core.mvp.IRepository;
import net.aihelp.core.net.http.callback.ReqCallback;
import net.aihelp.core.net.mqtt.AIHelpMqtt;
import net.aihelp.core.net.mqtt.callback.IMqttCallback;
import net.aihelp.core.net.mqtt.config.MqttConfig;
import net.aihelp.core.util.concurrent.ApiExecutor;
import net.aihelp.core.util.concurrent.ApiExecutorFactory;
import net.aihelp.db.AIHelpDBHelper;
import net.aihelp.db.faq.pojo.DisplayFaq;
import net.aihelp.db.faq.pojo.RealFaq;
import net.aihelp.db.faq.pojo.Section;
import net.aihelp.ui.faq.BaseFaqFragment;
import net.aihelp.ui.faq.QuestionContentFragment;
import net.aihelp.ui.faq.QuestionListFragment;
import net.aihelp.ui.faq.SectionListFragment;
import net.aihelp.ui.helper.StatisticHelper;
import net.aihelp.core.net.json.JsonHelper;
import net.aihelp.utils.ListUtil;
import net.aihelp.utils.LocalizeHelper;
import net.aihelp.utils.Styles;
import net.aihelp.utils.TLog;

import org.json.JSONArray;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.List;

public class FaqPresenter extends AbsPresenter<BaseFaqFragment, IRepository> {

    private static final ApiExecutor sApiExecutor = ApiExecutorFactory.getHandlerExecutor();

    public FaqPresenter(Context context) {
        super(context);
    }

    private boolean isDataSourcePrepared(final String sectionOrFaqId) {
        if (!AIHelpDBHelper.getInstance().isFaqStoredSuccessfully()) {

            if (!isNetworkAvailable()) {
                mView.showNetError();
                return false;
            }

            mView.showLoading();
            get(LocalizeHelper.getUrl(LocalizeHelper.FLAG_FAQ), null, new ReqCallback<String>() {
                @Override
                public void onReqSuccess(String result) {
                    try {
                        if (TextUtils.isEmpty(result)) {
                            getFaqFromApiAfterLocalizeFailed(sectionOrFaqId);
                            return;
                        }

                        JSONArray jsonArray = JsonHelper.getJsonArray(new JSONObject(result), "faqlist");
                        if (jsonArray.length() > 0) {
                            storeFaqsAfterDataPrepared(sectionOrFaqId, jsonArray);
                        } else {
                            getFaqFromApiAfterLocalizeFailed(sectionOrFaqId);
                        }
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                }

                @Override
                public boolean onFailure(int errorCode, String errorMsg) {
                    getFaqFromApiAfterLocalizeFailed(sectionOrFaqId);
                    return false;
                }

            });

            return false;
        }
        return true;
    }

    private void getFaqFromApiAfterLocalizeFailed(final String sectionOrFaqId) {
        get(API.FAQ_URL, null, new ReqCallback<JSONObject>() {
            @Override
            public void onReqSuccess(JSONObject jsonObject) {
                JSONArray faqList = JsonHelper.getJsonArray(jsonObject, "faqlist");
                if (faqList.length() > 0) {
                    storeFaqsAfterDataPrepared(sectionOrFaqId, faqList);
                } else {
                    mView.showEmpty();
                }
            }
        });
    }

    private void storeFaqsAfterDataPrepared(final String sectionOrFaqId,
                                            final JSONArray faqList) {
        sApiExecutor.runAsync(new Runnable() {
            @Override
            public void run() {
                AIHelpDBHelper.getInstance().storeFaqList(faqList);
                sApiExecutor.runOnUiThread(new Runnable() {
                    @Override
                    public void run() {
                        mView.restoreViewState();
                        refreshFaqs(sectionOrFaqId);
                    }
                });
            }
        });
    }

    private void refreshFaqs(String sectionOrFaqId) {

        if (mView == null || mView.isDetached()) {
            return;
        }

        if (mView instanceof SectionListFragment) {
            List<DisplayFaq> allSections = AIHelpDBHelper.getInstance().getAllSections();
            if (allSections.isEmpty()) {
                mView.showEmpty();
            } else {
                mView.refreshList(allSections);
            }
            return;
        }

        if (mView instanceof QuestionListFragment) {
            Section section = AIHelpDBHelper.getInstance().getSection(sectionOrFaqId);
            if (section != null) {
                List<DisplayFaq> list = section.isHasSubSection() ?
                        AIHelpDBHelper.getInstance().getSubSections(sectionOrFaqId) :
                        AIHelpDBHelper.getInstance().getDisplayFaqList(sectionOrFaqId);
                if (list.isEmpty()) {
                    mView.showEmpty();
                } else {
                    mView.refreshList(list, section.getSecTitle());
                }
            } else {
                mView.showEmpty();
            }
            return;
        }

        if (mView instanceof QuestionContentFragment) {
            RealFaq faq = AIHelpDBHelper.getInstance().getFaqByMainId(sectionOrFaqId);
            if (faq != null) {
                mView.refreshQuestionContent(faq);
            } else {
                mView.showEmpty();
            }
        }

    }

    public void goFetchFAQDataSource() {
        StatisticHelper.markFAQListViewed();
        if (isDataSourcePrepared(null)) {
            List<DisplayFaq> allSections = AIHelpDBHelper.getInstance().getAllSections();
            if (!ListUtil.isListEmpty(allSections)) {
                mView.refreshList(allSections);
            } else {
                mView.showEmpty();
            }
        }
    }

    public void goFetchAutomaticForList(String sectionId) {
        if (!TextUtils.isEmpty(sectionId)) StatisticHelper.markFAQListViewed(sectionId);
        if (isDataSourcePrepared(sectionId)) {
            Section section = AIHelpDBHelper.getInstance().getSection(sectionId);
            if (section == null) {
                section = AIHelpDBHelper.getInstance().getSubSection(sectionId);
            }

            if (section != null) {
                if (section.isHasSubSection()) {
                    mView.refreshList(AIHelpDBHelper.getInstance().getSubSections(sectionId), section.getSecTitle());
                } else {
                    mView.refreshList(AIHelpDBHelper.getInstance().getDisplayFaqList(sectionId), section.getSecTitle());
                }
            } else {
                mView.showEmpty();
            }
        }
    }

    public void goFetchQuestionContent(final String faqId, final String searchTerm) {
        if (isDataSourcePrepared(faqId)) {
            final RealFaq faq = AIHelpDBHelper.getInstance().getFaqByMainId(faqId);
            if (faq != null) {
                sApiExecutor.runAsync(new Runnable() {
                    @Override
                    public void run() {
                        final RealFaq highlightedFaq = Styles.getFAQWithHighlightedSearchTerms(mContext, faq, searchTerm);
                        sApiExecutor.runOnUiThread(new Runnable() {
                            @Override
                            public void run() {
                                if (mView == null || mView.isDetached()) {
                                    return;
                                }
                                mView.refreshQuestionContent(highlightedFaq != null ? highlightedFaq : faq);
                            }
                        });
                    }
                });
            } else {
                mView.showEmpty();
            }
        }
    }

    public void goQueryFAQList(final String query) {

        if (TextUtils.isEmpty(query) || !AIHelpDBHelper.getInstance().isFaqStoredSuccessfully()) {
            mView.refreshList(null);
            return;
        }

        sApiExecutor.runAsync(new Runnable() {
            @Override
            public void run() {
                final ArrayList<DisplayFaq> matchedFaqList = AIHelpDBHelper.getInstance().getMatchedFaqList(query);
                sApiExecutor.runOnUiThread(new Runnable() {
                    @Override
                    public void run() {
                        mView.refreshList(matchedFaqList);
                    }
                });
            }
        });
    }

    public void prepareFAQNotification() {
        if (!Const.TOGGLE_OPEN_FAQ_NOTIFICATION) return;

        try {
            // connect to mqtt when toggle is open
            IMqttCallback mqttCallback = MqttCallbackImpl.getInstance();
            mqttCallback.updateHostView(mView);
            AIHelpMqtt.getInstance().prepare(MqttConfig.TYPE_FAQ, mqttCallback);

            JSONObject params = new JSONObject();
            params.put("appid", Const.APP_ID);
            params.put("uid", UserProfile.USER_ID);
            get(API.FETCH_NEW_MSG, params, new ReqCallback<String>() {
                @Override
                public void onReqSuccess(String result) {
                    try {
                        if (!TextUtils.isEmpty(result)) {
                            JSONObject jsonObject = new JSONObject(result);
                            int newCount = jsonObject.optInt("cs_message_count");
                            boolean isTicketActive = jsonObject.optBoolean("isHaveChat");

                            int cachedCount = mSp.getInt(UserProfile.USER_ID, 0);
                            if (Math.max(0, newCount - cachedCount) > 0) {
                                mView.showSupportActionUnread();
                            } else if (isTicketActive) {
                                mView.showSupportAction();
                            }
                        }
                    } catch (Exception e) {
                        TLog.e("FAQ fetch new msg failed, because " + e.toString());
                    }
                }
            });
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public void logoutFaqMqtt() {
        AIHelpMqtt.getInstance().onFaqDestroy();
    }

}
