package net.aihelp.data.logic;

import android.app.Activity;
import android.content.Context;
import android.text.TextUtils;
import android.view.Gravity;
import android.view.View;
import android.widget.Toast;

import net.aihelp.R;
import net.aihelp.common.API;
import net.aihelp.common.Const;
import net.aihelp.common.UserProfile;
import net.aihelp.common.IntentValues;
import net.aihelp.core.mvp.IRepository;
import net.aihelp.core.net.check.Ping;
import net.aihelp.core.net.check.TraceRoute;
import net.aihelp.core.net.mqtt.AIHelpMqtt;
import net.aihelp.core.net.mqtt.callback.IMqttCallback;
import net.aihelp.core.net.mqtt.config.MqttConfig;
import net.aihelp.core.ui.dialog.AlertDialog;
import net.aihelp.core.util.bus.EventBus;
import net.aihelp.core.util.concurrent.ApiExecutorFactory;
import net.aihelp.data.event.SupportActionEvent;
import net.aihelp.data.model.cs.BotTag;
import net.aihelp.data.model.cs.ConversationMsg;
import net.aihelp.data.model.cs.ElvaBotMsg;
import net.aihelp.db.AIHelpDBHelper;
import net.aihelp.ui.cs.BaseCSFragment;
import net.aihelp.ui.helper.ConversationHelper;
import net.aihelp.ui.helper.ElvaBotHelper;
import net.aihelp.ui.helper.StatisticHelper;
import net.aihelp.utils.DeviceUuidFactory;
import net.aihelp.core.net.json.JsonHelper;
import net.aihelp.utils.LocalizeHelper;
import net.aihelp.utils.TLog;

import org.json.JSONArray;
import org.json.JSONObject;

import java.util.ArrayList;


public class ElvaBotPresenter extends MqttPresenter<BaseCSFragment, IRepository> {

    // To avoid adding welcome msg in every onResume
    private boolean isLocalElvaReplied;

    public ElvaBotPresenter(Context context) {
        super(context);
    }

    public void prepareMqttConnection(IMqttCallback callback) {
        callback.showMqttLoading();

        if (TextUtils.isEmpty(Const.CUSTOM_STORY_NODE)) {
            // LocalizeHelper.goFetchElvaBotData(mContext);
            if (LocalizeHelper.isAIMLLocalized() && !isLocalElvaReplied) {
                ArrayList<ConversationMsg> list = new ArrayList<>();
                ElvaBotMsg defaultMsg = ConversationHelper.getDefaultMsg();
                if (!defaultMsg.isBotStupid()) {
                    list.add(defaultMsg);
                    Const.isLocalWelcomeAvailable = true;
                }
                // clear all elva message histories to avoid sending old messages to server
                AIHelpDBHelper.getInstance().clearElvaMsg();
                mView.updateChatList(list);
                isLocalElvaReplied = true;
            }
        }

        if (isNetworkAvailable()) {
            callback.updateHostView(mView);
            AIHelpMqtt.getInstance().prepare(MqttConfig.TYPE_ELVA_BOT, callback);
        } else {
            mView.showError(mContext.getResources().getString(R.string.aihelp_network_no_connect));
        }

    }

    public void askForAnswer(String msg) {
        try {
            JSONObject jsonObject = new JSONObject();
            jsonObject.put("msg", msg);
//        jsonObject.put("imgFlag", "0");
            jsonObject.put("answer_type", "wildcard");
            mqtt(API.TOPIC_BOT_CHAT, jsonObject);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public void markWhetherFaqHelpful(boolean helpful, long timeStamp) {
        try {
            JSONObject faqFeedback = new JSONObject();
            faqFeedback.put("isLike", helpful ? "1" : "2");
            faqFeedback.put("timeMillis", timeStamp);
            mqtt(API.TOPIC_BOT_FAQ, faqFeedback);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public void postFeedbackOnFaq(ElvaBotMsg botMsg, String faqContentId, String feedbackMsg) {
        try {
            JSONObject params = new JSONObject();
            params.put("PlayerId", String.format("%s|%s", Const.APP_ID, UserProfile.USER_ID));
            params.put("PlayerName", UserProfile.USER_NAME);
            params.put("PlayerQuestion", botMsg.getUserInput());
            params.put("TicketId", botMsg.getFaqTicketId());
            params.put("PitchonQuestion", botMsg.getBotMsg());
            params.put("FaqId", botMsg.getFaqId());
            params.put("Message", feedbackMsg);
            params.put("Type", "1");
            params.put("Language", Const.TARGET_LAN);
            params.put("CreateTime", String.valueOf(botMsg.getTimeStamp()));
            post(API.FAQ_FEEDBACK_URL, params, null);
            StatisticHelper.whenBotFAQGotNegativeFeedback(String.valueOf(botMsg.getTimeStamp()), faqContentId, botMsg.getFaqId());
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * 1、Save elva chat history to database;
     * 2、Log chat msg group (bot & user) to server, log similar match status;
     * 3、check whether to start network checking
     */
    public void handleBotMsg(ElvaBotMsg botMsg) {
        AIHelpDBHelper.getInstance().storeElvaMsg(botMsg.getTimeStamp(), botMsg.getRawResponse());
        StatisticHelper.whenBotMessagePrepared(botMsg);
        prepareNetworkCheck(botMsg);
    }

    private void prepareNetworkCheck(ElvaBotMsg botMsg) {
        if (!Const.TOGGLE_NET_CHECK || TextUtils.isEmpty(Const.NET_CHECK_HOST) || !botMsg.isHasTag())
            return;
        if (mView.isNetCheckingInProgress()) {
            Toast.makeText(mContext, mContext.getString(R.string.aihelp_network_check_in_progress), Toast.LENGTH_SHORT).show();
            return;
        }
        tagLoop:
        for (BotTag botTag : botMsg.getBotTagList()) {
            for (String pingTags : Const.NET_PING.split(",")) {
                if (pingTags.equalsIgnoreCase(botTag.getTagName())) {

                    if (!isNetworkAvailable()) {
                        Toast.makeText(mContext, mContext.getString(R.string.aihelp_network_no_connect), Toast.LENGTH_SHORT).show();
                        return;
                    }

                    mView.updateNetCheckingStatus(true);
                    Ping.start(Const.NET_CHECK_HOST, new Ping.Callback() {
                        @Override
                        public void complete(Ping.Result result) {
                            handleNetworkResult(result);
                        }
                    });
                    break tagLoop;
                }
            }
            for (String pingTags : Const.NET_TRACE_ROUTE.split(",")) {
                if (pingTags.equalsIgnoreCase(botTag.getTagName())) {

                    if (!isNetworkAvailable()) {
                        Toast.makeText(mContext, mContext.getString(R.string.aihelp_network_no_connect), Toast.LENGTH_SHORT).show();
                        return;
                    }

                    mView.updateNetCheckingStatus(true);
                    TraceRoute.start(Const.NET_CHECK_HOST, new TraceRoute.Callback() {
                        @Override
                        public void complete(TraceRoute.Result result) {
                            handleNetworkResult(result);
                        }
                    });
                    break tagLoop;
                }
            }
        }
    }

    private void handleNetworkResult(final Object result) {
        ApiExecutorFactory.getHandlerExecutor().runOnUiThread(new Runnable() {
            @Override
            public void run() {
                mView.updateNetCheckingStatus(false);
                if (result instanceof Ping.Result) {
                    Ping.Result pingResult = (Ping.Result) result;
                    if (Const.sCheckResultListener != null) {
                        Const.sCheckResultListener.onNetworkCheckResult(pingResult.result);
                    }
                    if (pingResult.sent > 0 && pingResult.avg < 300) {
                        showNetworkFineToast();
                        return;
                    }
                    showNetworkSlowDialog(pingResult.result, true);
                }

                if (result instanceof TraceRoute.Result) {
                    TraceRoute.Result traceRouteResult = (TraceRoute.Result) result;
                    if (Const.sCheckResultListener != null) {
                        Const.sCheckResultListener.onNetworkCheckResult(traceRouteResult.content());
                    }
                    showNetworkSlowDialog(traceRouteResult.content(), false);
                }
            }
        });

    }

    private void showNetworkFineToast() {
        if (mContext instanceof Activity) {
            Toast toast = new Toast(mContext);
            toast.setView(View.inflate(mContext, R.layout.aihelp_toast_network_fine, null));
            toast.setGravity(Gravity.CENTER, 0, 0);
            Activity activity = (Activity) mContext;
            if (!activity.isFinishing()) {
                toast.show();
            } else {
                TLog.e("Activity finished when network toast is about to show.");
            }
        }
    }

    private void showNetworkSlowDialog(final String result, final boolean isPing) {
        // FIXME: 2020/11/18 show dialog when in manual support page
        if (mContext instanceof Activity && mView.isVisible()) {

            final AlertDialog uploadLogDialog = new AlertDialog.Builder(mContext)
                    .setContentView(R.layout.aihelp_dia_upload_net_check)
                    .setWidthByDevice()
                    .create();

            uploadLogDialog.setOnClickListener(R.id.aihelp_tv_cancel, new View.OnClickListener() {
                @Override
                public void onClick(View v) {
                    uploadLogDialog.dismiss();
                }
            });
            uploadLogDialog.setOnClickListener(R.id.aihelp_tv_confirm, new View.OnClickListener() {
                @Override
                public void onClick(View v) {
                    EventBus.getDefault().post(new SupportActionEvent(IntentValues.SHOW_SUPPORT_ACTION,
                            IntentValues.SUPPORT_ACTION_MSG_UNREAD));
                    postNetworkResultToServer(result, isPing ? "network,ping" : "network,traceroute");
                    logCheckResultSent(isPing ? "ping" : "traceroute");
                    uploadLogDialog.dismiss();
                }
            });

            Activity activity = (Activity) mContext;
            if (!activity.isFinishing()) {
                uploadLogDialog.show();
            } else {
                TLog.e("Activity finished when network dialog is about to show.");
            }

        }
    }

    private void logCheckResultSent(String checkType) {
        try {
            JSONObject jsonObject = new JSONObject();
            jsonObject.put("deviceId", DeviceUuidFactory.id(mContext));
            jsonObject.put("playerId", UserProfile.USER_ID);
            jsonObject.put("type", checkType);
            post(API.LOG_NETWORK_RESULT_SENT, jsonObject, null);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private void postNetworkResultToServer(String userSay, String chatTags) {
        try {
            JSONObject chatParams = new JSONObject();
            chatParams.put("msg", userSay);
            chatParams.put("timeStamp", System.currentTimeMillis());
            chatParams.put("chatTags", chatTags);
            chatParams.put("imgFlag", "0");
            mqtt(API.TOPIC_CONVERSATION_SEND, chatParams);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public void syncLogoutTypeToServer() {
        mqtt(API.MQTT_TOPIC, new JSONObject());
    }

}
