package net.aihelp.core.util.viewer;

import android.app.Dialog;
import android.content.Context;
import android.content.DialogInterface;
import android.view.KeyEvent;
import android.view.View;

import net.aihelp.R;
import net.aihelp.core.ui.image.Picasso;

import androidx.appcompat.app.AlertDialog;
import androidx.appcompat.widget.AppCompatImageView;

public class ImageViewer implements DialogInterface.OnShowListener, DialogInterface.OnKeyListener, View.OnClickListener {

    private Context context;
    private Dialog transDialog;
    private ViewerLayout rootLayout;

    // 因为Dialog的关闭有动画延迟，固不能使用 dialog.isShowing, 去判断 transferee 的显示逻辑
    private boolean shown;

    private ImageViewer(Context context) {
        this.context = context;
        createLayout();
        createDialog();
    }

    /**
     * @param context
     * @return {@link ImageViewer}
     */
    public static ImageViewer getDefault(Context context) {
        return new ImageViewer(context);
    }

    private void createLayout() {
        rootLayout = new ViewerLayout(context);
        rootLayout.setOnChildViewClickedListener(this);
    }

    private void createDialog() {
        transDialog = new AlertDialog.Builder(context,
                android.R.style.Theme_Translucent_NoTitleBar_Fullscreen)
                .setView(rootLayout)
                .create();
        transDialog.setOnShowListener(this);
        transDialog.setOnKeyListener(this);
    }

    public ImageViewer updateImageResource(String imagePath) {
        if (rootLayout != null) {
            rootLayout.updateImageResource(imagePath);
        }
        return this;
    }

    public ImageViewer updateVideoResource(String videoThumbnail, final String videoPath) {
        if (rootLayout != null) {
            rootLayout.updateImageResource(videoThumbnail);
            rootLayout.updateVideoResource(videoPath);
        }
        return this;
    }

    /**
     * transferee 是否显示
     *
     * @return true ：显示, false ：关闭
     */
    public boolean isShown() {
        return shown;
    }

    /**
     * 显示 transferee
     */
    public void show() {
        if (shown) return;
        transDialog.show();
        shown = true;
    }

    /**
     * 关闭 transferee
     */
    public void dismiss() {
        if (shown) {
            shown = false;
        }
        transDialog.dismiss();
    }

    @Override
    public void onShow(DialogInterface dialog) {
        rootLayout.show();
    }

    @Override
    public boolean onKey(DialogInterface dialog, int keyCode, KeyEvent event) {
        if (keyCode == KeyEvent.KEYCODE_BACK &&
                event.getAction() == KeyEvent.ACTION_UP &&
                !event.isCanceled()) {
            dismiss();
            return true;
        }
        return false;
    }

    @Override
    public void onClick(View v) {
        dismiss();
    }

}
