package net.aihelp.core.util.permission;

import android.Manifest;
import android.annotation.TargetApi;
import android.app.Activity;
import android.content.ActivityNotFoundException;
import android.content.Context;
import android.content.Intent;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.net.Uri;
import android.os.Build;

import net.aihelp.config.AIHelpContext;
import net.aihelp.core.net.http.AIHelpRequest;
import net.aihelp.core.net.http.config.HttpConfig;
import net.aihelp.utils.AppInfoUtil;

import java.lang.reflect.Method;

import androidx.core.app.ActivityCompat;
import androidx.core.content.ContextCompat;
import androidx.fragment.app.Fragment;
import okhttp3.OkHttpClient;

public class PermissionHelper implements IPermissionCallback {

    private Object object;
    private int requestCode;
    private String permission;

    private PermissionHelper(Object object, String permission, int requestCode) {
        this.object = object;
        this.permission = permission;
        this.requestCode = requestCode;
    }

    public static PermissionHelper getInstance(Object object, String permission, int requestCode) {
        if (object == null || permission == null || requestCode == 0) {
            throw new IllegalArgumentException("mObject == null || permission == null || requestCode == 0!");
        }
        return new PermissionHelper(object, permission, requestCode);
    }

    void invokePermissionCallback(Permission.Result result) {
        Method[] methods = object.getClass().getDeclaredMethods();
        for (Method method : methods) {
            Permission grantedMethod = method.getAnnotation(Permission.class);
            if (grantedMethod != null) {
                if (requestCode == grantedMethod.requestCode()) {
                    method.setAccessible(true);
                    try {
                        method.invoke(object, result, this);
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                }
            }
        }
    }

    private boolean isPermissionGranted() {
        if (getActivity() != null) {
            return ContextCompat.checkSelfPermission(getActivity(), permission) == PackageManager.PERMISSION_GRANTED;
        }
        return false;
    }

    private Activity getActivity() {
        if (object instanceof Activity) {
            return (Activity) object;
        }
        if (object instanceof Fragment) {
            return ((Fragment) object).getActivity();
        }
        return null;
    }

    void onRequestPermissionsResult(boolean isPermissionGranted) {

        if (isPermissionGranted) {
            invokePermissionCallback(Permission.Result.GRANTED);
        } else {
            if (shouldShowRequestPermissionRationale()) {
                invokePermissionCallback(Permission.Result.DENIED);
            } else {
                invokePermissionCallback(Permission.Result.GO_SETTING);
            }
        }

    }

    Permission.State checkPermissionState() {
        Context context = AIHelpContext.getInstance().getContext();
        Permission.State permissionState;
        if (isPermissionGranted() || Build.VERSION.SDK_INT < Build.VERSION_CODES.M) {
            permissionState = Permission.State.AVAILABLE;
        } else {
            boolean hasPermissionInManifest = hasPermissionInManifest(context, permission);
            if (hasPermissionInManifest) {
                permissionState = Permission.State.ASKABLE;
                if (shouldShowRequestPermissionRationale()) {
                    permissionState = Permission.State.RATIONAL;
                }
            } else {
                permissionState = Permission.State.UNAVAILABLE;
            }
        }
        return permissionState;
    }

    void requestPermission() {
        if (object instanceof Activity) {
            ActivityCompat.requestPermissions((Activity) object, new String[]{permission}, requestCode);
        } else if (object instanceof Fragment) {
            Fragment fragment = (Fragment) object;
            fragment.requestPermissions(new String[]{permission}, requestCode);
        }
    }

    private boolean shouldShowRequestPermissionRationale() {
        if (object instanceof Activity) {
            if (android.os.Build.VERSION.SDK_INT >= android.os.Build.VERSION_CODES.M) {
                return ((Activity) object).shouldShowRequestPermissionRationale(permission);
            }
        } else if (object instanceof Fragment) {
            Fragment fragment = (Fragment) object;
            return fragment.shouldShowRequestPermissionRationale(permission);
        }
        return false;
    }

    private static boolean hasPermissionInManifest(Context context, String permission) {
        try {
            PackageInfo info = context.getPackageManager().getPackageInfo(context.getPackageName(), PackageManager.GET_PERMISSIONS);
            if (info.requestedPermissions != null) {
                for (String p : info.requestedPermissions) {
                    if (p.equals(permission)) {
                        return true;
                    }
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return false;
    }

    public static boolean hasReadPermission(Context context) {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN) {
            return hasPermissionInManifest(context, Manifest.permission.READ_EXTERNAL_STORAGE);
        }
        return true;
    }

    private void showSettingsPage() {
        Activity activity = getActivity();
        if (activity != null) {
            try {
                Intent settingsIntent = new Intent("android.settings.APPLICATION_DETAILS_SETTINGS");
                settingsIntent.addCategory("android.intent.category.DEFAULT");
                String packageName = activity.getPackageName();
                settingsIntent.setData(Uri.parse("package:" + packageName));
                activity.startActivity(settingsIntent);
            } catch (ActivityNotFoundException var3) {
                Intent i = new Intent("android.settings.MANAGE_APPLICATIONS_SETTINGS");
                i.addCategory("android.intent.category.DEFAULT");
                activity.startActivity(i);
            }
        }
    }

    @Override
    public void onPermissionDenied() {
        // showSettingsPage();
    }

    @Override
    public void onPermissionRational() {
        requestPermission();
    }

    @Override
    public void onPermissionIgnored() {
        showSettingsPage();
    }

}
