package net.aihelp.core.util.logger;

import android.util.Log;

import net.aihelp.core.util.logger.controller.LoggerDBController;

import org.json.JSONArray;

import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import androidx.annotation.NonNull;

public class AIHelpLogger {

    private final String TAG;
    private final LoggerDBController logStorage;
    private ThreadPoolExecutor threadPoolExecutor;

    private static final class LazyHolder {
        static final AIHelpLogger INSTANCE = new AIHelpLogger();
    }

    public static AIHelpLogger getInstance() {
        return AIHelpLogger.LazyHolder.INSTANCE;
    }

    private AIHelpLogger() {
        TAG = AIHelpLogger.class.getSimpleName();
        logStorage = LoggerDBController.getInstance();

        if (threadPoolExecutor == null) {
            LinkedBlockingQueue<Runnable> logMessagesQueue = new LinkedBlockingQueue<Runnable>();
            this.threadPoolExecutor = new ThreadPoolExecutor(1, 1, 60L, TimeUnit.SECONDS, logMessagesQueue, new ThreadFactory() {
                public Thread newThread(@NonNull Runnable r) {
                    return new Thread(r, "AIHelpLogger");
                }
            });
        }
    }

    public void log(int logLevel, String message, Throwable tr) {
        this.logMessageToDatabase(logLevel, message, getStackTraceString(tr), System.currentTimeMillis());
    }

    public JSONArray getCachedLogs() {
        return logStorage.getCachedLogs();
    }

    public void deleteAllCachedLogs() {
        this.logStorage.deleteAll();
    }

    private void logMessageToDatabase(final int level, final String message, final String stacktrace, final long timeStamp) {
        try {
            threadPoolExecutor.submit(new Runnable() {
                @Override
                public void run() {
                    logStorage.insert(level, message, stacktrace, timeStamp);
                }
            });
        } catch (Exception e) {
            Log.e(this.TAG, "Rejected execution of log message : " + message, e);
        }
    }

    private static String getStackTraceString(Throwable throwable) {
        String stacktrace = "";
        if (throwable != null) {
            StringBuilder builder = new StringBuilder();
            String throwableString = Log.getStackTraceString(throwable);
            builder.append(throwableString);
            stacktrace = builder.toString();
        }
        return stacktrace;
    }

    public static final int LEVEL_FATAL = 1;
    public static final int LEVEL_ERROR = 2;
    public static final int LEVEL_WARN = 3;

    public static void fatal(String message, Throwable tr) {
        getInstance().log(LEVEL_FATAL, message, tr);
    }

    public static void error(String message, Throwable tr) {
        getInstance().log(LEVEL_ERROR, message, tr);
    }

    public static void warn(String message, Throwable tr) {
        getInstance().log(LEVEL_WARN, message, tr);
    }

}

