package net.aihelp.core.ui;

import android.content.Context;
import android.content.Intent;
import android.content.res.Configuration;
import android.content.res.Resources;
import android.os.Build;
import android.os.Bundle;
import android.view.KeyEvent;
import android.widget.Toast;

import net.aihelp.R;
import net.aihelp.common.Const;
import net.aihelp.config.AIHelpContext;
import net.aihelp.core.net.monitor.NetworkMonitorManager;
import net.aihelp.core.net.monitor.NetworkState;
import net.aihelp.core.util.bus.EventBus;
import net.aihelp.utils.AppInfoUtil;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.appcompat.app.AppCompatActivity;
import androidx.appcompat.app.AppCompatDelegate;

public abstract class BaseActivity extends AppCompatActivity {

    protected Context mContext;

    static {
        if (Build.VERSION.SDK_INT < Build.VERSION_CODES.LOLLIPOP) {
            AppCompatDelegate.setCompatVectorFromResourcesEnabled(true);
        }
    }

    @Override
    protected void onCreate(@Nullable Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);

        mContext = getBaseContext();
        if (savedInstanceState != null) {
            Intent launchIntent = AppInfoUtil.getLaunchIntent(getApplicationContext(), getPackageName());
            if (launchIntent != null) {
                finish();
                startActivity(launchIntent);
            }
            return;
        }

        setContentView(getLayoutId());
        initView();
        if (isApplyPendingTransition()) {
            overridePendingTransition(R.anim.aihelp_right_in, R.anim.aihelp_exit_trans);
        }

        NetworkMonitorManager.getInstance().register(this);
    }

    @Override
    public void finish() {
        super.finish();
        if (isApplyPendingTransition()) {
            overridePendingTransition(0, R.anim.aihelp_right_out);
        }
    }

    @Override
    public void onConfigurationChanged(@NonNull Configuration newConfig) {
        if (newConfig.fontScale != 1) getResources();
        super.onConfigurationChanged(newConfig);
    }

    @Override
    public Resources getResources() {
        Resources resources = super.getResources();
        if (resources.getConfiguration().fontScale != 1) {
            Configuration newConfig = new Configuration();
            newConfig.setToDefaults();
            resources.updateConfiguration(newConfig, resources.getDisplayMetrics());
        }
        return resources;
    }

    @Override
    protected void onDestroy() {
        super.onDestroy();
        NetworkMonitorManager.getInstance().unregister(this);
    }

    public void onNetworkStateChanged(NetworkState state) {
        if (state == NetworkState.NONE) {
            Toast.makeText(mContext, mContext.getResources().getString(R.string.aihelp_network_no_connect), Toast.LENGTH_SHORT).show();
        }
        EventBus.getDefault().post(state);
    }

    public abstract int getLayoutId();

    public void initView() {
    }

    public boolean isApplyPendingTransition() {
        return false;
    }

    @Override
    protected void attachBaseContext(Context newBase) {
        Context context = AIHelpContext.successfullyInit.get() ?
                AIHelpContext.createContextWithLocale(newBase) : newBase;
        super.attachBaseContext(context);
    }

    @Override
    public boolean onKeyDown(int keyCode, KeyEvent event) {
        if (keyCode == KeyEvent.KEYCODE_BACK) {
            if (!Const.isNestedFragmentOnResume) return false;
        }
        return super.onKeyDown(keyCode, event);
    }
}
