package net.aihelp.core.net.mqtt;

import android.os.Handler;
import android.os.Message;

import net.aihelp.common.API;
import net.aihelp.core.net.mqtt.callback.ConnectCallback;
import net.aihelp.core.net.mqtt.callback.IMqttCallback;
import net.aihelp.core.net.mqtt.callback.ReceiveListener;
import net.aihelp.core.net.mqtt.callback.SendCallback;
import net.aihelp.core.net.mqtt.callback.SubscribeCallback;
import net.aihelp.core.net.mqtt.config.MqttConfig;
import net.aihelp.core.util.logger.AIHelpLogger;
import net.aihelp.ui.helper.LoginMqttHelper;
import net.aihelp.utils.TLog;

import net.aihelp.core.net.mqtt.client.CallbackConnection;
import net.aihelp.core.net.mqtt.client.QoS;

import org.json.JSONObject;

import androidx.annotation.NonNull;

public class AIHelpMqtt {

    private CallbackConnection mqttConnection;
    private CallbackConnection faqMqttConnection;

    public void prepare(int type, IMqttCallback callback) {
        boolean isFaqRequest = type == MqttConfig.TYPE_FAQ;

        if (!isFaqRequest) {
            MqttConfig.getInstance().setLoginType(type);
        }

        if (MqttConfig.getInstance().isConnected()) {
            if (type == MqttConfig.TYPE_CONVERSATION) {
                postToServer(API.TOPIC_LOGIN, LoginMqttHelper.getLoginParams());
                return;
            }
            callback.dismissMqttLoading();
            return;
        } else {
            mConnectingMonitor.sendEmptyMessageDelayed(0, 5000);
        }


        mqttConnection = MqttConfig.getInstance().getMqttConnection(isFaqRequest);
        mqttConnection.connect(new ConnectCallback(isFaqRequest, callback));
        mqttConnection.listener(new ReceiveListener(callback, mConnectingMonitor));
        mqttConnection.subscribe(MqttConfig.getInstance().getTopic(isFaqRequest), new SubscribeCallback(isFaqRequest, this, callback));

        if (isFaqRequest) {
            faqMqttConnection = mqttConnection;
        }

    }

    public void postToServer(String topicName, JSONObject params) {
        try {
            if (mqttConnection != null) {
                String topic = MqttConfig.newTopic(topicName);
                byte[] bytes = params.toString().getBytes();
                TLog.json(String.format("MQTT [send message] %s", topicName), new String(bytes));
                mqttConnection.publish(topic, bytes, QoS.AT_MOST_ONCE, false, new SendCallback());
            }
        } catch (Exception e) {
            TLog.e("MQTT postToServer error -> " + e.toString());
        }
    }

    public void onFaqDestroy() {
        if (faqMqttConnection != null) {
            faqMqttConnection.disconnect(null);
            faqMqttConnection = null;
        }
    }

    public void logoutMqttConnection() {
        if (mqttConnection != null) {
            mqttConnection.unregisterListener();
            mqttConnection.disconnect(null);
            mqttConnection = null;
        }
        // update connect status after disconnect, but this can be misleading
        // because the disconnect operation will take quite long time
        MqttConfig.getInstance().setConnected(false);
    }

    public static AIHelpMqtt getInstance() {
        return AIHelpMqtt.Holder.INSTANCE;
    }

    private static class Holder {
        private static final AIHelpMqtt INSTANCE = new AIHelpMqtt();
    }

    private AIHelpMqtt() {
        mConnectingMonitor = new ConnectingHandler();
    }

    private static ConnectingHandler mConnectingMonitor;

    public static class ConnectingHandler extends Handler {
        @Override
        public void handleMessage(@NonNull Message msg) {
            JSONObject loginParams = LoginMqttHelper.getLoginParams();
            AIHelpLogger.warn("mqtt connect for more than 5s.", new IllegalStateException(loginParams.toString()));
        }
    }

}
