package net.aihelp.core.net.http.config;

import android.os.Build;
import android.text.TextUtils;

import net.aihelp.BuildConfig;
import net.aihelp.common.API;
import net.aihelp.common.Const;
import net.aihelp.common.UserProfile;
import net.aihelp.core.net.http.FileProgressRequestBody;
import net.aihelp.core.net.http.interceptor.HeaderInterceptor;
import net.aihelp.core.net.http.interceptor.LogInterceptor;
import net.aihelp.core.net.http.interceptor.SignInterceptor;
import net.aihelp.utils.TLog;

import java.io.File;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.security.MessageDigest;
import java.security.SecureRandom;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Random;
import java.util.concurrent.TimeUnit;

import javax.net.ssl.HostnameVerifier;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLSession;
import javax.net.ssl.TrustManagerFactory;
import javax.net.ssl.X509TrustManager;

import androidx.annotation.NonNull;
import okhttp3.CipherSuite;
import okhttp3.ConnectionSpec;
import okhttp3.Cookie;
import okhttp3.CookieJar;
import okhttp3.Headers;
import okhttp3.HttpUrl;
import okhttp3.MediaType;
import okhttp3.MultipartBody;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.TlsVersion;

public class HttpConfig {

    public static final MediaType MEDIA_TYPE_URLENCODED = MediaType.parse("application/x-www-form-urlencoded; charset=utf-8");
    public static final MediaType MEDIA_TYPE_JSON = MediaType.parse("application/json; charset=utf-8");
    //    private static final MediaType MEDIA_TYPE_JSON = MediaType.parse("multipart/form-data; charset=utf-8");
    public static final MediaType MEDIA_OBJECT_STREAM = MediaType.parse("application/octet-stream");

    private static final int TIME_OUT_LIMIT = 60;

    private static ConnectionSpec spec = new ConnectionSpec.Builder(ConnectionSpec.COMPATIBLE_TLS)
            .supportsTlsExtensions(true)
            .tlsVersions(TlsVersion.TLS_1_2, TlsVersion.TLS_1_1, TlsVersion.TLS_1_0)
            .cipherSuites(
                    CipherSuite.TLS_ECDHE_ECDSA_WITH_AES_128_GCM_SHA256,
                    CipherSuite.TLS_ECDHE_RSA_WITH_AES_128_GCM_SHA256,
                    CipherSuite.TLS_DHE_RSA_WITH_AES_128_GCM_SHA256,
                    CipherSuite.TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA,
                    CipherSuite.TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA,
                    CipherSuite.TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA,
                    CipherSuite.TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA,
                    CipherSuite.TLS_ECDHE_ECDSA_WITH_RC4_128_SHA,
                    CipherSuite.TLS_ECDHE_RSA_WITH_RC4_128_SHA,
                    CipherSuite.TLS_DHE_RSA_WITH_AES_128_CBC_SHA,
                    CipherSuite.TLS_DHE_DSS_WITH_AES_128_CBC_SHA,
                    CipherSuite.TLS_DHE_RSA_WITH_AES_256_CBC_SHA)
            .build();

    public static OkHttpClient getOkHttpClient(boolean isAddHeader) {
        OkHttpClient.Builder clientBuilder = new OkHttpClient().newBuilder()
                .connectTimeout(TIME_OUT_LIMIT, TimeUnit.SECONDS)
                .readTimeout(TIME_OUT_LIMIT, TimeUnit.SECONDS)
                .writeTimeout(TIME_OUT_LIMIT, TimeUnit.SECONDS)
                .hostnameVerifier(createInsecureHostnameVerifier())
                .cookieJar(new CookieJar() {
                    private final HashMap<HttpUrl, List<Cookie>> cookieStore = new HashMap<>();

                    @Override
                    public void saveFromResponse(@NonNull HttpUrl url, @NonNull List<Cookie> cookies) {
                        cookieStore.put(url, cookies);
                    }

                    @NonNull
                    @Override
                    public List<Cookie> loadForRequest(@NonNull HttpUrl url) {
                        List<Cookie> cookies = cookieStore.get(url);
                        return cookies != null ? cookies : new ArrayList<Cookie>();
                    }
                });
        if (isAddHeader) {
            clientBuilder.addInterceptor(new HeaderInterceptor());
        }
        clientBuilder.addInterceptor(new LogInterceptor());
        // clientBuilder.addInterceptor(new SignInterceptor());

        clientBuilder.connectionSpecs(Collections.singletonList(spec));
        if (Build.VERSION.SDK_INT < Build.VERSION_CODES.LOLLIPOP_MR1) {
            List<ConnectionSpec> specsList = getSpecsBelowLollipopMR1(clientBuilder);
            if (specsList != null) {
                clientBuilder.connectionSpecs(specsList);
            }
        }

        return clientBuilder.build();
    }

    private static List<ConnectionSpec> getSpecsBelowLollipopMR1(OkHttpClient.Builder okb) {

        try {
            SSLContext sc = SSLContext.getInstance("TLSv1.2");
            sc.init(null, null, null);
            okb.sslSocketFactory(new Tls12SocketFactory(sc.getSocketFactory()));
            ConnectionSpec cs = new ConnectionSpec.Builder(ConnectionSpec.MODERN_TLS)
                    .tlsVersions(TlsVersion.TLS_1_2)
                    .build();

            List<ConnectionSpec> specs = new ArrayList<>();
            specs.add(cs);
            specs.add(ConnectionSpec.COMPATIBLE_TLS);

            return specs;

        } catch (Exception exc) {
            exc.printStackTrace();
            return null;
        }
    }

    private static HostnameVerifier createInsecureHostnameVerifier() {
        return new HostnameVerifier() {
            @Override
            public boolean verify(String hostname, SSLSession session) {
                return hostname.contains("aihelp.net");
            }
        };
    }

    public static Request getUploadRequest(String url, File file) {
        String formData = null;
        try {
            formData = String.format("form-data;name=file;filename=%s", URLEncoder.encode(file.getName(), "utf-8"));
        } catch (UnsupportedEncodingException e) {
            e.printStackTrace();
        }
        if (formData == null) return null;
        FileProgressRequestBody filePart = new FileProgressRequestBody(MultipartBody.FORM, file, null);
        MultipartBody.Builder requestBodyBuilder = new MultipartBody.Builder().setType(MultipartBody.FORM)
                .addPart(Headers.of("Content-Disposition", formData), filePart);
        if (url.contains(API.UPLOAD_VIDEO_URL)) {
            String randomValue = String.valueOf(new Random().nextInt(Integer.MAX_VALUE));
            String timeStamp = String.valueOf(System.currentTimeMillis());
            requestBodyBuilder.addFormDataPart("appId", Const.APP_ID)
                    .addFormDataPart("random", randomValue)
                    .addFormDataPart("timespan", timeStamp)
                    .addFormDataPart("userId", UserProfile.USER_ID)
                    .addFormDataPart("siga", getUploadVideoSig(randomValue, timeStamp));
        }
        MultipartBody body = requestBodyBuilder.build();
        return new Request.Builder().url(url).post(body).build();
    }

    private static String getUploadVideoSig(String randomValue, String timeStamp) {
        String source = "appId_" + Const.APP_ID + "random_" + randomValue
                + "timespan_" + timeStamp + "userId_" + UserProfile.USER_ID;
        return md5(source);
    }

    public static String md5(String string) {
        if (!TextUtils.isEmpty(string)) {
            byte[] hash;
            try {
                hash = MessageDigest.getInstance("MD5").digest(string.getBytes("UTF-8"));
            } catch (Exception e) {
                e.printStackTrace();
                hash = new byte[2];
            }
            StringBuilder hex = new StringBuilder(hash.length * 2);
            for (byte b : hash) {
                if ((b & 0xFF) < 0x10) {
                    hex.append("0");
                }
                hex.append(Integer.toHexString(b & 0xFF));
            }
            return hex.toString();
        }
        return "";
    }

}
