package net.aequologica.neo.serioulizer.jackson;

import static com.fasterxml.jackson.annotation.JsonInclude.Include.NON_NULL;
import static com.fasterxml.jackson.databind.SerializationFeature.INDENT_OUTPUT;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import net.aequologica.neo.serioulizer.ReaderWriter;

public abstract class AbstractReaderWriter<T> implements ReaderWriter<T> {
    final protected ObjectMapper     objectMapper;
    final protected TypeReference<T> valueTypeRef;

    public AbstractReaderWriter(TypeReference<T> valueTypeRef) {
        this.objectMapper = createMapper();
        this.valueTypeRef = valueTypeRef;
    }

    @Override
    public void write(OutputStream outputStream, T t) throws IOException {
        this.objectMapper.writeValue(outputStream, t);
    }

    @Override
    public void write(java.io.Writer writer, T t) throws IOException {
        this.objectMapper.writeValue(writer, t);
    }

    @Override
    public T read(InputStream inputStream) throws IOException {
        return this.objectMapper.readValue(inputStream, valueTypeRef);
    }

    @Override
    public T read(java.io.Reader reader) throws IOException {
        return this.objectMapper.readValue(reader, valueTypeRef);
    }
    
    public static ObjectMapper createMapper() {
        final ObjectMapper ret = new ObjectMapper();

        ret.enable(INDENT_OUTPUT);
        ret.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
        ret.configure(SerializationFeature.WRITE_DATES_AS_TIMESTAMPS, false );
        ret.configure(SerializationFeature.FAIL_ON_EMPTY_BEANS, false );
        ret.setSerializationInclusion(NON_NULL);
        
        ret.registerModule(new JavaTimeModule());
        
        return ret;
    }
}
