package net.aequologica.neo.serioulizer;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.io.StringReader;
import java.nio.charset.StandardCharsets;
import java.nio.file.Path;
import java.util.List;

import org.apache.commons.io.input.ReaderInputStream;

public interface Serioulizer extends Closeable {

    List<Path> list     (Path path)                throws IOException; // list content
    void       write    (Path path, Stream stream) throws IOException; // 'C': create, and 'U': update
    Stream     getStream(Path path)                throws IOException; // 'R': read (get Stream to read from)
    void       delete   (Path path)                throws IOException; // 'D': delete
    
    //
    // nested interface
    //
    interface Stream extends Closeable {
        InputStream getInputStream();
        String      getMimeType();
        long        getLength();
    }

    //
    // defaults methods
    //
    default void close() throws IOException {
    }

    //
    // static helpers
    //
    static public Stream createStream(final InputStream inputStream, final String mimeType, final long length) {
        if (inputStream == null) {
            return null;
        }
        return new StreamImpl(inputStream, mimeType, length);
    }

    static public Stream createStream(final byte[] bytes, String mimeType) {
        if (bytes == null) {
            return null;
        }
        return new StreamImpl(new ByteArrayInputStream(bytes), mimeType, bytes.length);
    }
    
    static public Stream createStream(final String content) {
        if (content == null) {
            return null;
        }
        String charset = StandardCharsets.UTF_8.name();
        return createStream(new ReaderInputStream(new StringReader(content), charset), "text/plain; charset="+charset, -1);
    }
    
    static public String read(InputStream inputStream) throws IOException {
        String charset = StandardCharsets.UTF_8.name();
        return read(inputStream, charset);
    }
    
    static public String read(InputStream inputStream, String charsetName) throws IOException {
        if (inputStream == null) {
            return null;
        }
        ByteArrayOutputStream result = new ByteArrayOutputStream();
        byte[] buffer = new byte[1024];
        int length;
        while ((length = inputStream.read(buffer)) != -1) {
            result.write(buffer, 0, length);
        }
        return result.toString(charsetName);
    }

    //
    // nested implementation class
    //

    class StreamImpl implements Stream {
        
        final public InputStream inputStream;
        final public String      mimeType;
        final public long        length;
    
        StreamImpl(final InputStream inputStream, String mimeType, final long length) {
            this.inputStream = inputStream;
            this.mimeType    = (mimeType != null ? mimeType : "text/plain;charset="+StandardCharsets.UTF_8.name());
            this.length      = length;
        }
    
        @Override public InputStream getInputStream() { return inputStream; }
        @Override public String      getMimeType()    { return mimeType;    }
        @Override public long        getLength()      { return length;      }
    
        @Override
        public void close() throws IOException {
            inputStream.close();
        }
    }

}