package net.aequologica.neo.quintessence.jaxrs;

import java.io.IOException;
import java.net.URI;
import java.util.Arrays;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.inject.Singleton;
import javax.servlet.http.HttpServletResponse;
import javax.ws.rs.GET;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.UriBuilder;
import javax.xml.bind.annotation.XmlAccessType;
import javax.xml.bind.annotation.XmlAccessorType;
import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlRootElement;

import org.eclipse.aether.RepositorySystem;
import org.eclipse.aether.RepositorySystemSession;
import org.eclipse.aether.artifact.Artifact;
import org.eclipse.aether.artifact.DefaultArtifact;
import org.eclipse.aether.repository.RemoteRepository;
import org.eclipse.aether.resolution.VersionRangeRequest;
import org.eclipse.aether.resolution.VersionRangeResolutionException;
import org.eclipse.aether.resolution.VersionRangeResult;
import org.eclipse.aether.resolution.VersionRequest;
import org.eclipse.aether.resolution.VersionResolutionException;
import org.eclipse.aether.resolution.VersionResult;
import org.eclipse.aether.version.Version;

import net.aequologica.neo.geppaequo.config.ConfigRegistry;
import net.aequologica.neo.quintessence.aether.Booter;
import net.aequologica.neo.quintessence.config.QuintessenceConfig;

@Singleton
@javax.ws.rs.Path("v1")
public class Resource {

    private QuintessenceConfig config = ConfigRegistry.getConfig(QuintessenceConfig.class);

    @GET
    @Produces("text/plain")
    public Response ping() {
        return Response.ok().entity("api/quintessence is here").build();
    }

    @GET
    @javax.ws.rs.Path("/repositories/{repositoryId: .+}/groups/{groupId: .+}/artifacts/{artifactId: .+}/versions/{versionRange: .+}/extensions/{extension: .+}")
    @Produces( MediaType.APPLICATION_JSON )
    public Response resolve(
            @Context final HttpServletResponse httpServletResponse,
            @PathParam( "repositoryId"  ) final String repositoryId,
            @PathParam( "groupId"       ) final String groupId, 
            @PathParam( "artifactId"    ) final String artifactId, 
            @PathParam( "versionRange"  ) final String versionRange,
            @PathParam( "extension"     ) final String extension) {
        
        String resolvedRepositoryId = repositoryId;
        if (repositoryId != null) {
            if (repositoryId.equals("SNAPSHOTS")) {
                resolvedRepositoryId = config.getSnapshotRepo().getName();
            } else if (repositoryId.equals("RELEASES")){
                resolvedRepositoryId = config.getReleaseRepo().getName();
            }
        }
        return legacy(
                httpServletResponse,
                Arrays.asList(resolvedRepositoryId),
                groupId,
                artifactId,
                versionRange == null ? "[,)" : versionRange,
                extension,
                null,
                null);
    }
    
    @XmlAccessorType(XmlAccessType.PUBLIC_MEMBER)
    @XmlRootElement
    public class ExceptionInfo {
        private int status;
        private String msg, desc;
        public ExceptionInfo(int status, String msg, String desc) {
            this.status=status;
            this.msg=msg;
            this.desc=desc;
        }

        @XmlElement public int getStatus() { return status; }
        @XmlElement public String getMessage() { return msg; }
        @XmlElement public String getDescription() { return desc; }
    }    
    
    static class MyIllegalArgumentException extends IllegalArgumentException {

        private static final long serialVersionUID = 8668098471962316302L;

        private final String d;
        MyIllegalArgumentException(String s, String d) {
            super(s);
            this.d = d;
        }
        
        String getDescription() {
            return this.d;
        }
        
    }
    
    VersionRangeResolverImpl versionRangeResolver = new VersionRangeResolverImpl();
    
    @GET
    @javax.ws.rs.Path("/legacy")
    @Produces( MediaType.APPLICATION_JSON )
    public Response legacy(
            @Context           final HttpServletResponse httpServletResponse,
            @QueryParam( "r" ) final List<String> repositories,
            @QueryParam( "g" ) final String  groupId, 
            @QueryParam( "a" ) final String  artifactId, 
            @QueryParam( "v" ) final String  versionRange,
            @QueryParam( "e" ) final String  extension, 
            @QueryParam( "c" ) final String  classifier, 
            @QueryParam( "p" ) final String  packaging) {

            try {
            RepositoryHolder repositoryHolder = new RepositoryHolder();
            
            Artifact artifact = versionRangeResolver.resolve(
                    repositories,
                    groupId, 
                    artifactId, 
                    classifier,
                    extension, 
                    versionRange,
                    repositoryHolder);

            String downloadURI = buildDownloadURI(URI.create(repositoryHolder.remoteRepository.getUrl()), artifact);
            httpServletResponse.setHeader("Location", downloadURI);
                
            return Response.ok().entity(artifact).build();

        } catch (MyIllegalArgumentException iae) {
            return Response.status(422).entity( new ExceptionInfo(422, iae.getMessage(), iae.getDescription())).type(MediaType.APPLICATION_JSON).build();
        } catch (VersionRangeResolutionException vrre) {
            return Response.status(404).entity(new ExceptionInfo(404, vrre.getMessage(), null)).type(MediaType.APPLICATION_JSON).build();
        } catch (VersionResolutionException vre) {
            return Response.status(404).entity(new ExceptionInfo(404, vre.getMessage(), null)).type(MediaType.APPLICATION_JSON).build();
        } catch (Throwable t) {
            return Response.status(500).entity(new ExceptionInfo(500, t.getMessage(), null)).type(MediaType.APPLICATION_JSON).build();
        }
    }

    public class RepositoryHolder {
        public String repositoryId;
        public RemoteRepository remoteRepository;
    }

    static class VersionRangeResolverImpl {

        public Artifact resolve(    final List<String> repositories, 
                                    final String groupId, 
                                    final String artifactId, 
                                    final String classifier, 
                                    final String extension, 
                                    final String version, 
                                    RepositoryHolder repositoryHolder) throws IOException, VersionRangeResolutionException, VersionResolutionException {
            
            Artifact artifact = new DefaultArtifact(groupId, artifactId, classifier, extension, version);

            if (repositories.size() == 0      ||
                StringUtils.isEmpty(artifact.getGroupId())      ||
                StringUtils.isEmpty(artifact.getArtifactId())   ||
                StringUtils.isEmpty(artifact.getVersion())) {
                throw new IllegalArgumentException("at least one of [ group id | artifact id | version range | repository ] is missing.");
            }
              
            /*
            @SuppressWarnings("unused")
            String[] versions = parseVersionRange(artifact.getVersion()); 
            */
            
            Artifact ret = artifact;
            
            try (Booter booter = new Booter()) {
                
                RepositorySystem system = booter.newRepositorySystem();
                RepositorySystemSession session = booter.newRepositorySystemSession(system);
    
                VersionRangeRequest rangeRequest = new VersionRangeRequest();
                rangeRequest.setArtifact(artifact);
                List<RemoteRepository> newRepositories = booter.newRepositories(system, session);
                if (repositories == null || repositories.size() == 0) {
                    rangeRequest.setRepositories(newRepositories);
                }
                for (RemoteRepository remoteRepository : newRepositories) {
                    if (repositories.contains(remoteRepository.getId())) {
                        rangeRequest.addRepository(remoteRepository);
                    }
                }
                
                VersionRangeResult rangeResult = system.resolveVersionRange(session, rangeRequest);
                
                @SuppressWarnings("unused")
                Version lowestVersion = rangeResult.getLowestVersion();
                Version newestVersion = rangeResult.getHighestVersion();
                
                if (newestVersion == null) {
                    throw new VersionRangeResolutionException(rangeResult);
                }

                repositoryHolder.repositoryId = rangeResult.getRepository(newestVersion).getId();
                for (RemoteRepository remoteRepository : newRepositories) {
                    if (remoteRepository.getId().equals(repositoryHolder.repositoryId)) {
                        repositoryHolder.remoteRepository = remoteRepository;
                    }
                }
                
                String resolvedSnapshotVersion = null;
                if (newestVersion.toString().endsWith("SNAPSHOT")) {
                    VersionRequest versionRequest = new VersionRequest();
                    ret = new DefaultArtifact(artifact.getGroupId(), artifact.getArtifactId(), artifact.getClassifier(), artifact.getExtension(), newestVersion.toString());
                    versionRequest.setArtifact(ret);
                    versionRequest.addRepository(repositoryHolder.remoteRepository);
                    VersionResult versionResult = system.resolveVersion( session, versionRequest );
                    
                    resolvedSnapshotVersion = versionResult.getVersion();
                }
                
                Artifact artifactResult = new DefaultArtifact(
                        ret.getGroupId(), 
                        ret.getArtifactId(), 
                        ret.getClassifier(), 
                        ret.getExtension(), 
                        resolvedSnapshotVersion==null?newestVersion.toString():resolvedSnapshotVersion);
                
                return artifactResult;
            }
            
        }

        static String[] parseVersionRange(String versionRange) {
            
            versionRange = StringUtils.trimToEmpty(versionRange);
            
            String reason = null;
            
            for (int i=0; i<1; i++) {
                if (versionRange.length()<3) {
                    reason = "less than 3 characters";
                    break;
                }

                // special cases for compatibility with nexus artifact.resolve REST service
                if ("LATEST".equals(versionRange)) {
                    return new String[]{"", versionRange};
                }
                if ("RELEASE".equals(versionRange)) {
                    return new String[]{"", versionRange};
                }
                if (versionRange.endsWith("SNAPSHOT")) {
                    return new String[]{"", versionRange};
                }
                
                // true version range parsing
                String[] versions = versionRange.split(",");

                if (versions.length == 1) {
                    reason = "must contain a ',' separator";
                    break;
                } if (versions.length > 2) {
                    reason = "can contain only one ',' separator";
                    break;
                } 

                if (versions[0].charAt(0) != '[' && versions[0].charAt(0) != '(' ) {
                    reason = "must start with '[' or '('";
                    break;
                }
                versions[0] = StringUtils.trimToEmpty(versions[0].substring(1));
                
                if (versions[1].charAt(versions[1].length()-1) != ']' && versions[1].charAt(versions[1].length()-1) != ')') {
                    reason = "must end with ']' or ')'";
                    break;
                }
                versions[1] = StringUtils.trimToEmpty(versions[1].substring(0, versions[1].length()-1));
                
                return versions;
            }
            if (reason != null) {
                String prefix = "not a valid version range : « " + versionRange + " » ";
                String suffix = " cf. https://docs.oracle.com/middleware/1212/core/MAVEN/maven_version.htm#MAVEN402";
                throw new MyIllegalArgumentException( prefix + reason, suffix );
            }
            return null;
        }

        public long[] parseTimeStampBuildNumberFromSnapshot(String snapshotVersion) {
            long[] ret = new long[]{0L, 0L};
            
            Pattern p = Pattern.compile( "([^- ]+)-([^- ]+)-([^- ]+)" );
            Matcher m = p.matcher( snapshotVersion );
            if ( !m.matches() || m.groupCount()!=3) {
                throw new MyIllegalArgumentException( "Bad snapshotVersion coordinates « " + snapshotVersion
                    + "»", "expected format is <non snapshot version>-<timestamp>-<build number>" );
            }
            String timestamp = m.group( 2 );
            int dot = timestamp.indexOf('.');
            if (dot != -1) {
                timestamp = timestamp.substring(0, dot) + timestamp.substring(dot+1);  
            }
            ret[0] = Long.parseLong(timestamp);
            ret[1] = Long.parseLong(m.group( 3 ));
            
            return ret;
        }
        
    }
    
    public static class StringUtils {

        public static String trimToEmpty(String string) {
            return string == null ? "" : string.trim();
        }

        public static boolean isEmpty(String string) {
            if (string == null) {
                return true;
            }
            if (string.trim().length() == 0) {
                return true;
            }
            return false;
        }

    }
    
    private String buildDownloadURI(final URI nexusURI, Artifact artifact) throws Exception {
        URI merged = new URI(   nexusURI.getScheme(), 
                                nexusURI.getUserInfo(), 
                                nexusURI.getHost(),
                                nexusURI.getPort(),
                                nexusURI.getPath(),
                                null, 
                                null);
        UriBuilder uriBuilder = UriBuilder.fromUri(merged);
        for (String bout_de_groupe :artifact.getGroupId().split("\\.")){
            uriBuilder.segment(bout_de_groupe);
        }
        uriBuilder.segment(artifact.getArtifactId());
        uriBuilder.segment(artifact.getBaseVersion());
        StringBuilder filename = new StringBuilder();
        filename.append(artifact.getArtifactId());
        filename.append("-");
        filename.append(artifact.getVersion());
        if (artifact.getClassifier() != null && !artifact.getClassifier().isEmpty()) {
            filename.append("-");
            filename.append(artifact.getClassifier());
        }
        filename.append(".");
        filename.append(artifact.getExtension());
        uriBuilder.segment(filename.toString());
        String repositoryPath = uriBuilder.toString();

        return repositoryPath;
    }
}    
