package net.aequologica.neo.quintessence.aether;

import static net.aequologica.neo.geppaequo.config.ConfigRegistry.getConfig;
import static org.eclipse.aether.repository.RepositoryPolicy.CHECKSUM_POLICY_WARN;
import static org.eclipse.aether.repository.RepositoryPolicy.UPDATE_POLICY_ALWAYS;

import java.io.Closeable;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import org.apache.maven.repository.internal.MavenRepositorySystemUtils;
import org.eclipse.aether.DefaultRepositorySystemSession;
import org.eclipse.aether.RepositorySystem;
import org.eclipse.aether.RepositorySystemSession;
import org.eclipse.aether.connector.basic.BasicRepositoryConnectorFactory;
import org.eclipse.aether.impl.DefaultServiceLocator;
import org.eclipse.aether.repository.LocalRepository;
import org.eclipse.aether.repository.Proxy;
import org.eclipse.aether.repository.RemoteRepository;
import org.eclipse.aether.repository.RemoteRepository.Builder;
import org.eclipse.aether.repository.RepositoryPolicy;
import org.eclipse.aether.spi.connector.RepositoryConnectorFactory;
import org.eclipse.aether.spi.connector.transport.TransporterFactory;
import org.eclipse.aether.transport.file.FileTransporterFactory;
import org.eclipse.aether.transport.http.HttpTransporterFactory;

import net.aequologica.neo.geppaequo.config.ConfigRegistry;
import net.aequologica.neo.geppaequo.config.geppaequo.GeppaequoConfig;
import net.aequologica.neo.quintessence.aether.utils.ConsoleRepositoryListener;
import net.aequologica.neo.quintessence.aether.utils.ConsoleTransferListener;
import net.aequologica.neo.quintessence.aether.utils.TempDirectory;
import net.aequologica.neo.quintessence.config.QuintessenceConfig;

public class Booter implements Closeable {

    final private TempDirectory tempDirectory;
    final private GeppaequoConfig geppaequoConfig;
    final private Proxy proxy;
    
    public Booter() throws IOException {
        super();
        this.tempDirectory = new TempDirectory("local-repo");
        this.geppaequoConfig = ConfigRegistry.getConfig(GeppaequoConfig.class);
        if (this.geppaequoConfig.getProxyHost() != null && !this.geppaequoConfig.getProxyHost().isEmpty()) {
            this.proxy = new Proxy(null, this.geppaequoConfig.getProxyHost(), this.geppaequoConfig.getProxyPort());
        } else {
            this.proxy = null;
        }
        
    }

    public RepositorySystem newRepositorySystem() {
        /*
         * Aether's components implement org.eclipse.aether.spi.locator.Service
         * to ease manual wiring and using the prepopulated
         * DefaultServiceLocator, we only need to register the repository
         * connector and transporter factories.
         */
        DefaultServiceLocator locator = MavenRepositorySystemUtils.newServiceLocator();
        locator.addService(RepositoryConnectorFactory.class, BasicRepositoryConnectorFactory.class);
        locator.addService(TransporterFactory.class, FileTransporterFactory.class);
        locator.addService(TransporterFactory.class, HttpTransporterFactory.class);

        locator.setErrorHandler(new DefaultServiceLocator.ErrorHandler() {
            @Override
            public void serviceCreationFailed(Class<?> type, Class<?> impl, Throwable exception) {
                exception.printStackTrace();
            }
        });

        return locator.getService(RepositorySystem.class);
    }    

    public DefaultRepositorySystemSession newRepositorySystemSession(RepositorySystem system) throws IOException {
        DefaultRepositorySystemSession session = MavenRepositorySystemUtils.newSession();
            
        LocalRepository localRepo = new LocalRepository(tempDirectory.toString());
        session.setLocalRepositoryManager(system.newLocalRepositoryManager(session, localRepo));

        session.setTransferListener(new ConsoleTransferListener());
        session.setRepositoryListener(new ConsoleRepositoryListener());

        // uncomment to generate dirty trees
        session.setDependencyGraphTransformer( null );

        return session;
    }

    public List<RemoteRepository> newRepositories(RepositorySystem system, RepositorySystemSession session) {
        List<RemoteRepository> ret = new ArrayList<>();

        List<QuintessenceConfig.Repo> repositories = getConfig(QuintessenceConfig.class).getRepositoryObjects();
        for (QuintessenceConfig.Repo repository : repositories) {
            Builder builder = new RemoteRepository
                .Builder(repository.getName(), 
                         "default", 
                         repository.getUri().toString())
                .setPolicy(new RepositoryPolicy(true, UPDATE_POLICY_ALWAYS, CHECKSUM_POLICY_WARN));
            
            if (this.proxy != null && !this.geppaequoConfig.matchNonProxyHosts(repository.getUri().getHost())) {
                ret.add(builder.setProxy(this.proxy).build());
            } else {
                ret.add(builder.build());
            }
        }
        return ret;
    }

    @Override
    public void close() throws IOException {
        this.tempDirectory.delete();
    }

}