package net.aequologica.neo.quintessence.jaxrs;

import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.net.URI;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.inject.Singleton;
import javax.servlet.http.HttpServletResponse;
import javax.ws.rs.GET;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.UriBuilder;

import org.eclipse.aether.RepositorySystem;
import org.eclipse.aether.RepositorySystemSession;
import org.eclipse.aether.artifact.Artifact;
import org.eclipse.aether.artifact.DefaultArtifact;
import org.eclipse.aether.collection.CollectRequest;
import org.eclipse.aether.collection.CollectResult;
import org.eclipse.aether.graph.Dependency;
import org.eclipse.aether.repository.RemoteRepository;
import org.eclipse.aether.resolution.VersionRangeRequest;
import org.eclipse.aether.resolution.VersionRangeResolutionException;
import org.eclipse.aether.resolution.VersionRangeResult;
import org.eclipse.aether.resolution.VersionRequest;
import org.eclipse.aether.resolution.VersionResolutionException;
import org.eclipse.aether.resolution.VersionResult;
import org.eclipse.aether.version.Version;

import net.aequologica.neo.quintessence.aether.Booter;
import net.aequologica.neo.quintessence.aether.utils.ConsoleDependencyGraphDumper;

@Singleton
@javax.ws.rs.Path("/")
public class Resource {

    @GET
    @Produces("text/plain")
    public Response ping() {
        return Response.ok().entity("aether-api is here").build();
    }

    @GET
    @javax.ws.rs.Path("/{groupId: .+}/{artifactId : .+}")
    @Produces("text/plain")
    public Response latest(@PathParam("groupId") String groupId, @PathParam("artifactId") String artifactId) throws Exception {

        System.out.println(Resource.class.getSimpleName());
        
        try (Booter booter = new Booter()) {
        
            RepositorySystem system = booter.newRepositorySystem();
        
            RepositorySystemSession session = booter.newRepositorySystemSession(system);
        
            Artifact artifact = new DefaultArtifact(groupId+":"+artifactId+":[0,)");
        
            VersionRangeRequest rangeRequest = new VersionRangeRequest();
            rangeRequest.setArtifact(artifact);
            rangeRequest.setRepositories(booter.newRepositories(system, session));
        
            VersionRangeResult rangeResult = system.resolveVersionRange(session, rangeRequest);
        
            Version newestVersion = rangeResult.getHighestVersion();
        
            System.out.println(
                    "Newest version " + newestVersion + " from repository " + rangeResult.getRepository(newestVersion));
        
            return Response.ok().entity(newestVersion.toString()).build();
        }
    }

    @GET
    @javax.ws.rs.Path("/{groupId: .+}/{artifactId : .+}/{version : .+}")
    @Produces("text/plain")
    public Response tree(@PathParam("groupId") String groupId, @PathParam("artifactId") String artifactId, @PathParam("version") String version) throws Exception {
        
        System.out.println( "------------------------------------------------------------" );
        System.out.println( Resource.class.getSimpleName() );
    
        try (Booter booter = new Booter()) {
            
            RepositorySystem system = booter.newRepositorySystem();
        
            RepositorySystemSession session = booter.newRepositorySystemSession( system );
        
            Artifact artifact = new DefaultArtifact( groupId+":"+artifactId+":"+version );
        
            CollectRequest collectRequest = new CollectRequest();
            collectRequest.setRoot( new Dependency( artifact, "" ) );
            collectRequest.setRepositories( booter.newRepositories( system, session ) );
        
            CollectResult collectResult = system.collectDependencies( session, collectRequest );
        
            StringWriter sw = new StringWriter();
            PrintWriter out = new PrintWriter(sw);
            
            collectResult.getRoot().accept( new ConsoleDependencyGraphDumper(out) );    
    
            return Response.ok().entity(sw.toString()).build();
        }
    }
    
    VersionRangeResolverImpl versionRangeResolver = new VersionRangeResolverImpl();
    
    @GET
    @javax.ws.rs.Path("/legacy")
    @Produces( "application/json" )
    public Response resolve(
            @Context           final HttpServletResponse httpServletResponse,
            @QueryParam( "r" ) final List<String> repositories,
            @QueryParam( "g" ) final String  groupId, 
            @QueryParam( "a" ) final String  artifactId, 
            @QueryParam( "e" ) final String  extension, 
            @QueryParam( "c" ) final String  classifier, 
            @QueryParam( "v" ) final String  versionRange,
            @QueryParam( "p" ) final String  packaging) {

        try {
            RepositoryHolder repositoryHolder = new RepositoryHolder();
            
            Artifact artifact = versionRangeResolver.resolve(
                    repositories,
                    groupId, 
                    artifactId, 
                    classifier,
                    extension, 
                    versionRange,
                    repositoryHolder);

            String downloadURI = buildDownloadURI(URI.create(repositoryHolder.remoteRepository.getUrl()), artifact);
            httpServletResponse.setHeader("Location", downloadURI);
                
            return Response.ok().entity(artifact).build();

        } catch (IllegalArgumentException iae) {
            return Response.status(422).entity(iae.getMessage()).build();
        } catch (VersionRangeResolutionException vrre) {
            return Response.status(404).entity(vrre.getMessage()).build();
        } catch (VersionResolutionException vre) {
            return Response.status(404).entity(vre.getMessage()).build();
        } catch (Throwable t) {
            return Response.status(500).entity(t.getMessage()).build();
        }
    }

    public class RepositoryHolder {
        public String repositoryId;
        public RemoteRepository remoteRepository;
    }

    private class VersionRangeResolverImpl {

        public Artifact resolve(    final List<String> repositories, 
                                    final String groupId, 
                                    final String artifactId, 
                                    final String classifier, 
                                    final String extension, 
                                    final String version, 
                                    RepositoryHolder repositoryHolder) throws IOException, VersionRangeResolutionException, VersionResolutionException {
            
            Artifact artifact = new DefaultArtifact(groupId, artifactId, classifier, extension, version);

            if (repositories.size() == 0      ||
                StringUtils.isEmpty(artifact.getGroupId())      ||
                StringUtils.isEmpty(artifact.getArtifactId())   ||
                StringUtils.isEmpty(artifact.getVersion())) {
                throw new IllegalArgumentException("at least one of [ group id | artifact id | version range | repository ] is missing.");
            }
                
            if (!isVersionRange(artifact.getVersion())) {
                throw new IllegalArgumentException("« "+artifact.getVersion()+" » does not seem to be a valid version range; cf. https://docs.oracle.com/middleware/1212/core/MAVEN/maven_version.htm#MAVEN402 ");
            }
            
            Artifact ret = artifact;
            
            try (Booter booter = new Booter()) {
                
                RepositorySystem system = booter.newRepositorySystem();
                RepositorySystemSession session = booter.newRepositorySystemSession(system);
    
                VersionRangeRequest rangeRequest = new VersionRangeRequest();
                rangeRequest.setArtifact(artifact);
                List<RemoteRepository> newRepositories = booter.newRepositories(system, session);
                if (repositories == null || repositories.size() == 0) {
                    rangeRequest.setRepositories(newRepositories);
                }
                for (RemoteRepository remoteRepository : newRepositories) {
                    if (repositories.contains(remoteRepository.getId())) {
                        rangeRequest.addRepository(remoteRepository);
                    }
                }

                VersionRangeResult rangeResult = system.resolveVersionRange(session, rangeRequest);

                /*
                Version newestReleaseVersion = null;
                Version newestSnapshotVersion = null;
    
                for (Exception e  : rangeResult.getExceptions()) {
                    if (e instanceof MetadataNotFoundException) {
                        MetadataNotFoundException mnfe = (MetadataNotFoundException)e;
                        if ((mnfe.getRepository() != null && mnfe.getRepository().equals(aetherObjects.getRemoteRepository()))
                            ||
                            mnfe.getMessage().contains(aetherObjects.getRemoteRepository().getUrl())) {
                            throw new VersionRangeResolutionException(rangeResult, "metadata not found");
                       }
                    }
                }
                
                for (Version version : rangeResult.getVersions()) {
                    if (!version.toString().endsWith("SNAPSHOT")) {
                        newestReleaseVersion = version;
                    } else {
                        newestSnapshotVersion = version;
                    }
                }
                */
                
                @SuppressWarnings("unused")
                Version lowestVersion = rangeResult.getLowestVersion();
                Version newestVersion = rangeResult.getHighestVersion();
                
                if (newestVersion == null) {
                    throw new VersionRangeResolutionException(rangeResult);
                }

                repositoryHolder.repositoryId = rangeResult.getRepository(newestVersion).getId();
                for (RemoteRepository remoteRepository : newRepositories) {
                    if (remoteRepository.getId().equals(repositoryHolder.repositoryId)) {
                        repositoryHolder.remoteRepository = remoteRepository;
                    }
                }
                
                
                String resolvedSnapshotVersion = null;
                if (newestVersion.toString().endsWith("SNAPSHOT")) {
                    VersionRequest versionRequest = new VersionRequest();
                    ret = new DefaultArtifact(artifact.getGroupId(), artifact.getArtifactId(), artifact.getClassifier(), artifact.getExtension(), newestVersion.toString());
                    versionRequest.setArtifact(ret);
                    versionRequest.addRepository(repositoryHolder.remoteRepository);
                    VersionResult versionResult = system.resolveVersion( session, versionRequest );
                    
                    /*
                    for (Exception e2  : versionResult.getExceptions()) {
                        if (e2 instanceof MetadataNotFoundException) {
                            MetadataNotFoundException mnfe = (MetadataNotFoundException)e2;
                            if ((mnfe.getRepository() != null && mnfe.getRepository().equals(aetherObjects.getRemoteRepository()))
                                ||
                                mnfe.getMessage().contains(aetherObjects.getRemoteRepository().getUrl())) {
                                throw new VersionResolutionException(versionResult);
                            }
                        }
                    }
                    */
    
                    resolvedSnapshotVersion = versionResult.getVersion();
                }
    
                Artifact artifactResult = new DefaultArtifact(
                        ret.getGroupId(), 
                        ret.getArtifactId(), 
                        ret.getClassifier(), 
                        ret.getExtension(), 
                        resolvedSnapshotVersion==null?newestVersion.toString():resolvedSnapshotVersion);
                
                return artifactResult;
            }
            
        }

        boolean isVersionRange(String versionRange) {
            versionRange = StringUtils.trimToEmpty(versionRange);
            
            if (versionRange.length()<3) {
                return false;
            }

            // special cases for compatibility with nexus artifact.resolve REST service
            if ("LATEST".equals(versionRange)) {
                return true;
            }
            if ("RELEASE".equals(versionRange)) {
                return true;
            }
            if (versionRange.endsWith("SNAPSHOT")) {
                return true;
            }
            
            // true version range parsing
            if (versionRange.charAt(0) != '[' && versionRange.charAt(0) != '(' ) {
                return false;
            }
            if (versionRange.charAt(versionRange.length()-1) != ']' && versionRange.charAt(versionRange.length()-1) != ')') {
                return false;
            }
            if (-1 == versionRange.indexOf(',')) {
                return false;
            }
            
            return true;
            
        }

        @SuppressWarnings("unused")
        public long[] parseTimeStampBuildNumberFromSnapshot(String snapshotVersion) {
            long[] ret = new long[]{0L, 0L};
            
            Pattern p = Pattern.compile( "([^- ]+)-([^- ]+)-([^- ]+)" );
            Matcher m = p.matcher( snapshotVersion );
            if ( !m.matches() || m.groupCount()!=3) {
                throw new IllegalArgumentException( "Bad snapshotVersion coordinates " + snapshotVersion
                    + ", expected format is <non snapshot version>-<timestamp>-<build number>" );
            }
            String timestamp = m.group( 2 );
            int dot = timestamp.indexOf('.');
            if (dot != -1) {
                timestamp = timestamp.substring(0, dot) + timestamp.substring(dot+1);  
            }
            ret[0] = Long.parseLong(timestamp);
            ret[1] = Long.parseLong(m.group( 3 ));
            
            return ret;
        }
        
    }
    
    public static class StringUtils {

        public static String trimToEmpty(String string) {
            return string == null ? "" : string.trim();
        }

        public static boolean isEmpty(String string) {
            if (string == null) {
                return true;
            }
            if (string.trim().length() == 0) {
                return true;
            }
            return false;
        }

    }
    
    private String buildDownloadURI(final URI nexusURI, Artifact artifact) throws Exception {
        URI merged = new URI(   nexusURI.getScheme(), 
                                nexusURI.getUserInfo(), 
                                nexusURI.getHost(),
                                nexusURI.getPort(),
                                nexusURI.getPath(),
                                null, 
                                null);
        UriBuilder uriBuilder = UriBuilder.fromUri(merged);
        for (String bout_de_groupe :artifact.getGroupId().split("\\.")){
            uriBuilder.segment(bout_de_groupe);
        }
        uriBuilder.segment(artifact.getArtifactId());
        uriBuilder.segment(artifact.getBaseVersion());
        StringBuilder filename = new StringBuilder();
        filename.append(artifact.getArtifactId());
        filename.append("-");
        filename.append(artifact.getVersion());
        if (artifact.getClassifier() != null && !artifact.getClassifier().isEmpty()) {
            filename.append("-");
            filename.append(artifact.getClassifier());
        }
        filename.append(".");
        filename.append(artifact.getExtension());
        uriBuilder.segment(filename.toString());
        String repositoryPath = uriBuilder.toString();

        return repositoryPath;
    }
}    
