package net.aequologica.neo.geppaequo.config;

import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.Modifier;
import java.net.JarURLConnection;
import java.net.URL;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;
import java.util.jar.Attributes;
import java.util.jar.Manifest;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.joda.time.DateTime;
import org.joda.time.format.DateTimeFormatter;
import org.joda.time.format.ISODateTimeFormat;
import org.reflections.Reflections;
import org.reflections.scanners.SubTypesScanner;
import org.reflections.scanners.TypeAnnotationsScanner;
import org.reflections.util.ClasspathHelper;
import org.reflections.util.ConfigurationBuilder;
import org.reflections.util.FilterBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public enum ConfigRegistry {
    
    CONFIG_REGISTRY;
    
    private final Logger log = LoggerFactory.getLogger(ConfigRegistry.class);
    /*   */ final String DEFAULT_PREFIX  = "/META-INF/resources/wizardry";

    private final Map <Class<? extends AbstractConfig>, AbstractConfig> map;
	private final Map <String, AbstractConfig>                          map2;
	private final String[]                                              packages;
	private final DateTimeFormatter                                     dateTimeFormatter;
	private final String                                                splitter = ".";
    
	ConfigRegistry() {
        this.map  = new HashMap<>();
        this.map2 = new HashMap<>();
        final String DEFAULT_PACKAGE =  Stream.of(ConfigRegistry.class.getPackage().getName().split("\\"+splitter)).limit(3).collect(Collectors.joining( splitter ) );
        this.packages = new String[] { DEFAULT_PACKAGE };
        this.dateTimeFormatter = ISODateTimeFormat.dateTime();
        
        scanConfigs();
    }

    public void registerConfig(AbstractConfig instance) {
        map.put(instance.getClass(), instance);
        map2.put(instance.getMetadata().getName(), instance);
    }

    public void unregisterConfig(Class<? extends AbstractConfig> clazz) {
        AbstractConfig removed = map.remove(clazz);
        map2.remove(removed.getMetadata().getName());
    }

    public <T> T getConfig(Class<T> clazz) {
        return clazz.cast(map.get(clazz));
    }

    public AbstractConfig getConfig(String name) {
        return map2.get(name);
    }

    public Map <String, AbstractConfig> getConfigMap() {
        return Collections.unmodifiableMap(map2);
    }

    public Set<AbstractConfig> getConfigs() {
        return Collections.unmodifiableSet(new HashSet<AbstractConfig>(map.values()));
    }

    public Set<String> getConfigNames() {
        TreeSet<String> names = new TreeSet<>();
        for (AbstractConfig config : map.values()) {
            names.add(config.getMetadata().getName());
        }
        return Collections.unmodifiableSortedSet(names);
    }

    
    void clearConfigs() {
        map.clear();
        map2.clear();
    }

    void scanConfigs() {
        scanConfigs(packages);
    }

    private DateTime safeToDate(Object o) {
        return o == null ? null : dateTimeFormatter.parseDateTime(o.toString());
    }

    public void scanConfigs(String[] packages) {
        for (String prefix: packages) {

            Reflections reflections = new Reflections(
                    new ConfigurationBuilder()
                        .setUrls(ClasspathHelper.forPackage(prefix))
                        .setScanners(
                            new SubTypesScanner(true),
                            new TypeAnnotationsScanner()
                        )
                        .filterInputsBy(new FilterBuilder().include(".*\\.class"))
                    );

            Set<Class<?>> classes = reflections.getTypesAnnotatedWith(Config.class);

            log.debug("Found {} classe(s) annotated with {} -> {}", classes.size(), Config.class.getName(), classes);

            for (Class<?> clazz : classes) {

                Config configAnnotation = clazz.getAnnotation(Config.class);
                if (configAnnotation == null) {
                    continue;
                }
                
                final String  buildTime = getBuildTimeFromManifestInSameJar(clazz);
                final Package package_  = clazz.getPackage();
                final String  g = "net.aequologica.neo";
                final String  a = package_.getImplementationTitle();
                final String  v = package_.getImplementationVersion();

                String name = configAnnotation.name();

                int modifiers = clazz.getModifiers();

                if (Modifier.isAbstract( modifiers )) {
                    log.debug("{} is abstract. ignored.", clazz.getName());
                    continue;
                }
                if (!Modifier.isPublic( modifiers )) {
                    log.debug("{} is not public. ignored.", clazz.getName());
                    continue;
                }

                AbstractConfig config = null;
                try {
                    config = (AbstractConfig)clazz.newInstance();
                    config.setBuildTime(safeToDate(buildTime));
                    config.setGav(new AbstractConfig.GAV(g, a, v));
                    registerConfig(config);
                } catch (Exception e) {
                    log.error("instantiation of class {} failed with exception: {}", clazz, e);
                    continue;
                }

                log.info("Config created [\"{}\" -> {} -> {}]", name, clazz, config);
            }
        }
    }

    private String getBuildTimeFromManifestInSameJar(Class<?> clazz) {
        try {
            if (clazz == null) {
                return null;
            }
            URL resource = clazz.getResource(clazz.getSimpleName() + ".class");
            if (resource == null) {
                return null;
            }
            String resourceAsString = resource.toString();
            if (resourceAsString.startsWith("jar:")) {
                return getBuildTimeFromJarManifest(resource);      
            }
            if (resourceAsString.startsWith("file:")) {
                int webInfIndex = resourceAsString.indexOf("WEB-INF"); 
                if (webInfIndex != -1) {
                    String manifestResource = resourceAsString.substring(0, webInfIndex) + "META-INF/MANIFEST.MF";
                    try (InputStream stream = new URL(manifestResource).openStream()) {
                        Manifest manifest = new Manifest(stream);
                        Attributes attributes = manifest.getMainAttributes();
                        return attributes.getValue("Build-Time");
                    }
                }
            }
        } catch (IOException e) {
            log.error("ignored exception", e);
        }
        return null;
   }

    private String getBuildTimeFromJarManifest(URL url) throws IOException {
        try {
            JarURLConnection jarConnection = (JarURLConnection) url.openConnection();
            Manifest manifest = jarConnection.getManifest();
            Attributes attributes = manifest.getMainAttributes();
            return attributes.getValue("Build-Time");
        } catch (NullPointerException npe) {
            log.error("ignored null pointer exception in getBuildTimeFromJarManifest", npe);
            return "";
        }
    }
}
