package net.aequologica.neo.geppaequo.config;

import static net.aequologica.neo.geppaequo.cmis.impl.ECMHelperFactory.ECMHELPERFACTORY;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.io.Reader;
import java.io.StringWriter;
import java.io.Writer;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;

import javax.naming.Context;
import javax.naming.NamingException;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.weakref.jmx.Managed;
import org.weakref.jmx.Nested;

import com.google.common.base.Charsets;

import net.aequologica.neo.geppaequo.cmis.ECMHelper;
import net.aequologica.neo.geppaequo.document.DocumentHelper;
import net.aequologica.neo.geppaequo.jndi.JndiUtil;

import com.typesafe.config.Config;
import com.typesafe.config.ConfigFactory;
import com.typesafe.config.ConfigParseOptions;
import com.typesafe.config.ConfigRenderOptions;
import com.typesafe.config.ConfigValue;
import com.typesafe.config.ConfigValueFactory;
public final class ConfigManager {

    static final Logger log = LoggerFactory.getLogger(ConfigManager.class);

    // mandatory in constructor
    private final String          applicationName;
    private final ContextProvider contextProvider;

    // constructor
    ConfigManager(final String applicationName, final ContextProvider contextProvider) {
        this.applicationName = applicationName;
        this.contextProvider = contextProvider;
    }

    // the final package configuration = merged from 1) application + 2) user + 3) system configs (cf .just below)
    Config                            config;

    private final ConfigSource        system              = new ConfigSourceSystem();       // systemconfig
    private final ConfigSource        user                = new ConfigSourceUser();         // userconfig
    private final ConfigSource        application         = new ConfigSourceApplication();  // applicationconfig

    private final static ConfigRenderOptions configRenderOptions = ConfigRenderOptions.defaults()
                                                                                      .setComments(true)
                                                                                      .setFormatted(true)
                                                                                      .setJson(false)
                                                                                      .setOriginComments(false);

    // generic setter
    // will throw exception if key is not found
    public void set(final String originalKey, final Object object) {
        final String key = applicationName + "." + originalKey;
        if (object == null || (object instanceof String && ((String) object).length() == 0)) {
            // remove
            application.setConfig(application.getConfig().withoutPath(key));
        } else {
            // create or update
            ConfigValue value = ConfigValueFactory.fromAnyRef(object, "modified at runtime ");
            application.setConfig(application.getConfig().withValue(key, value));
        }
        // merge global config (no reloading)
        config = application.getConfig().withFallback(user.getConfig()).withFallback(system.getConfig());
    }

    // THE BIG RESOLVE
    void resolveConfigURIs() throws IOException {
        system.resolveLocation();
        user.resolveLocation();
        application.resolveLocation();
    }

    // THE BIG LOAD AND MERGE
    void loadConfigs() throws IOException {

        application.loadConfig();
        user.loadConfig();
        system.loadConfig();

        config = application.getConfig().withFallback(user.getConfig()).withFallback(system.getConfig());
        dump2debugLog("MERGED", config);
    }

    @Managed(description = "system configuration")
    @Nested
    public ConfigSource getSystem() {
        return system;
    }

    @Managed(description = "user configuration")
    @Nested
    public ConfigSource getUser() {
        return user;
    }

    @Managed(description = "application configuration")
    @Nested
    public ConfigSource getApplication() {
        return application;
    }

    @Managed(description = "reload and merge all configurations (this will undo any unsaved change)")
    public void reload() throws IOException {
        resolveConfigURIs();
        loadConfigs();
    }

    @Override
    public String toString() {
        return config.origin().toString();
    }

    interface ConfigSource {
        URI getLocation();

        Config getConfig();

        void setConfig(Config config);

        void resolveLocation();

        void loadConfig() throws IOException;

        String save() throws IOException;

        @Override
        String toString();
    }

    public abstract class AbstractConfigSource implements ConfigSource {
        protected final String name;
        protected URI          uri;
        protected Config       config = ConfigFactory.empty();

        protected AbstractConfigSource(String name) {
            this.name = name;
        }

        @Override
        @Managed
        public URI getLocation() {
            return uri;
        }

        @Override
        public Config getConfig() {
            return config;
        }

        @Override
        public void setConfig(Config config) {
            this.config = config;
        }

        @Override
        public abstract void resolveLocation();

        @Override
        public String save() throws IOException {
            throw new UnsupportedOperationException();
        }

        @Override
        public void loadConfig() throws IOException {
            if (uri != null) {
                Config localConfig = ConfigFactory.empty();
                localConfig = ConfigFactory.parseURL(uri.toURL(), ConfigParseOptions.defaults().setOriginDescription(uri.toString()));
                config = localConfig;
                dump2debugLog(name, config);
            } else {
                log.debug("nothing to load, uri is null for {}", getClass().getSimpleName());
            }
        }

        @Override
        public String toString() {
            return name + " " + config.origin().toString();
        }

    } // end of class ConfigSource

    static void dumpConfig(final String name, final Config config, final PrintWriter printWriter) {

        if (printWriter == null) {
            throw new IllegalArgumentException("writer argument cannot be null");
        }

        printWriter.println();
        printWriter.println("^  ^  ^  ^  ^  ^  ^  ^  ^  ^  ^  ^  ^  ^  ^  ^  ^  ^  ^  ^  ^  ^  ^  ^  ^  ^  ^  ^  ^  ^");
        printWriter.println(name + " " + config.root().origin());
        printWriter.println(config.root().render(configRenderOptions));
        printWriter.flush();
    }

    static void dump2debugLog(String name, final Config config) {
        if (log.isDebugEnabled()) {
            final StringWriter sw = new StringWriter();
            try (PrintWriter pw = new PrintWriter(sw)) {
                dumpConfig(name, config, pw);
                log.debug(sw.toString());
            }
        }
    }

    public final class ConfigSourceSystem extends AbstractConfigSource {

        protected ConfigSourceSystem() {
            super("SYSTEM");
        }

        /**
         * <pre>
         * look for
         * 1. "/META-INF/resources/wizardry/.<applicationName>/application-test.conf" on the classpath.
         *
         * If not found, look for
         * 2. "/META-INF/resources/wizardry/.<applicationName>/application.conf" on the classpath
         * </pre>
         */
        @Override
        public void resolveLocation() {

            uri = null;
            final String DEFAULT_PREFIX     = "/META-INF/resources/wizardry";
            final String start              = DEFAULT_PREFIX + "/." + applicationName + "/";
            final String uriAsStringForTest = start + "application-test.conf";
            final String uriAsString        = start + "application.conf";

            uri = resolveLocationClasspath(uriAsStringForTest);
            if (uri == null) {
                uri = resolveLocationClasspath(uriAsString);
            }
        }

    } // end of class ConfigSourceSystem

    public final class ConfigSourceUser extends AbstractConfigSource {

        protected ConfigSourceUser() {
            super("USER");
        }

        /**
         * <pre>
         * look for "~/.<applicationName>/application.conf" on the file system
         * </pre>
         */
        @Override
        public void resolveLocation() {

            uri = null;

            final File userHome = new File(System.getProperty("user.home"));
            final File userAppConfigDir = new File(userHome, "." + applicationName);
            final File userAppConfigFile = new File(userAppConfigDir, "application.conf");
            if (userAppConfigFile.exists()) {
                uri = userAppConfigFile.toURI();
            } else {
                log.warn("File '" + userAppConfigFile.getAbsolutePath() + "' does not exist.");
            }
        }
    } // end of class ConfigSourceUser

    public final class ConfigSourceApplication extends AbstractConfigSource {

        protected ConfigSourceApplication() {
            super("APPLICATION");
        }

        /**
         * <pre>
         * if running inside neo,
         * ==> look for a file in HANA Cloud Document Service,
         * ==> at path "/.<applicationName>/application.conf".
         *
         * if running inside plain tomcat,
         * ==> look for "/.<applicationName>/application.conf" environment variable in JNDI,
         * ==> then look for the actual file on the file system
         *
         *      or try the document helper
         * </pre>
         */
        @Override
        public void resolveLocation() {

            uri = null;

            // running under neo ? try HANA Cloud Document Service
            if (ECMHELPERFACTORY.getEcmService() != null) {
                final String path = "/." + applicationName + "/application.conf";
                try {
                    uri = new URI("cmis", path, null);
                } catch (URISyntaxException ignored) {
                    log.warn("'cmis:{}' is not a valid URI. Exception: {}", path, ignored.getMessage());
                }
            }

            if (uri == null) { // probably running under tomcat
                final String path = getFilePathFromJNDI();
                if (path != null && path.length() > 0) {
                    final File file = new File(path);
                    uri = file.toURI();
                } else {
                    Path path2 = Paths.get("/." + applicationName, "application.conf");
                    List<String> documentPaths2;
                    try {
                        documentPaths2 = DocumentHelper.getDocumentPaths(path2);
                        if (documentPaths2.size() > 0) {
                            uri = new File(documentPaths2.get(0)).toURI();
                        }
                    } catch (IOException ignored2) {
                        log.warn("ignoring {} raised by DocumentHelper.getDocumentPaths(\"{}\"): {}", ignored2.getClass().getName(), path2, ignored2.getMessage());
                    }
                }
            }
        }

        @Override
        public void loadConfig() throws IOException {

            config = ConfigFactory.empty();

            if (uri != null) {
                if (uri.getScheme() != null) {
                    String applicationConfigAsAString = null;

                    if (uri.getScheme().equals("cmis")) {
                        try (ECMHelper ecmHelper = ECMHELPERFACTORY.getEcmHelper()) {
                            applicationConfigAsAString = ecmHelper.readUTF8Document(Paths.get(uri.getPath()));
                        } catch (IOException e) {
                            log.warn(e.getMessage());
                        }
                    } else if (uri.getScheme().equals("file")) {
                        final File file = new File(uri.getPath());
                        if (!file.exists()) {
                            log.warn("file '{}' not found.", file.getAbsolutePath());
                        } else {
                            // http://stackoverflow.com/questions/696626/java-filereader-encoding-issue
                            try (Reader fileReader = new InputStreamReader(new FileInputStream(file), Charsets.UTF_8.name())) {
                                applicationConfigAsAString = ConfigManager.reader2String(fileReader);
                            } catch (Exception ignored) {
                                log.warn("error reading '{}': {}", uri, ignored.getMessage());
                                applicationConfigAsAString = null;
                            }
                        }
                    }

                    if (applicationConfigAsAString != null) {
                        config = ConfigFactory.parseString(applicationConfigAsAString, ConfigParseOptions.defaults().setOriginDescription(uri.toString()));
                        dump2debugLog(name, config);
                    }
                }
            }
        }

        @Override
        @Managed(description = "save application configuration")
        public String save() throws IOException {
            final String ret = save(null);
            log.info(ret);
            return ret;
        }

        /**
         * <pre>
         * this save function variant with a file argument is only here to enable testing
         * hence the 'package' visibility
         * </pre>
         */
        String save(File fileArgument) throws IOException {
            if (config == null) {
                return "Current application config is null. This is a bug. Contact the administrator.";
            }

            if (uri != null) {
                if (uri.getScheme() != null) {
                    if (uri.getScheme().equals("cmis")) {
                        if (uri.getPath() != null) {
                            final Path path = Paths.get(uri.getPath());
                            try (ECMHelper ecmHelper = ECMHELPERFACTORY.getEcmHelper()) {
                                final String configAsAString = config.root().render(configRenderOptions);
                                ecmHelper.createOrUpdateUTF8Document(path, configAsAString);
                                log.debug("\n{}\n saved @ {}", configAsAString, path);
                            }
                            return "New configuration saved to \n\t" + path;
                        }
                    }
                }
            }

            // rename the existing file with a timestamp
            final String standardName;
            if (fileArgument != null) {
                standardName = fileArgument.getCanonicalPath();
            } else {
                if (uri == null || uri.getScheme() == null || !uri.getScheme().equals("file")) {
                    throw new IOException("uri is null, or uri.scheme is null, or uri.scheme is not 'file' : nowhere to save ...");
                }
                standardName = uri.getPath();
            }
            final String timestampedName;
            int lastDot = standardName.lastIndexOf('.');
            if (lastDot != -1) {
                timestampedName = standardName.substring(0, lastDot) + getTimeStamp() + standardName.substring(lastDot);
            } else {
                timestampedName = standardName + getTimeStamp();
            }
            final File backup = new File(timestampedName);
            new File(standardName).renameTo(backup);

            // write the new content to the standard application config filename
            final File newConfig = new File(standardName);
            try (Writer fileWriter = new OutputStreamWriter(new FileOutputStream(newConfig), Charsets.UTF_8.name())) {
                final String configAsAString = config.root().render(configRenderOptions);
                fileWriter.write(configAsAString);
                fileWriter.flush();
                fileWriter.close();
            }
            return "New configuration saved to \n\t" + newConfig.getCanonicalPath() + "\nPrevious configuration backuped to \n\t" + backup.getCanonicalPath();
        }
    } // end of class ConfigSourceApplication

    //
    // helper methods
    //

    private String getFilePathFromJNDI() {
        String ret = null;
        String contextString = null; // just for clearer logging messages

        // give testing the opportunity to pass a file system naming context
        // (used for testing)
        Context ctx = contextProvider.getNamingContext();

        if (ctx == null) {
            final String name = JndiUtil.TOMCAT_MAGIC_JNDI_ENV_STRING;
            try {
                ctx = new JndiUtil(name).getContext();
                contextString = name;
            } catch (NamingException e) {
                log.warn("'" + name + "' naming context not found. Exception is [" + e.getMessage() + "]");
                return null;
            }
        }

        String lookupName = "/." + applicationName + "/application.conf";
        try {
            final Object obj = ctx.lookup(lookupName);
            if (obj instanceof File) {
                ret = ((File) obj).getAbsolutePath();
            } else if (obj.toString().length() > 0) {
                ret = obj.toString();
            } else {
                log.warn("Name '{}' in JNDI context '{}' must be of type File or String", lookupName, contextString != null ? contextString : ctx);
                return null;
            }
        } catch (Exception ignored) {
            log.warn("Name '{}' not found in JNDI context '{}'.", lookupName, contextString != null ? contextString : ctx);
            return null;
        }
        return ret;
    }

    private final static DateFormat FORMAT = new SimpleDateFormat("-yyyyMMdd.hhmmss");

    private static String getTimeStamp() {
        synchronized (FORMAT) {
            final String timeStamp = FORMAT.format(new Date());
            return timeStamp;
        }
    }

    // convert Reader to String
    private static String reader2String(final Reader reader) throws IOException {

        final StringBuilder stringBuilder = new StringBuilder();

        try (BufferedReader bufferedReader = new BufferedReader(reader)) {
            String line = null;
            while ((line = bufferedReader.readLine()) != null) {
                stringBuilder.append(line);
                stringBuilder.append("\n"); // I know, if the original data
                                            // stream has no ending line feed,
                                            // this will add one...
            }
        }
        return stringBuilder.toString();
    }

    private static URI resolveLocationClasspath(final String string) {
        URI uri = null;
        try {
            final URL url = ConfigManager.class.getResource(string);
            if (url == null) {
                log.warn("'{}' not found on the classpath.", string);
            } else {
                final String urlNoSpaces = url.toExternalForm().replace(" ", "%20");
                uri = new URI(urlNoSpaces);
            }
        } catch (Exception e) {
            log.warn("'{}' not found on the class path, or cannot convert classpath to a valid URI : {}", string, e);
        }
        return uri;
    }

}
