package net.aequologica.neo.geppaequo.config;

import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.Arrays;
import java.util.List;

import javax.naming.Context;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.weakref.jmx.Managed;
import org.weakref.jmx.Nested;

import net.aequologica.neo.geppaequo.git.GitRepositoryState;

/**
 * 1. verify in constructor that concrete sub classes declare a valid @Config annotation
 * 2. embed a ConfigManager to read / write 3 level configs (system, app, user)
 * 3. at construction time, register config to ConfigRegistry, keyed on class name, nb. any config previously registered will be gone, lost
 * 4. exposes the underlying typesafe Config interface, proxied (cf. getProxy()) to adjust key with metadata.name
 */
public abstract class AbstractConfig implements ContextProvider {

    static final Logger log = LoggerFactory.getLogger(AbstractConfig.class);

    private final ConfigManager serializer;
    private final Metadata      metadata;

    private GitRepositoryState gitRepositoryState = null;

    public AbstractConfig() throws IOException {

        Config configAnnotation = this.getClass().getAnnotation(Config.class);

        if (configAnnotation == null) {
            throw new RuntimeException("This class must be annotated with the '" + Config.class.getName() + "' annotation, with a not null, not empty, all-lowercase 'name' attribute.");
        }

        String name = configAnnotation.name();

        if (name == null || name.length() == 0 || !name.toLowerCase().equals(name)) {
            throw new RuntimeException(Config.class.getName() + ".name (\"" + name + "\") annotation attribute must be not null, not empty, and all lowercase.");
        }

        this.metadata = new Metadata(name);
        this.serializer = new ConfigManager(name, this);
        this.serializer.resolveConfigURIs();
        this.serializer.loadConfigs();

        ConfigRegistry.registerConfig(this);
    }

    /**
     * cf. net.aequologica.neo.geppaequo.config.ContextProvider
     */
    @Override
    public Context getNamingContext() {
        return null;
    }

    @Managed
    @Nested
    public ConfigManager getConfiguration() {
        return serializer;
    }

    @Managed
    @Nested
    public Metadata getMetadata() {
        return metadata;
    }

    @Managed(description = "git info")
    @Nested
    public GitRepositoryState getGitRepositoryState() {
        return gitRepositoryState;
    }

    public void setGitRepositoryState(GitRepositoryState gitRepositoryState ) {
        this.gitRepositoryState = gitRepositoryState;
    }

    protected String get(final String key) {
        // fetch from underlying typesafe config, proxied
        return getProxy().getString(key);
    }

    protected Integer getInt(final String key) {
        // fetch from underlying typesafe config, proxied
        return getProxy().getInt(key);
    }

    protected List<String> getStringList(final String key) {
        // fetch from underlying typesafe config, proxied
        return getProxy().getStringList(key);
    }

    protected void set(final String key, final Object value) {
        // send to configuration, to take care of serialization
        this.serializer.set(key, value);
    }

    public void reload() throws Exception {
        this.serializer.reload();
    }

    public static class Metadata {
        final private String name;

        Metadata(final String name) {
            this.name = name;
        }

        @Managed
        public String getName() {
            return this.name;
        }
    }

    private com.typesafe.config.Config proxy;

    public class ConfigProxy implements java.lang.reflect.InvocationHandler {

        @Override
        public Object invoke(Object proxy, Method m, Object[] args) throws Throwable {
            Object result;
            try {
                Object[] newArgs = Arrays.copyOf(args, args.length);
                int i = 0;
                for (Object object : args) {
                    // only add metadata.name to first String argument
                    if (object instanceof String) {
                        newArgs[i] = metadata.name + "." + (String) object;
                        break;
                    }
                    i++;
                }
                result = m.invoke(serializer.config, newArgs);
            } catch (InvocationTargetException e) {
                throw e.getTargetException();
            } catch (Exception e) {
                throw new RuntimeException("unexpected invocation exception: " + e.getMessage(), e);
            } finally {
            }
            return result;
        }
    }

    protected com.typesafe.config.Config getProxy() {
        if (proxy == null) {
            proxy = (com.typesafe.config.Config) java.lang.reflect.Proxy.newProxyInstance(serializer.config.getClass().getClassLoader(),
                    new Class<?>[] { com.typesafe.config.Config.class }, new ConfigProxy());
        }
        return proxy;
    }
}
