package net.aequologica.neo.dagr.bus;

import static net.aequologica.neo.dagr.bus.BusEvent.Type.BUILD_ERROR;
import static net.aequologica.neo.dagr.bus.BusEvent.Type.BUILD_OK;
import static net.aequologica.neo.dagr.bus.BusEvent.Type.BUILD_STARTED;
import static net.aequologica.neo.dagr.bus.BusEvent.Type.DEPENDENCY;
import static net.aequologica.neo.dagr.bus.BusEvent.Type.PUSH;
import static net.aequologica.neo.dagr.bus.BusEvent.Type.TRAVIS_ERROR;
import static net.aequologica.neo.dagr.bus.BusEvent.Type.TRAVIS_OK;
import static net.aequologica.neo.dagr.model.Dag.Node.State.BEING_CLEANED;
import static net.aequologica.neo.dagr.model.Dag.Node.State.CLEAN;
import static net.aequologica.neo.dagr.model.Dag.Node.State.CLEANING_FAILED;
import static net.aequologica.neo.dagr.model.Dag.Node.State.DIRTY;

import java.util.List;

import javax.inject.Inject;
import javax.inject.Singleton;

import org.atmosphere.cpr.AtmosphereConfig;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.aequologica.neo.dagr.Dags;
import net.aequologica.neo.dagr.model.Dag.Node;
import net.aequologica.neo.dagr.model.Dag.Node.State;
import rx.Observable;
import rx.subjects.PublishSubject;
import rx.subjects.SerializedSubject;
import rx.subjects.Subject;

@Singleton
public class NodeBusImpl implements Bus<Node>, org.atmosphere.inject.Injectable{

    private final static Logger log = LoggerFactory.getLogger(NodeBusImpl.class);

    // cf. http://nerds.weddingpartyapp.com/tech/2014/12/24/implementing-an-event-nodeBus-with-rxjava-rxbus/
    private final Subject<BusEvent<Node>, BusEvent<Node>> nodeBus;
    
    @Inject Dags dags;

    private NodeBusImpl() {
        this.nodeBus = new SerializedSubject<>(PublishSubject.create());
        log.debug("created bus {} {}", this, this.nodeBus);
    }

    @Override
    public void send(final BusEvent.Type type, final String scm, final String branch ) {
        try {
            // the state machine
            final Node.State newState;
            // no switch on enum value in java ... sob
            if (       type.equals( PUSH          )) { newState = DIRTY           ; 
            } else if (type.equals( DEPENDENCY    )) { newState = DIRTY           ;
            } else if (type.equals( BUILD_STARTED )) { newState = BEING_CLEANED   ; 
            } else if (type.equals( BUILD_ERROR   )) { newState = CLEANING_FAILED ; 
            } else if (type.equals( BUILD_OK      )) { newState = CLEAN           ; 
            } else if (type.equals( TRAVIS_ERROR  )) { newState = CLEANING_FAILED ; 
            } else if (type.equals( TRAVIS_OK     )) { newState = CLEAN           ; 
            } else {
                newState = null; 
            }
    
            // from repo_fullname + ref to Node list
            List<Node> nodes = dags.getNodesByScmAndBranch(scm, branch);
    
            log.debug("[bus] found {} nodes with scm='{}' and branch='{}'", nodes.size(), scm, branch);
            // System.out.println("[bus] found " + nodes.size() + " nodes with repo_fullname='"+repo_fullname+"' and ref='"+ref+"'");
            
            for (final Node node : nodes) {
                State previousState = node.getState();
                if (newState != null && previousState != null && !previousState.equals(newState)) {
                    node.setState(newState);
                }
            
                log.debug("[bus] notiying observers that state of node {} ({}) changed from {} to {}", node.getName(), node.getId(), previousState, node.getState());
                // System.out.println("[bus] notiying observers that state of node "+node.getName()+" ("+node.getId()+") changed from "+previousState+" to "+node.getState());
                this.nodeBus.onNext(new BusEvent<Node>() {
                    @Override public Type getType() { return type; }
                    @Override public Node get()     { return node; }
                });
            }
        } catch (Exception e) {
            log.error("[bus] exception {} logged and re-thrown", e.getMessage());
            System.err.println("[bus] exception " + e.getMessage() + "logged and re-thrown");
            throw e;
        } finally {
        }
    }

    @Override
    public Observable<BusEvent<Node>> toObservable() {
      return this.nodeBus;
    }

    @Override
    public Object injectable(AtmosphereConfig arg0) {
        // TODO Auto-generated method stub
        return null;
    }

    @Override
    public boolean supportedType(java.lang.reflect.Type arg0) {
        // TODO Auto-generated method stub
        return false;
    }
    
}
