package net.aequologica.neo.dagr.jaxrs;

import static javax.ws.rs.core.Response.Status.UNAUTHORIZED;
import static net.aequologica.neo.dagr.bus.BusEvent.Type.STATE_CHANGE_HACK;

import java.io.IOException;
import java.security.Principal;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

import javax.inject.Inject;
import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.Consumes;
import javax.ws.rs.DELETE;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.WebApplicationException;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.Response.Status;

import org.glassfish.jersey.server.mvc.Viewable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.base.Function;
import com.google.common.base.Splitter;
import com.google.common.collect.Iterators;
import com.google.common.collect.Lists;

import net.aequologica.neo.dagr.Dags;
import net.aequologica.neo.dagr.bus.Bus;
import net.aequologica.neo.dagr.bus.BusEvent;
import net.aequologica.neo.dagr.jgrapht.DagJGraphT;
import net.aequologica.neo.dagr.model.Dag;
import net.aequologica.neo.dagr.model.Dag.Node;
import net.aequologica.neo.dagr.model.Dag.NodeValue;

@javax.ws.rs.Path("/v1")
public class ResourceDags<T> {

    private final static Logger log = LoggerFactory.getLogger(ResourceDags.class);

    @Inject private Dags dags;
    @Inject private Bus<Node> bus;

    public ResourceDags() throws IOException {
        super();
    }

    // ping service
    @GET
    @Path("")
    @Produces(MediaType.TEXT_PLAIN)
    public String ping() {
        return "pong";
    }

    // reload DAGS
    @POST
    @Path("/reload")
    @Produces(MediaType.APPLICATION_JSON)
    public Response reload() {
        return Response.status(Response.Status.OK).entity(this.dags.load()).build();
    }

    // go to pick-dag JSP 
    @GET
    @Path("/pick-dag")
    @Produces(MediaType.TEXT_HTML)
    public Viewable pickDate(@Context HttpServletRequest request) throws Exception {
        return new Viewable("/WEB-INF/dagr/pick-dag");
    }

    // get DAGs of logged user
    @GET
    @Path("/dags")
    @Produces(MediaType.APPLICATION_JSON)
    public List<DagInfo> getUserDagInfos(@Context HttpServletRequest request) {
        String username = getUsername(request);
        return getAllDagInfos(username);
    }

    // get DAGs of param user
    @GET
    @Path("/dags/users/{user : .+}")
    @Produces(MediaType.APPLICATION_JSON)
    public List<DagInfo> getAllDagInfos(@PathParam("user") String username) {
        Collection<Dag> dagsCollection  = this.dags.getDAGs();
        List<DagInfo>   ret             = buildDagInfoList(dagsCollection, username);
        return ret;
    }

    // list of dags of logged user as html 
    @GET
    @Path("/dags")
    @Produces(MediaType.TEXT_HTML)
    public Viewable viewDags(@Context HttpServletRequest request) throws IOException {
        Collection<Dag> dagsCollection  = this.dags.getDAGs();
        String          username        = getUsername(request);
        List<DagInfo>   ret;
        if (username == null) {
            ret = Collections.<DagInfo>emptyList();
        } else {
            List<DagInfo> tmp = buildDagInfoList(dagsCollection, username);
            ret = new ArrayList<>();
            for (DagInfo dagInfo : tmp) {
                if (dagInfo.getSubscribed()) {
                    ret.add(dagInfo);
                }
            }
        }
        return new Viewable("/WEB-INF/dagr/dags", ret);
    }

    // one dag as json 
    @GET
    @Path("/dags/{dag : .+}")
    @Produces(MediaType.APPLICATION_JSON)
    public Dag getDAG(@PathParam("dag") String dagkey) throws Exception {
        Dag dag = this.dags.getDAG(dagkey);
        if (dag == null) {
            throw new WebApplicationException("dag ["+dagkey+"] not found", Status.NOT_FOUND);
        }
        return dag;
    }

    // one dag as html
    @GET
    @Path("/dags/{dag : .+}")
    @Produces(MediaType.TEXT_HTML)
    public Viewable getDAGasHTML(@PathParam("dag") String dagkey, @Context HttpServletRequest request) throws Exception {
        Dag dag = getDAG(dagkey);
        List<Dag> list = new ArrayList<>();
        list.add(dag);
        String          username        = getUsername(request);
        List<DagInfo> buildDagInfoList = buildDagInfoList(list, username);
        return new Viewable("/WEB-INF/dagr/dags", buildDagInfoList);
    }

    // dag topological 1
    @GET
    @Path("/dags/{dag : .+}/topological")
    @Produces(MediaType.APPLICATION_JSON)
    public List<String> getTopological(@PathParam("dag") String dagkey) throws Exception {
        DagJGraphT dag = this.dags.getDagJGraphT(dagkey);
        if (dag == null) {
            throw new WebApplicationException("dag ["+dagkey+"] not found", Status.NOT_FOUND);
        }
        Function<Node, String> node2nodeId = new Function<Node, String>() {
            public String apply(Node node) {
                return node.getId();
            }
        };
        return Lists.<String>newArrayList(Iterators.transform(dag.getTopologicalOrderIterator(), node2nodeId));
    }

    
    // dag topological 2
    @GET
    @Path("/dags/{dag : .+}/topologicartifacts")
    @Produces(MediaType.APPLICATION_JSON)
    public List<String> getTopologicartifacts(@PathParam("dag") String dagkey) throws Exception {
        DagJGraphT dag = this.dags.getDagJGraphT(dagkey);
        if (dag == null) {
            throw new WebApplicationException("dag ["+dagkey+"] not found", Status.NOT_FOUND);
        }
        Function<Node, String> node2artifactId = new Function<Node, String>() {
            public String apply(Node node) {
                NodeValue value = node.getValue();
                if (value == null) { 
                    return node.getId();
                }
                String gubrid = value.getGubrid();
                if (gubrid == null || gubrid.length() == 0) {
                    return node.getId();
                }
                Iterable<String> split = Splitter.on(':').split(gubrid);
                Iterator<String> iterator = split.iterator();
                if (!iterator.hasNext()) {
                    return node.getId();
                }                
                @SuppressWarnings("unused")
                String groupId = iterator.next();
                if (!iterator.hasNext()) {
                    return node.getId();
                }
                String artifactId = iterator.next();
                if (artifactId == null || artifactId.toString().length() == 0) {
                    return node.getId();
                }
                return artifactId;
            }
        };
        return Lists.<String>newArrayList(Iterators.transform(dag.getTopologicalOrderIterator(), node2artifactId));
    }

    // dag nodes as json 
    @GET
    @Path("/dags/{dag : .+}/nodes")
    @Produces(MediaType.APPLICATION_JSON)
    public List<Node> getDAGNodes(@PathParam("dag") String dagkey) throws Exception {
        Dag dag = getDAG(dagkey);
        return dag.getNodes();
    }

    // one dag node as json
    @GET
    @Path("/dags/{dag : .+}/nodes/{nodename : .+}")
    @Produces(MediaType.APPLICATION_JSON)
    public Node getDAGNode(@PathParam("dag") String dagkey, @PathParam("nodename") String nodeName) throws Exception {
        Dag dag = getDAG(dagkey);
        List<Node> nodes = dag.getNodesNamedAs(nodeName);
        if (nodes.size() == 0) {
            throw new WebApplicationException("dag ["+dagkey+"] node + ["+nodeName+"] + not found", Status.NOT_FOUND);
        } else if (nodes.size() > 1) {
            throw new WebApplicationException("application error. multiple nodes with the same name is not supported", Status.NOT_IMPLEMENTED);
        }
        return nodes.get(0);
    }

    // one dag node state as String 
    @GET
    @Path("/dags/{dag : .+}/nodes/{nodename : .+}/state")
    @Produces(MediaType.APPLICATION_JSON)
    public Node.State getDAGNodeState(@PathParam("dag") String dagkey, @PathParam("nodename") String nodeName) throws Exception {
        Node node = getDAGNode(dagkey, nodeName);
        return node.getState();
    }

    // helper to send an event on the bus
    private void sendBusEvent(BusEvent.Type eventType, Node node) {
        log.debug("[dagr-api] sending event {}, owner_name/name='{}', branch='{}' to the bus", eventType, node.getName(), node.getValue().getBranch());
        // System.out.println("[dagr-api] sending event "+eventType+", owner_name/name='"+node.getName()+"', branch='"+node.getValue().getBranch()+"' to the bus");
        
        // balance dans le bus
        bus.send(eventType, node.getName(), node.getValue() != null ? node.getValue().getBranch() : null);
    }

    // set one dag node state 
    @POST
    @Path("/dags/{dag : .+}/nodes/{nodename : .+}/state/{state : .+}")
    @Produces(MediaType.APPLICATION_JSON)
    public Response setDAGNodeState(@PathParam("dag") String dagkey, @PathParam("nodename") String nodeName, @PathParam("state") Dag.Node.State state) throws Exception {
        try {
            Node node = getDAGNode(dagkey, nodeName);

            sendBusEvent(STATE_CHANGE_HACK, node);

            if (node.getState().equals(state)) {
                return Response.status(Response.Status.NOT_MODIFIED).build();
            }
            node.setState(state);
            
            return Response.status(Response.Status.OK).build();
        } catch (Exception e) {
            log.error("[dagr-api] exception {} logged and re-thrown", e.getMessage());
            System.err.println("[dagr-api] exception " + e.getMessage() + "logged and re-thrown");
            throw e;
        } finally {
        }
    }

    // send an event on the bus
    @PATCH
    @Path("/dags/{dag : .+}/nodes/{nodename : .+}/event/{event : .+}")
    @Consumes("text/plain")
    @Produces(MediaType.APPLICATION_JSON)
    public Response sendBusEvent(String entity, @PathParam("dag") String dagkey, @PathParam("nodename") String nodeName, @PathParam("event") BusEvent.Type eventType, @Context HttpServletRequest request) throws Exception {
        log.debug("dags jersey resource received GET {} with content =''", request.getRequestURI(), entity);
        try {
            Node node = getDAGNode(dagkey, nodeName);
            if (eventType.equals(BusEvent.Type.BUILD_STARTED)) {
                node.setCleaner(entity);
            }
            sendBusEvent(eventType, node);
            
            return Response.status(Response.Status.OK).build();
        } catch (Exception e) {
            log.error("[dagr-api] exception {} logged and re-thrown", e.getMessage());
            System.err.println("[dagr-api] exception " + e.getMessage() + "logged and re-thrown");
            throw e;
        } finally {
        }
    }

    // get my subscriptions
    @GET
    @Path("/dags/subscription")
    @Produces(MediaType.APPLICATION_JSON)
    public List<DagInfo> getUserDAGs(@Context HttpServletRequest request) throws Exception {
        String username = getUsername(request);
        if (username == null) {
            throw new WebApplicationException(UNAUTHORIZED);
        }
        Collection<Dag> dagsCollection = this.dags.getUserDAGs(username);
        List<DagInfo> ret = buildDagInfoList(dagsCollection, username);
        return ret;
    }

    // subscribe
    @POST
    @Path("/dags/subscription/{dag : .+}")
    public Response subscribe(@PathParam("dag") String dagkey, @Context HttpServletRequest request) throws IOException {
        String username = getUsername(request);
        if (username == null) {
            return Response.status(Response.Status.UNAUTHORIZED).build();
        }
        if (this.dags.subscribe(dagkey, username)) {
            return Response.status(Response.Status.OK).build();
        } else {
            return Response.status(Response.Status.NOT_MODIFIED).build();
        }
    }

    // unsubscribe
    @DELETE
    @Path("/dags/subscription/{dag : .+}")
    public Response unsubscribe(@PathParam("dag") String dagkey, @Context HttpServletRequest request) throws IOException {
        String username = getUsername(request);
        if (username == null) {
            return Response.status(Response.Status.UNAUTHORIZED).build();
        }
        if (this.dags.unsubscribe(dagkey, username)) {
            return Response.status(Response.Status.OK).build();
        } else {
            return Response.status(Response.Status.NOT_MODIFIED).build();
        }
    }

    static public class DagInfo {
        private String  name;
        private String  key;
        private String  url;
        private Boolean subscribed;
        private String  username;

        public String getName() {
            return name;
        }
        public void setName(String name) {
            this.name = name;
        }
        public String getKey() {
            return key;
        }
        public void setKey(String key) {
            this.key = key;
        }
        public String getUrl() {
            return url;
        }
        public void setUrl(String url) {
            this.url = url;
        }
        public Boolean getSubscribed() {
            return subscribed;
        }
        public void setSubscribed(Boolean subscribed) {
            this.subscribed = subscribed;
        }
        public String getUsername() {
            return username;
        }
        public void setUsername(String username) {
            this.username = username;
        }
    }

    private List<DagInfo> buildDagInfoList(Collection<Dag> dagsCollection, String username) {
        List<DagInfo> ret = new ArrayList<>(dagsCollection.size());
        for (Dag dag : dagsCollection) {
            DagInfo dagInfo     = new DagInfo();
            dagInfo.name        = dag.getName();
            dagInfo.key         = dag.getKey();
            dagInfo.url         = dag.getSource();
            dagInfo.username    = username;
            dagInfo.subscribed  = username == null ? false : this.dags.isUserSubscribed(dag.getKey(), username);
            ret.add(dagInfo);
        }
        return ret;
    }

    private static String getUsername(HttpServletRequest request) {
        Principal userPrincipal = request.getUserPrincipal();
        if (userPrincipal == null) {
            return null;
        }
        String username = userPrincipal.getName();
        if (username == null || username.isEmpty()) {
            return null;
        }
        return username;
    }

}
