package net.aequologica.neo.dagr.garance;

import java.util.Date;
import java.util.HashMap;
import java.util.Map;

import javax.inject.Inject;
import javax.inject.Singleton;
import javax.servlet.ServletContextEvent;
import javax.servlet.ServletContextListener;
import javax.servlet.annotation.WebListener;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.aequologica.neo.dagr.bus.Bus;
import net.aequologica.neo.dagr.bus.BusEvent;
import net.aequologica.neo.dagr.model.Dag.Node;
import net.aequologica.neo.garance.SeriesSet;
import rx.Subscription;

@Singleton
@WebListener
public class Listener implements ServletContextListener {
    
    private final static Logger log = LoggerFactory.getLogger(Listener.class);
    
    @Inject private SeriesSet seriesMap;
    @Inject private Bus<Node> bus;

    private Map<String, Date> startMap = new HashMap<>();

    private Subscription subscription;

    public void contextInitialized(ServletContextEvent arg0)  { 

        this.subscription = bus.toObservable()
                .filter(    event -> event.getType().equals(BusEvent.Type.BUILD_STARTED) || 
                                     event.getType().equals(BusEvent.Type.BUILD_OK)      || 
                                     event.getType().equals(BusEvent.Type.BUILD_ERROR)   )
                .map(       event -> event.get())
                .subscribe( node  -> sendNotification(node));
        
        log.debug("[garance] subbscribed !");
    }

    public void contextDestroyed(ServletContextEvent arg0)  { 
        if (this.subscription != null) {
            this.subscription.unsubscribe();
            this.subscription = null;
            log.debug("[garance] unsubscribed !");
        }
    }
	
    void sendNotification(final Node node) {
        if (node.getState().equals(Node.State.BEING_CLEANED)) {
            startMap.put(node.getValue().getGubrid(), new Date());
        } else if (node.getState().equals(Node.State.CLEAN)) {
            Date start = startMap.get(node.getValue().getGubrid());
            if (start != null) {
                this.seriesMap.put(node.getValue().getGubrid(), (double)(new Date().getTime() - start.getTime()));
                this.seriesMap.save();
            }
            startMap.remove(node.getValue().getGubrid());
        } else if (node.getState().equals(Node.State.CLEANING_FAILED)) {
           // do not store, likely the duration is shorter than normal
        }  
    }
    
}
