package net.aequologica.neo.dagr.mobile;

import static net.aequologica.neo.geppaequo.config.ConfigRegistry.getConfig;
import static net.aequologica.neo.geppaequo.config.ConfigRegistry.scanConfigs;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.nio.charset.StandardCharsets;

import javax.servlet.ServletContextEvent;
import javax.servlet.ServletContextListener;
import javax.servlet.annotation.WebListener;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.io.CharStreams;

import net.aequologica.neo.dagr.bus.AbhorrentNodeBus;
import net.aequologica.neo.dagr.bus.Bus;
import net.aequologica.neo.dagr.bus.Event;
import net.aequologica.neo.dagr.config.DagrConfig;
import net.aequologica.neo.dagr.model.Dag.Node;
import net.aequologica.neo.geppaequo.crypto.CodecPBEWithMD5AndDES;
import rx.Subscription;

@WebListener
public class Listener implements ServletContextListener {
    
    static {
        scanConfigs();
    }
    
    private final static Logger log = LoggerFactory.getLogger("net.aequologica.neo.dagr.mobile");

    private final ObjectMapper  mapper;
    private final String        googleCloudMessagingAPIKey;

    private Subscription subscription;

    public Listener() throws Exception {
        this.mapper                     = new ObjectMapper();
        this.googleCloudMessagingAPIKey = new CodecPBEWithMD5AndDES().decrypt(getConfig(DagrConfig.class).getEncryptedGoogleCloudMessagingAPIKey().toCharArray());
    }

    public void contextInitialized(ServletContextEvent arg0)  { 
        Bus<Node> bus = AbhorrentNodeBus.getInstance();
        
        subscription = bus.toObservable()
                .filter(    event -> event.getType().equals(Event.Type.BUILD_OK) || event.getType().equals(Event.Type.BUILD_ERROR))
                .map(       event -> event.get())
                .subscribe( node  -> sendNotification(node));
        
        log.debug("[gcm] subbscribed !");
    }
    public void contextDestroyed(ServletContextEvent arg0)  { 
        if (subscription != null) {
            subscription.unsubscribe();
            log.debug("[gcm] unsubscribed !");
        }
    }
	
    void sendNotification(Node node) {
        
        // balance le groove sur les portables
        StringBuilder sb = new StringBuilder();
        sb.append(node.getState().toString());
        sb.append(" ");
        sb.append(node.getName());
        sb.append(" @ ");
        sb.append("travis");
        
        String message = sb.toString();
        log.debug("[gcm] sending notification {} for node {} to google cloud messaging", message, node.getName());
        
        try {
            GCMMessage msg = new GCMMessage(new GCMMessage.Data(message));
            
            String gcmMessageAsString = mapper.writeValueAsString(msg);
            
            // Create connection to send GCM Message request.
            URL url = new URL("https://android.googleapis.com/gcm/send");
            HttpURLConnection conn = (HttpURLConnection) url.openConnection();
            conn.setRequestProperty("Authorization", "key=" + googleCloudMessagingAPIKey);
            conn.setRequestProperty("Content-Type", "application/json; charset=utf-8");
            conn.setRequestMethod("POST");
            conn.setDoOutput(true);

            // Send GCM message content.
            OutputStream outputStream = conn.getOutputStream();
            outputStream.write(gcmMessageAsString.getBytes(StandardCharsets.UTF_8));

            // Read GCM response.
            InputStream inputStream = conn.getInputStream();
            String responseText = CharStreams.toString(new InputStreamReader(inputStream));
            
            log.debug("[gcm] Check your device/emulator for notification or logcat for confirmation of the receipt of the GCM message. GCM Response text is {}", responseText);
        
        } catch (IOException e) {
            e.printStackTrace();
        }
    }
    
    /* e.g.
     * { "data" : {
     *     "message" : "blah"
     *   },
     *   "to" : "/topics/global"
     * } 
     */
    // GCM = Google Cloud Messaging
    private static class GCMMessage {
        
        @JsonProperty
        private final String to = "/topics/global";
        
        @JsonProperty
        private Data data;
        
        @SuppressWarnings("unused")
        public GCMMessage() {
        }

        public GCMMessage(final Data data) {
            this.data = data;
        }
        
        private static class Data {
            @JsonProperty
            private String message;

            @SuppressWarnings("unused")
            public Data() {
            }

            public Data(final String message) {
                this.message = message;
            }

        }
    }
    
}
