package net.aequologica.neo.dagr.bus;

import static net.aequologica.neo.dagr.bus.Event.Type.*;
import static net.aequologica.neo.dagr.model.Dag.Node.State.*;

import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.aequologica.neo.dagr.Dags;
import net.aequologica.neo.dagr.model.Dag.Node;
import net.aequologica.neo.dagr.model.Dag.Node.State;
import rx.Observable;
import rx.subjects.PublishSubject;
import rx.subjects.SerializedSubject;
import rx.subjects.Subject;

public class AbhorrentNodeBus implements Bus<Node> {

    private final static Logger log = LoggerFactory.getLogger("net.aequologica.neo.dagr.bus");

    /* |               |
       | the good part |
       v               v */
    
    // cf. http://nerds.weddingpartyapp.com/tech/2014/12/24/implementing-an-event-nodeBus-with-rxjava-rxbus/
    private final Subject<Event<Node>, Event<Node>> nodeBus;
    private final Dags                              dags;

    @Override
    public void send(final Event.Type type, final String repo_fullname, final String ref ) {
        try {
            // the state machine
            final Node.State newState;
            // no switch on enum value in java ... sob
            if (       type.equals( BUILD_STARTED )) { newState = BEING_CLEANED   ; 
            } else if (type.equals( PUSH          )) { newState = DIRTY           ; 
            } else if (type.equals( DEPENDENCY    )) { newState = DIRTY           ;
            } else if (type.equals( BUILD_ERROR   )) { newState = CLEANING_FAILED ; 
            } else if (type.equals( BUILD_OK      )) { newState = CLEAN           ; 
            } else {
                newState = null; 
            }
    
            // from repo_fullname + ref to Node list
            List<Node> nodes = dags.getNodesByRepoFullnameAndRef(repo_fullname, ref);
    
            log.debug("[bus] found {} nodes with repo_fullname='{}' and ref='{}'", nodes.size(), repo_fullname, ref);
            System.out.println("[bus] found " + nodes.size() + " nodes with repo_fullname='"+repo_fullname+"' and ref='"+ref+"'");
            
            for (final Node node : nodes) {
                State previousState = node.getState();
                if (newState != null && previousState != null && !previousState.equals(newState)) {
                    node.setState(newState);
                }
            
                log.debug("[bus] notiying observers that state of node {} ({}) changed from {} to {}", node.getName(), node.getId(), previousState, node.getState());
                System.out.println("[bus] notiying observers that state of node "+node.getName()+" ("+node.getId()+") changed from "+previousState+" to "+node.getState());
                this.nodeBus.onNext(new Event<Node>() {
                    @Override public Type getType() { return type; }
                    @Override public Node get()     { return node; }
                });
            }
        } catch (Exception e) {
            log.error("[bus] exception {} logged and re-thrown", e.getMessage());
            System.err.println("[bus] exception " + e.getMessage() + "logged and re-thrown");
            throw e;
        } finally {
        }
    }

    @Override
    public Observable<Event<Node>> toObservable() {
      return this.nodeBus;
    }
    
    /* |              |
       | the bad part |
       v              v */
    
    // 'hand-made' singleton
    // cf. https://en.wikipedia.org/wiki/Initialization-on-demand_holder_idiom for the singleton pattern used
    // maybe someday, we'll understand enough CDI on HCP to remove that shit

    private AbhorrentNodeBus() {
        this.nodeBus = new SerializedSubject<>(PublishSubject.create());
        this.dags    = Dags.getInstance();
    }

    private static class LazyHolder {
        private static final AbhorrentNodeBus INSTANCE = new AbhorrentNodeBus();
    }

    public static AbhorrentNodeBus getInstance() {
        return LazyHolder.INSTANCE;
    }

}
