package net.aequologica.neo.dagr.jaxrs;

import static net.aequologica.neo.geppaequo.config.ConfigRegistry.getConfig;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.nio.charset.StandardCharsets;

import javax.json.JsonObject;
import javax.ws.rs.Consumes;
import javax.ws.rs.FormParam;
import javax.ws.rs.POST;
import javax.ws.rs.core.Response;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.datatype.jsr353.JSR353Module;
import com.google.common.io.CharStreams;

import net.aequologica.neo.dagr.config.DagrConfig;
import net.aequologica.neo.dagr.crypto.Codec;
import net.aequologica.neo.dagr.crypto.CodecPBEWithMD5AndDES;

@javax.ws.rs.Path("/travis")
public class TravisResource {

    @SuppressWarnings("unused")
    private static final Logger log = LoggerFactory.getLogger(TravisResource.class);

    private final ObjectMapper mapper = new ObjectMapper();
    private final Codec codec;

    public TravisResource() throws IOException {
        mapper.registerModule(new JSR353Module());
        codec = new CodecPBEWithMD5AndDES();
    }

    /*
     * cf. http://docs.travis-ci.com/user/notifications/#Webhook-notification
     */
    @POST
    @javax.ws.rs.Path("notifications")
    @Consumes("application/x-www-form-urlencoded")
    public Response receiveNotification(@FormParam("payload") String payloadInput) throws Exception {

        JsonObject payload = mapper.readValue(payloadInput, JsonObject.class);
        
        StringBuilder sb = new StringBuilder();
        sb.append(payload.getInt("status") == 0 ? "SUCCESS" : "FAILURE");
        sb.append(" ");
        sb.append(payload.getJsonObject("repository").getString("name"));
        sb.append(" @ ");
        sb.append(payload.getString("build_url"));
        
        sendNotification(sb.toString());
        
        return Response.noContent().build();
    }
    
    
    void sendNotification(String message) throws Exception {
        try {
            GCMMessage msg = new GCMMessage(new GCMMessage.Data(message));
            
            String msgAsString = mapper.writeValueAsString(msg);
            System.out.println("======= message:");
            System.out.println(msgAsString);
            
            GCMMessage readValue = mapper.readValue(msgAsString, GCMMessage.class);
            System.out.println(readValue);
            
            DagrConfig config = getConfig(DagrConfig.class);
            
            String encryptedGoogleCloudMessagingAPIKey = config.getEncryptedGoogleCloudMessagingAPIKey();
            String googleCloudMessagingAPIKey = codec.decrypt(encryptedGoogleCloudMessagingAPIKey.toCharArray());
            
            // Create connection to send GCM Message request.
            URL url = new URL("https://android.googleapis.com/gcm/send");
            HttpURLConnection conn = (HttpURLConnection) url.openConnection();
            conn.setRequestProperty("Authorization", "key=" + googleCloudMessagingAPIKey);
            conn.setRequestProperty("Content-Type", "application/json; charset=utf-8");
            conn.setRequestMethod("POST");
            conn.setDoOutput(true);

            // Send GCM message content.
            OutputStream outputStream = conn.getOutputStream();
            outputStream.write(msgAsString.getBytes(StandardCharsets.UTF_8));

            // Read GCM response.
            InputStream inputStream = conn.getInputStream();
            String resp = CharStreams.toString(new InputStreamReader(inputStream));
            System.out.println("======= response:");
            System.out.println(resp);
            System.out.println("Check your device/emulator for notification or logcat for confirmation of the receipt of the GCM message.");
        } catch (IOException e) {
            e.printStackTrace();
        }
    }
    
    /* e.g.
     * { "data" : {
     *     "message" : "blah"
     *   },
     *   "to" : "/topics/global"
     * } 
     */
    // GCM = Google Cloud Messaging
    private static class GCMMessage {
        
        @JsonProperty
        private final String to = "/topics/global";
        
        @JsonProperty
        private Data data;
        
        @SuppressWarnings("unused")
        public GCMMessage() {
        }

        public GCMMessage(final Data data) {
            this.data = data;
        }
        
        private static class Data {
            @JsonProperty
            private String message;

            @SuppressWarnings("unused")
            public Data() {
            }

            public Data(final String message) {
                this.message = message;
            }

        }
    }
    
}
