package net.aequologica.neo.dagr.jaxrs;

import static javax.ws.rs.core.Response.Status.UNAUTHORIZED;

import java.io.IOException;
import java.security.Principal;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

import javax.servlet.ServletContext;
import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.DELETE;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.WebApplicationException;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.Response.Status;

import org.glassfish.jersey.server.mvc.Viewable;

import com.google.common.base.Function;
import com.google.common.base.Splitter;
import com.google.common.collect.Iterators;
import com.google.common.collect.Lists;

import net.aequologica.neo.dagr.Dags;
import net.aequologica.neo.dagr.jgrapht.DagJGraphT;
import net.aequologica.neo.dagr.model.Dag;
import net.aequologica.neo.dagr.model.Dag.Node;
import net.aequologica.neo.dagr.model.Dag.NodeValue;

@javax.ws.rs.Path("/v1")
public class Resource<T> {

    @javax.ws.rs.core.Context
    ServletContext context;

    private final Dags dags = Dags.getInstance();

    public Resource() throws IOException {
        super();
    }

    // ping service
    @GET
    @Path("")
    @Produces(MediaType.TEXT_PLAIN)
    public String ping() {
        return "pong";
    }

    // reload DAGS
    @POST
    @Path("/reload")
    @Produces(MediaType.APPLICATION_JSON)
    public Response reload() {
        return Response.status(Response.Status.OK).entity(this.dags.load()).build();
    }

    // unique dag as html (user agnostic)
    @GET
    @Path("/pick-dag")
    @Produces(MediaType.TEXT_HTML)
    public Viewable pickDate(@Context HttpServletRequest request) throws Exception {
        return new Viewable("/WEB-INF/dagr/pick-dag");
    }

    @GET
    @Path("/dags")
    @Produces(MediaType.APPLICATION_JSON)
    public List<DagInfo> getUserDagInfos(@Context HttpServletRequest request) {
        Collection<Dag> dagsCollection  = this.dags.getDAGs();
        String          username        = getUsername(request);
        List<DagInfo>   ret             = buildDagInfoList(dagsCollection, username);
        return ret;
    }

    @GET
    @Path("/dags/users/{user : .+}")
    @Produces(MediaType.APPLICATION_JSON)
    public List<DagInfo> getAllDagInfos(@PathParam("user") String username) {
        Collection<Dag> dagsCollection  = this.dags.getDAGs();
        List<DagInfo>   ret             = buildDagInfoList(dagsCollection, username);
        return ret;
    }

    // list of dags as html (user must be logged + limit to subscribed dags
    @GET
    @Path("/dags")
    @Produces(MediaType.TEXT_HTML)
    public Viewable viewDags(@Context HttpServletRequest request) throws IOException {
        Collection<Dag> dagsCollection  = this.dags.getDAGs();
        String          username        = getUsername(request);
        List<DagInfo>   ret;
        if (username == null) {
            ret = Collections.<DagInfo>emptyList();
        } else {
            List<DagInfo> tmp = buildDagInfoList(dagsCollection, username);
            ret = new ArrayList<>();
            for (DagInfo dagInfo : tmp) {
                if (dagInfo.getSubscribed()) {
                    ret.add(dagInfo);
                }
            }
        }
        return new Viewable("/WEB-INF/dagr/dags", ret);
    }

    // unique dag as json (user agnostic)
    @GET
    @Path("/dags/{dag : .+}")
    @Produces(MediaType.APPLICATION_JSON)
    public Dag getDAG(@PathParam("dag") String dagkey) throws Exception {
        Dag dag = this.dags.getDAG(dagkey);
        if (dag == null) {
            throw new WebApplicationException("dag ["+dagkey+"] not found", Status.NOT_FOUND);
        }
        return dag;
    }

    // unique dag as html (user agnostic)
    @GET
    @Path("/dags/{dag : .+}")
    @Produces(MediaType.TEXT_HTML)
    public Viewable getDAGasHTML(@PathParam("dag") String dagkey, @Context HttpServletRequest request) throws Exception {
        Dag dag = getDAG(dagkey);
        List<Dag> list = new ArrayList<>();
        list.add(dag);
        String          username        = getUsername(request);
        List<DagInfo> buildDagInfoList = buildDagInfoList(list, username);
        return new Viewable("/WEB-INF/dagr/dags", buildDagInfoList);
    }

    // dag topological 1
    @GET
    @Path("/dags/{dag : .+}/topological")
    @Produces(MediaType.APPLICATION_JSON)
    public List<String> getTopological(@PathParam("dag") String dagkey) throws Exception {
        DagJGraphT dag = this.dags.getDAG2(dagkey);
        if (dag == null) {
            throw new WebApplicationException("dag ["+dagkey+"] not found", Status.NOT_FOUND);
        }
        Function<Node, String> node2nodeId = new Function<Node, String>() {
            public String apply(Node node) {
                return node.getId();
            }
        };
        return Lists.<String>newArrayList(Iterators.transform(dag.getTopologicalOrderIterator(), node2nodeId));
    }

    
    // dag topological 2
    @GET
    @Path("/dags/{dag : .+}/topologicartifacts")
    @Produces(MediaType.APPLICATION_JSON)
    public List<String> getTopologicartifacts(@PathParam("dag") String dagkey) throws Exception {
        DagJGraphT dag = this.dags.getDAG2(dagkey);
        if (dag == null) {
            throw new WebApplicationException("dag ["+dagkey+"] not found", Status.NOT_FOUND);
        }
        Function<Node, String> node2artifactId = new Function<Node, String>() {
            public String apply(Node node) {
                NodeValue value = node.getValue();
                if (value == null) { 
                    return node.getId();
                }
                String gubrid = value.getGubrid();
                if (gubrid == null || gubrid.length() == 0) {
                    return node.getId();
                }
                Iterable<String> split = Splitter.on(':').split(gubrid);
                Iterator<String> iterator = split.iterator();
                if (!iterator.hasNext()) {
                    return node.getId();
                }                
                @SuppressWarnings("unused")
                String groupId = iterator.next();
                if (!iterator.hasNext()) {
                    return node.getId();
                }
                String artifactId = iterator.next();
                if (artifactId == null || artifactId.toString().length() == 0) {
                    return node.getId();
                }
                return artifactId;
            }
        };
        return Lists.<String>newArrayList(Iterators.transform(dag.getTopologicalOrderIterator(), node2artifactId));
    }

    // dag nodes as json (user agnostic)
    @GET
    @Path("/dags/{dag : .+}/nodes")
    @Produces(MediaType.APPLICATION_JSON)
    public List<Node> getDAGNodes(@PathParam("dag") String dagkey) throws Exception {
        Dag dag = getDAG(dagkey);
        return dag.getNodes();
    }

    // dag node as json (user agnostic)
    @GET
    @Path("/dags/{dag : .+}/nodes/{nodename : .+}")
    @Produces(MediaType.APPLICATION_JSON)
    public Node getDAGNode(@PathParam("dag") String dagkey, @PathParam("nodename") String nodeName) throws Exception {
        Dag dag = getDAG(dagkey);
        List<Node> nodes = dag.getNodesNamedAs(nodeName);
        if (nodes.size() == 0) {
            throw new WebApplicationException("dag ["+dagkey+"] node + ["+nodeName+"] + not found", Status.NOT_FOUND);
        } else if (nodes.size() > 1) {
            throw new WebApplicationException("application error. multiple nodes with the same name is not supported", Status.NOT_IMPLEMENTED);
        }
        return nodes.get(0);
    }

    // dag node state as String (user agnostic)
    @GET
    @Path("/dags/{dag : .+}/nodes/{nodename : .+}/state")
    @Produces(MediaType.APPLICATION_JSON)
    public Node.State getDAGNodeState(@PathParam("dag") String dagkey, @PathParam("nodename") String nodeName) throws Exception {
        Node node = getDAGNode(dagkey, nodeName);
        return node.getState();
    }

    @POST
    @Path("/dags/{dag : .+}/nodes/{nodename : .+}/state/{state : .+}")
    @Produces(MediaType.APPLICATION_JSON)
    public Response setDAGNodeState(@PathParam("dag") String dagkey, @PathParam("nodename") String nodeName, @PathParam("state") Dag.Node.State state) throws Exception {
        Node node = getDAGNode(dagkey, nodeName);
        if (node.getState().equals(state)) {
            return Response.status(Response.Status.NOT_MODIFIED).build();
        }
        node.setState(state);
        return Response.status(Response.Status.OK).build();
    }

    // get my subscriptions
    @GET
    @Path("/dags/subscription")
    @Produces(MediaType.APPLICATION_JSON)
    public List<DagInfo> getUserDAGs(@Context HttpServletRequest request) throws Exception {
        String username = getUsername(request);
        if (username == null) {
            throw new WebApplicationException(UNAUTHORIZED);
        }
        Collection<Dag> dagsCollection = this.dags.getUserDAGs(username);
        List<DagInfo> ret = buildDagInfoList(dagsCollection, username);
        return ret;
    }

    // subscribe
    @POST
    @Path("/dags/subscription/{dag : .+}")
    public Response subscribe(@PathParam("dag") String dagkey, @Context HttpServletRequest request) throws IOException {
        String username = getUsername(request);
        if (username == null) {
            return Response.status(Response.Status.UNAUTHORIZED).build();
        }
        if (this.dags.subscribe(dagkey, username)) {
            return Response.status(Response.Status.OK).build();
        } else {
            return Response.status(Response.Status.NOT_MODIFIED).build();
        }
    }

    // unsubscribe
    @DELETE
    @Path("/dags/subscription/{dag : .+}")
    public Response unsubscribe(@PathParam("dag") String dagkey, @Context HttpServletRequest request) throws IOException {
        String username = getUsername(request);
        if (username == null) {
            return Response.status(Response.Status.UNAUTHORIZED).build();
        }
        if (this.dags.unsubscribe(dagkey, username)) {
            return Response.status(Response.Status.OK).build();
        } else {
            return Response.status(Response.Status.NOT_MODIFIED).build();
        }
    }

    static public class DagInfo {
        private String  name;
        private String  key;
        private String  url;
        private Boolean subscribed;
        private String  username;

        public String getName() {
            return name;
        }
        public void setName(String name) {
            this.name = name;
        }
        public String getKey() {
            return key;
        }
        public void setKey(String key) {
            this.key = key;
        }
        public String getUrl() {
            return url;
        }
        public void setUrl(String url) {
            this.url = url;
        }
        public Boolean getSubscribed() {
            return subscribed;
        }
        public void setSubscribed(Boolean subscribed) {
            this.subscribed = subscribed;
        }
        public String getUsername() {
            return username;
        }
        public void setUsername(String username) {
            this.username = username;
        }
    }

    private List<DagInfo> buildDagInfoList(Collection<Dag> dagsCollection, String username) {
        List<DagInfo> ret = new ArrayList<>(dagsCollection.size());
        for (Dag dag : dagsCollection) {
            DagInfo dagInfo     = new DagInfo();
            dagInfo.name        = dag.getName();
            dagInfo.key         = dag.getKey();
            dagInfo.url         = dag.getSource();
            dagInfo.username    = username;
            dagInfo.subscribed  = username == null ? false : this.dags.isUserSubscribed(dag.getKey(), username);
            ret.add(dagInfo);
        }
        return ret;
    }

    private static String getUsername(HttpServletRequest request) {
        Principal userPrincipal = request.getUserPrincipal();
        if (userPrincipal == null) {
            return null;
        }
        String username = userPrincipal.getName();
        if (username == null || username.isEmpty()) {
            return null;
        }
        return username;
    }

}
