package net.aequologica.neo.dagr.jgrapht;

import java.util.ArrayList;
import java.util.BitSet;
import java.util.List;
import java.util.Set;

import org.jgrapht.DirectedGraph;

/**
 * cf. https://en.wikipedia.org/wiki/Transitive_reduction
 * 
 * ported from python example by Michael Clerx, posted as an answer to http://stackoverflow.com/questions/1690953/transitive-reduction-algorithm-pseudocode
 * 
 */
public class TransitiveReduction<V, E> {

    final private DirectedGraph<V, E> graph;
    final private List<V> vertices;
    final private BitSet[] pathMatrix;
    
    public TransitiveReduction(DirectedGraph<V, E> graph) {
        super();
        this.graph = graph;
        this.vertices = new ArrayList<V>(graph.vertexSet());
        BitSet[] original = new BitSet[vertices.size()];
        for (int i = 0; i < original.length; i++) {
            original[i] = new BitSet(vertices.size());
        }

        // initialize matrix with zeros
        // 'By default, all bits in the set initially have the value false.' cf. http://docs.oracle.com/javase/7/docs/api/java/util/BitSet.html
        
        // initialize matrix with edges
        Set<E> edges = graph.edgeSet();
        for (E edge : edges) {
            V v1 = graph.getEdgeSource(edge);
            V v2 = graph.getEdgeTarget(edge);

            int v_1 = vertices.indexOf(v1);
            int v_2 = vertices.indexOf(v2);

            original[v_1].set(v_2); 
        }
        
        this.pathMatrix = original;
        transformToPathMatrix(this.pathMatrix);
    }
    
    // (package visible for unit testing)
    static BitSet[] transformToPathMatrix(BitSet[] matrix) {
        // compute path matrix 
        for (int i = 0; i < matrix.length; i++) {
            for (int j = 0; j < matrix.length; j++) { 
                if (i == j) {
                    continue;
                }
                if (matrix[j].get(i)){
                    for (int k = 0; k < matrix.length; k++) {
                        if (!matrix[j].get(k)) {
                            matrix[j].set(k, matrix[i].get(k));
                        }
                    }
                }
            }
        }
        return matrix;
    }

    // (package visible for unit testing)
    static BitSet[] transitiveReduction(BitSet[] pathMatrix) {
        // transitively reduce
        for (int j = 0; j < pathMatrix.length; j++) { 
            for (int i = 0; i < pathMatrix.length; i++) {
                if (pathMatrix[i].get(j)){
                    for (int k = 0; k < pathMatrix.length; k++) {
                        if (pathMatrix[j].get(k)) {
                            pathMatrix[i].set(k, false);
                        }
                    }
                }
            }
        }
        return pathMatrix;
    }

    public void reduce() {
        
        int n = pathMatrix.length;
        BitSet[] transitivelyReducedMatrix = new BitSet[n];
        System.arraycopy(pathMatrix, 0, transitivelyReducedMatrix, 0, pathMatrix.length);
        transitiveReduction(transitivelyReducedMatrix);
        
        for (int i = 0; i <n; i++) {
            for (int j = 0; j < n; j++) { 
                if (!transitivelyReducedMatrix[i].get(j)) {
                    // System.out.println("removing "+vertices.get(i)+" -> "+vertices.get(j));
                    graph.removeEdge(graph.getEdge(vertices.get(i), vertices.get(j)));
                }
            }
        }
    }

}