package net.aequologica.neo.dagr;

import static com.fasterxml.jackson.annotation.JsonInclude.Include.NON_NULL;
import static com.fasterxml.jackson.databind.SerializationFeature.INDENT_OUTPUT;
import static com.google.common.net.MediaType.JSON_UTF_8;

import java.io.ByteArrayInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Lists;
import com.google.common.collect.Multimap;

import net.aequologica.neo.dagr.jgrapht.DagJGraphT;
import net.aequologica.neo.dagr.model.Dag;
import net.aequologica.neo.dagr.model.DagDocumentSerializer;
import net.aequologica.neo.geppaequo.cmis.ECMHelper.Stream;
import net.aequologica.neo.geppaequo.document.DocumentHelper;

public class Dags {

    private static DagDocumentSerializer serializer = new DagDocumentSerializer();

    private Map<String, Dag>         dagmap    = null;
    private Map<String, DagJGraphT>  graphmap  = null;
    private Multimap<String, String> user2dags = null;
    private Multimap<String, String> dag2users = null;

    private ObjectMapper mapper = new ObjectMapper();
    private Path userDagTuplesPath = Paths.get("/.dagr/userdagtuples.json");
    
    /////////////////////////////
    // hand-made singleton
    static private Dags instance = null;
    static public Dags getInstance() {
        if (instance == null) {
            instance = new Dags();
        }
        return instance;
    }
    private Dags() {
        super();
        mapper.enable(INDENT_OUTPUT);
        mapper.setSerializationInclusion(NON_NULL);
    }
    /////////////////////////////

    public Collection<Dag> getDAGs() {
        return this.dagmap.values();
    }

    public Set<String> getDAGKeys() {
        return this.dagmap.keySet();
    }

    public Dag getDAG(String dagkey) {
        return this.dagmap.get(dagkey);
    }

    public DagJGraphT getDAG2(String dagkey) {
        return this.graphmap.get(dagkey);
    }
    public List<Dag> getUserDAGs(String username) {
        Collection<String> thisUserDags = user2dags.get(username);
        List<Dag> ret = new ArrayList<>(thisUserDags.size());
        for (String dagkey : thisUserDags) {
            Dag dag = this.dagmap.get(dagkey);
            if (dag != null) {
                ret.add(dag);
            }
        }
        return ret;
    }

    public List<Map.Entry<String, String>> load() {

        List<Map.Entry<String, String>> exceptions = Lists.newArrayList();

        this.dagmap    = new HashMap<>();
        this.graphmap  = new HashMap<>();
        this.user2dags = ArrayListMultimap.create();
        this.dag2users = ArrayListMultimap.create();

        Path path = Paths.get("/dags");

        List<java.nio.file.Path> sources;
        try {
            sources = DocumentHelper.list(path);
        } catch (Exception e) {
            exceptions.add(new AbstractMap.SimpleImmutableEntry<String, String>(path.toString(), e.getClass().getSimpleName()+" - " +e.getMessage()));
            return exceptions;
        }

        for (java.nio.file.Path source : sources) {
            try {
                String lowerCase = source.getFileName().toString().toLowerCase();
                if (lowerCase.endsWith(userDagTuplesPath.getFileName().toString())) {
                    continue;
                }
                if (!lowerCase.endsWith(".json")) {
                    continue;
                }
                Dag tmpDag = serializer.read(source);
                if (tmpDag == null) {
                    continue;
                }
                Dag dag = new DagJGraphT(tmpDag).detectAndFlagTransitiveEdges();
                dag.setSource(source.toString());
                String dagkey = source.getFileName().toString();
                dag.setKey(dagkey);

                dagmap.put(dagkey, dag);
                graphmap.put(dagkey, new DagJGraphT(dag));

            } catch (Exception e) {
                exceptions.add(new AbstractMap.SimpleImmutableEntry<String, String>(source.toString(), e.getClass().getSimpleName()+" - " +e.getMessage()));
            }
        }
        
        try {
            readUserDagTuples(userDagTuplesPath);
        } catch (Exception e) {
            e.printStackTrace();
        }
        
        return exceptions;
    }

    public boolean subscribe(String dagkey, String username) throws IOException {
        Dag dag = this.dagmap.get(dagkey);
        if (dag == null) {
            throw new IOException("["+dagkey+"] not found");
        }
        
        Collection<String> existingDags = user2dags.get(username);
        Collection<String> existingUsers = dag2users.get(dagkey);
        if (existingDags.contains(dagkey) && existingUsers.contains(username)) {
            return false;
        }
        
        user2dags.put(username, dagkey);
        dag2users.put(dagkey, username);

        writeUserDagTuples(userDagTuplesPath);

        return true;
    }

    public boolean unsubscribe(String dagkey, String username) throws IOException {
        Dag dag = dagmap.get(dagkey);
        if (dag == null) {
            throw new IOException("["+dagkey+"] not found");
        }
        
        Collection<String> existingDags = user2dags.get(username);
        Collection<String> existingUsers = dag2users.get(dagkey);
        if (!existingDags.contains(dagkey) && !existingUsers.contains(username)) {
            return false;
        }

        user2dags.remove(username, dagkey);
        dag2users.remove(dagkey, username);
        
        writeUserDagTuples(userDagTuplesPath);

        return true;
    }

    public Boolean isUserSubscribed(String dagkey, String username) {
        Collection<String> existingUsers = dag2users.get(dagkey);
        return existingUsers.contains(username);
    }

    private void writeUserDagTuples(Path path) throws JsonProcessingException, IOException {
        List<UserDagTuple> userdags = new ArrayList<>();
        Collection<Entry<String, String>> entries = user2dags.entries();
        for (Entry<String, String> entry : entries) {
            userdags.add(new UserDagTuple(entry.getKey(), entry.getValue()));
        }
        final byte[] bytes = this.mapper.writeValueAsBytes(userdags); // "Encoding used will be UTF-8."
        DocumentHelper.write(path, new Stream() {
            @Override public String      getMimeType()      { return JSON_UTF_8.toString(); }
            @Override public long        getLength()        { return bytes.length; }
            @Override public InputStream getInputStream()   { return new ByteArrayInputStream(bytes); }
        });
    }

    private void readUserDagTuples(Path path) throws JsonProcessingException, IOException {
        final InputStream inputStream = DocumentHelper.getInputStream(path);
        if (inputStream == null) {
            throw new FileNotFoundException(path.toString());
        }
        List<UserDagTuple> userdags = this.mapper.readValue(inputStream, new TypeReference<List<UserDagTuple>>() {});
        
        for (UserDagTuple tuple : userdags) {
            user2dags.put(tuple.user, tuple.dag);
            dag2users.put(tuple.dag, tuple.user);
        }
    } 

    private static class UserDagTuple {
        @JsonProperty
        String user;
        @JsonProperty
        String dag;
        
        private UserDagTuple() {
        }
        
        private UserDagTuple(String user, String dag) {
            this.user = user;
            this.dag = dag;
        }
    }

    public static void dumpDag(Path path, Dag dag) throws IOException {
        serializer.write(path, dag);
    }

}
