package net.aequologica.neo.dagr;

import java.io.IOException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import net.aequologica.neo.dagr.jgrapht.DagJGraphT;
import net.aequologica.neo.dagr.model.Dag;
import net.aequologica.neo.dagr.model.DagDocumentSerializer;
import net.aequologica.neo.geppaequo.document.DocumentHelper;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Lists;
import com.google.common.collect.Multimap;

public class Dags {

    private static DagDocumentSerializer serializer = new DagDocumentSerializer();

    private Map<String, Dag>         dags      = null;
    private Map<String, DagJGraphT>  dag2s     = null;
    private Multimap<String, String> user2dags = null;
    private Multimap<String, String> dag2users = null;

    /////////////////////////////
    // hand-made singleton
    static private Dags instance = null;
    static public Dags getInstance() {
        if (instance == null) {
            instance = new Dags();
        }
        return instance;
    }
    private Dags() {
        super();
    }
    /////////////////////////////

    public Collection<Dag> getDAGs() {
        return this.dags.values();
    }

    public Set<String> getDAGKeys() {
        return this.dags.keySet();
    }

    public Dag getDAG(String dagkey) {
        return this.dags.get(dagkey);
    }

    public DagJGraphT getDAG2(String dagkey) {
        return this.dag2s.get(dagkey);
    }
    public List<Dag> getUserDAGs(String username) {
        Collection<String> thisUserDags = user2dags.get(username);
        List<Dag> ret = new ArrayList<>(thisUserDags.size());
        for (String dagkey : thisUserDags) {
            Dag dag = this.dags.get(dagkey);
            if (dag != null) {
                ret.add(dag);
            }
        }
        return ret;
    }

    public List<Map.Entry<String, String>> load() {

        List<Map.Entry<String, String>> exceptions = Lists.newArrayList();

        Multimap<String, String> dag2Users0 = null;
        if (this.dag2users != null) {
            dag2Users0 = this.dag2users;
        }

        this.dags      = new HashMap<>();
        this.dag2s     = new HashMap<>();
        this.user2dags = ArrayListMultimap.create();
        this.dag2users = ArrayListMultimap.create();

        Path path = Paths.get("/dags");

        List<java.nio.file.Path> sources;
        try {
            sources = DocumentHelper.list(path);
        } catch (Exception e) {
            exceptions.add(new AbstractMap.SimpleImmutableEntry<String, String>(path.toString(), e.getMessage()));
            return exceptions;
        }

        for (java.nio.file.Path source : sources) {
            try {
                if (!source.getFileName().toString().toLowerCase().endsWith(".json")) {
                    continue;
                }
                Dag dag = serializer.read(source);
                if (dag == null) {
                    continue;
                }
                dag.setSource(source.toString());
                String dagkey = source.getFileName().toString();
                dag.setKey(dagkey);
                dags.put(dagkey, dag);

                // jgrapht
                dag2s.put(dagkey, new DagJGraphT(dag));

                if (dag2Users0 != null) {
                    Collection<String> users = dag2Users0.get(dagkey);
                    for (String user : users) {
                        user2dags.put(user, dagkey);
                        dag2users.put(dagkey, user);
                    }
                }
            } catch (Exception e) {
                exceptions.add(new AbstractMap.SimpleImmutableEntry<String, String>(source.toString(), e.getMessage()));
            }
        }
        return exceptions;
    }

    public boolean subscribe(String dagkey, String username) throws IOException {
        Dag dag = this.dags.get(dagkey);
        if (dag == null) {
            throw new IOException("["+dagkey+"] not found");
        }
        Collection<String> existingDags = user2dags.get(username);
        Collection<String> existingUsers = dag2users.get(dagkey);
        if (existingDags.contains(dagkey) && existingUsers.contains(username)) {
            return false;
        }
        user2dags.put(username, dagkey);
        dag2users.put(dagkey, username);
        return true;
    }

    public boolean unsubscribe(String dagkey, String username) throws IOException {
        Dag dag = dags.get(dagkey);
        if (dag == null) {
            throw new IOException("["+dagkey+"] not found");
        }
        Collection<String> existingDags = user2dags.get(username);
        Collection<String> existingUsers = dag2users.get(dagkey);
        if (!existingDags.contains(dagkey) && !existingUsers.contains(username)) {
            return false;
        }
        user2dags.remove(username, dagkey);
        dag2users.remove(dagkey, username);
        return true;
    }

    public Boolean isUserSubscribed(String dagkey, String username) {
        Collection<String> existingUsers = dag2users.get(dagkey);
        return existingUsers.contains(username);
    }

    public static void dumpDag(Path path, Dag dag) throws IOException {
        serializer.write(path, dag);
    }

}
