package net.aequologica.neo.dagr.jaxrs;

import static javax.ws.rs.core.Response.Status.UNAUTHORIZED;

import java.io.IOException;
import java.security.Principal;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import javax.servlet.ServletContext;
import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.DELETE;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.WebApplicationException;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;

import net.aequologica.neo.dagr.Dags;
import net.aequologica.neo.dagr.model.Dag;

import org.glassfish.jersey.server.mvc.Viewable;

@javax.ws.rs.Path("/")
public class Resource<T> {

    @javax.ws.rs.core.Context
    ServletContext context;

    Dags dags = Dags.getInstance();

    public Resource() throws IOException {
        super();
        this.dags.load();
    }

    @GET
    @Path("")
    @Produces(MediaType.APPLICATION_JSON)
    public List<DagInfo> getAllDAGs(@Context HttpServletRequest request) {
        String          username        = getUsername(request);
        Collection<Dag> dagsCollection  = this.dags.getDAGs();
        List<DagInfo>   ret             = buildDagInfoList(dagsCollection, username);
        return ret;
    }

    @GET
    @Path("")
    @Produces(MediaType.TEXT_HTML)
    public Viewable getAllDAGsAsHtml(@Context HttpServletRequest request) throws IOException {
        return new Viewable("/WEB-INF/dagr/dags", getAllDAGs(request));
    }

    @POST
    @Path("")
    @Produces(MediaType.APPLICATION_JSON)
    public Response reload() {
        return Response.status(Response.Status.OK).entity(this.dags.load()).build();
    }

    @GET
    @Path("{dag : .+}")
    @Produces(MediaType.TEXT_HTML)
    public Viewable getDAGasHTML(@PathParam("dag") String dagkey, @Context HttpServletRequest request) throws Exception {
        List<DagInfo> allDAGs = getAllDAGs(request);
        for (DagInfo di : allDAGs) {
            unsubscribe(di.key, request);
        }
        subscribe(dagkey, request);
        return new Viewable("/WEB-INF/dagr/dags", getAllDAGs(request));
    }

    @GET
    @Path("{dag : .+}")
    @Produces(MediaType.APPLICATION_JSON)
    public Dag getDAG(@PathParam("dag") String dagkey) throws Exception {
        return this.dags.getDAG(dagkey);
    }

    @GET
    @Path("{dag : .+}/topological")
    @Produces(MediaType.APPLICATION_JSON)
    public List<String> getTopological(@PathParam("dag") String dagkey) throws Exception {
        return this.dags.getDAG2(dagkey).getTopologicalOrder();
    }

    @GET
    @Path("{dag : .+}/topologicartifacts")
    @Produces(MediaType.APPLICATION_JSON)
    public List<String> getTopologicartifacts(@PathParam("dag") String dagkey) throws Exception {
        return this.dags.getDAG2(dagkey).getTopologicartifactsOrder();
    }

    @GET
    @Path("subscription")
    @Produces(MediaType.APPLICATION_JSON)
    public List<DagInfo> getUserDAGs(@Context HttpServletRequest request) throws Exception {
        String username = getUsername(request);
        if (username == null) {
            throw new WebApplicationException(UNAUTHORIZED);
        }
        Collection<Dag> dagsCollection = this.dags.getUserDAGs(username);
        List<DagInfo> ret = buildDagInfoList(dagsCollection, "" /* empty, not null username ====> dagInfo.subscribed = true */ );
        return ret;
    }

    private List<DagInfo> buildDagInfoList(Collection<Dag> dagsCollection, String username /* empty, not null username ====> dagInfo.subscribed = true */) {
        List<DagInfo> ret = new ArrayList<>(dagsCollection.size());
        for (Dag dag : dagsCollection) {
            DagInfo dagInfo     = new DagInfo();
            dagInfo.name        = dag.getName();
            dagInfo.key         = dag.getKey();
            dagInfo.url         = dag.getSource();
            dagInfo.subscribed  = username == null ? null : username.length() == 0 ? true : this.dags.isUserSubscribed(dag.getKey(), username);
            ret.add(dagInfo);
        }
        return ret;
    }

    @POST
    @Path("subscription/{dag : .+}")
    public Response subscribe(@PathParam("dag") String dagkey, @Context HttpServletRequest request) throws IOException {
        String username = getUsername(request);
        if (username == null) {
            return Response.status(Response.Status.UNAUTHORIZED).build();
        }
        if (this.dags.subscribe(dagkey, username)) {
            return Response.status(Response.Status.OK).build();
        } else {
            return Response.status(Response.Status.NOT_MODIFIED).build();
        }
    }

    @DELETE
    @Path("subscription/{dag : .+}")
    public Response unsubscribe(@PathParam("dag") String dagkey, @Context HttpServletRequest request) throws IOException {
        String username = getUsername(request);
        if (username == null) {
            return Response.status(Response.Status.UNAUTHORIZED).build();
        }
        if (this.dags.unsubscribe(dagkey, username)) {
            return Response.status(Response.Status.OK).build();
        } else {
            return Response.status(Response.Status.NOT_MODIFIED).build();
        }
    }

    public static String ANONYMOUS = "anonymous";

    private static String getUsername(HttpServletRequest request) {
        Principal userPrincipal = request.getUserPrincipal();
        if (userPrincipal == null) {
            return ANONYMOUS;
        }
        String username = userPrincipal.getName();
        if (username == null || username.isEmpty()) {
            return ANONYMOUS;
        }
        return username;
    }

    static public class DagInfo {
        private String name;
        private String key;
        private String url;
        private Boolean subscribed;

        public String getName() {
            return name;
        }
        public void setName(String name) {
            this.name = name;
        }
        public String getKey() {
            return key;
        }
        public void setKey(String key) {
            this.key = key;
        }
        public String getUrl() {
            return url;
        }
        public void setUrl(String url) {
            this.url = url;
        }
        public Boolean getSubscribed() {
            return subscribed;
        }
        public void setSubscribed(Boolean subscribed) {
            this.subscribed = subscribed;
        }
    }

}
