package net.aequologica.neo.dagr.jaxrs.travis;

import static com.fasterxml.jackson.annotation.JsonInclude.Include.NON_NULL;

import java.io.UnsupportedEncodingException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URLEncoder;
import java.util.Map;

import org.apache.http.Header;
import org.apache.http.HttpEntity;
import org.apache.http.StatusLine;
import org.apache.http.client.fluent.Request;
import org.apache.http.client.utils.URIBuilder;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.message.BasicHeader;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;

import net.aequologica.neo.dagr.DagOnSteroids.NodeCleaner;
import net.aequologica.neo.dagr.Scope;
import net.aequologica.neo.dagr.model.Dag.Node;
import net.aequologica.neo.geppaequo.config.ConfigRegistry;
import net.aequologica.neo.geppaequo.config.geppaequo.GeppaequoConfig;

import com.pivovarit.function.ThrowingBiFunction;

/*
cf. https://docs.travis-ci.com/user/triggering-builds

e.g.

POST
https://api.travis-ci.com/repo/repo_owner%2Frepo_name/requests

HEADERS
Authorization:token "<your travis token>" 
Content-Type:application/json
Accept:application/json
Travis-API-Version:3

BODY
{"request": {"branch":"develop"}}

*/
public class NodeCleanerTravis {
    
    private final static Logger log = LoggerFactory.getLogger(NodeCleanerTravis.class);
    
    final URI                                                                                                                   uri;
    final ThrowingBiFunction<Map.Entry<Scope, Boolean>, Node, NodeCleaner.NodeCleaningResult, NodeCleaner.NodeCleanerException> cleaningFunction;
    
    public NodeCleanerTravis(final String url, final String token) {
        this.uri     = URI.create(url);
        this.cleaningFunction = (scope, node) -> {
            URI nodeURI = this.uri;
            final Header[] headers = new Header[] {
                    new BasicHeader("Accept"             , "application/json; charset=utf-8"),
                    new BasicHeader("Authorization"      , "token " + token),
                    new BasicHeader("Content-Type"       , "application/json; charset=utf-8"),
                    new BasicHeader("Travis-API-Version" , "3"),
            };
            final String body = Body.string(
                    Body.Request.request(
                        node.getValue().getBranch() == null || node.getValue().getBranch().isEmpty() 
                        ? "master" 
                        : node.getValue().getBranch() 
                    )
            );
            final String proxyIfNeeded; {
                    GeppaequoConfig config = ConfigRegistry.CONFIG_REGISTRY.getConfig(GeppaequoConfig.class);
                    proxyIfNeeded = config.getProxyIfNeeded(this.uri.getHost());
            }
            
            try {
                nodeURI = calcNodeURI(this.uri, node);
                Request request = Request
                        .Post(nodeURI)
                        .setHeaders(headers)
                        .connectTimeout(5000)
                        .socketTimeout(5000)
                        .bodyString(body, ContentType.APPLICATION_JSON);
                if (proxyIfNeeded != null) {
                    request = request.viaProxy(proxyIfNeeded);
                }

                org.apache.http.client.fluent.Response response = request.execute();
                StatusLine statusLine = response.returnResponse().getStatusLine();
                log.debug("NodeCleanerTravis \nrequest {} \nheaders {} \nbody {} \nproxy {} \nstatus {}", request, headers, body, proxyIfNeeded, statusLine);
                return NodeCleaner.NodeCleaningResult.from(scope.getKey(), statusLine.getStatusCode(), statusLine.getReasonPhrase(), nodeURI);
                
            } catch (Throwable t) {
                log.debug("NodeCleanerTravis exception wrapped and rethrown as NodeCleanerException. {} | {} | \"{}\" | {}", nodeURI, headers, body, t.getMessage());
                throw new NodeCleaner.NodeCleanerException(nodeURI + " | "+ headers + " | " + body, t);
            }
            
        };
    }

    public ThrowingBiFunction<Map.Entry<Scope, Boolean>, Node,NodeCleaner.NodeCleaningResult, NodeCleaner.NodeCleanerException> getCleaningFunction() {
        return this.cleaningFunction;
    }
    
    static URI calcNodeURI(URI baseURI, Node node) throws URISyntaxException, UnsupportedEncodingException {
        if (baseURI                  == null || 
            node                     == null || 
            node.getValue()          == null || 
            node.getValue().getScm() == null || 
            node.getValue().getScm().isEmpty()) {
            throw new RuntimeException("no base uri, no node, or no node.value.scm");
        }
        
        final String path = getPathFromGitProtocol(node.getValue().getScm());
        final URI uri = URI.create(baseURI + "/repo/" + URLEncoder.encode(path, "UTF8") + "/requests");
        return uri;
    }

    
    static String getPathFromGitProtocol(String scm) throws URISyntaxException {
        org.eclipse.jgit.transport.URIish uri = new org.eclipse.jgit.transport.URIish(scm);
        String path = uri.getPath();
        path = removeLeadingSlashIfAny(path);
        path = removeTrailingDotGitIfAny(path);
        return path;
    }
    
    private static String removeTrailingDotGitIfAny(String path) {
        if (path.endsWith(".git")) return path.substring(0,path.length()-4);
        return path;
    }

    private static String removeLeadingSlashIfAny(String path) {
        if (path.startsWith("/")) return path.substring(1);
        return path;
    }

    @JsonIgnoreProperties
    static class Body {
        final static ObjectMapper mapper;

        static {
            mapper = new ObjectMapper();
            mapper.setSerializationInclusion(NON_NULL);
        }
        
        @JsonProperty
        final Request request;

        Body(final Request request) {
            this.request = request;
        }
        
        @Override
        public String toString() {
            try {
                return mapper.writeValueAsString(this);
            } catch (JsonProcessingException e) {
                log.error(e.getMessage());
            }
            return null;
        }

        static Body body(final Request request) {
            return new Body(request);
        }

        static String string(final Request request) {
            return body(request).toString();
        }
        
        static HttpEntity entity(final Request request) {
            return new StringEntity(string(request), ContentType.APPLICATION_JSON);
        }

        @JsonIgnoreProperties
        static class Request {
            @JsonProperty
            final String branch;

            Request(final String branch) {
                this.branch = branch;
            }

            static Request request(final String branch) {
                return new Request(branch);
            }
        }
    }

}
