package net.aequologica.neo.dagr.jaxrs.jenkins;
import static net.aequologica.neo.geppaequo.config.ConfigRegistry.CONFIG_REGISTRY;

import java.io.IOException;
import java.io.StringWriter;
import java.io.UnsupportedEncodingException;
import java.io.Writer;
import java.net.URI;
import java.net.URLEncoder;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.util.Map;
import java.util.Set;

import javax.ws.rs.core.UriBuilder;

import org.apache.http.HttpHost;
import org.apache.http.StatusLine;
import org.apache.http.client.fluent.Executor;
import org.apache.http.client.fluent.Form;
import org.apache.http.client.fluent.Request;
import org.apache.http.conn.ssl.TrustStrategy;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.github.jknack.handlebars.Handlebars;
import com.github.jknack.handlebars.Parser;
import com.github.jknack.handlebars.Template;
import com.github.jknack.handlebars.cache.ConcurrentMapTemplateCache;
import com.github.jknack.handlebars.io.StringTemplateSource;
import com.github.jknack.handlebars.io.TemplateSource;
import com.google.common.base.Charsets;
import net.aequologica.neo.dagr.DagOnSteroids.NodeCleaner;
import net.aequologica.neo.dagr.Scope;
import net.aequologica.neo.dagr.jaxrs.ResourceDags;
import net.aequologica.neo.dagr.model.Dag;
import net.aequologica.neo.dagr.model.Dag.Bumper;
import net.aequologica.neo.dagr.model.Dag.Gucrid;
import net.aequologica.neo.dagr.model.Dag.Node;
import net.aequologica.neo.geppaequo.config.geppaequo.GeppaequoConfig;
import net.aequologica.neo.geppaequo.config.geppaequo.GeppaequoConfig.HostNameAndPort;

import com.pivovarit.function.ThrowingBiFunction;

public class NodeCleanerJenkins {
    
    private final static Logger LOG = LoggerFactory.getLogger(NodeCleanerJenkins.class);
    
    final private URI                                                                                                                   uri;
    final private Executor                                                                                                              executor;
    final private ThrowingBiFunction<Map.Entry<Scope, Boolean>, Node, NodeCleaner.NodeCleaningResult, NodeCleaner.NodeCleanerException> cleaningFunction;
    final private GeppaequoConfig                                                                                                       geppaequoConfig;
    
    private static class HB {
        final private ConcurrentMapTemplateCache                                        cache;
        final private Handlebars                                                        handlebars; 
        final private Parser                                                            parser;
        private HB() {
            this.cache      = new ConcurrentMapTemplateCache();
            this.handlebars = new Handlebars();
            this.parser     = handlebars.getParserFactory().create(handlebars, "{{", "}}");
        }
        private String applyTemplateAsString(Dag dag, Dag.Node node, Scope scope, String templateAsString) throws IOException {

            // strange cache where you need to read the content ('source') before querying ?!
            // I guess caching means 'template generation caching'
            final TemplateSource    source           = new StringTemplateSource(dag.getName()+"_"+scope.toString()+"_nameTemplate", templateAsString);  
            final Template          template         = this.cache.get(source, this.parser);
            final Writer            writer           = new StringWriter();

            template.apply(DagAndNode.get(dag, node), writer);
            writer.flush();
            
            return writer.toString();
        }
    }
    final private HB hb;
    
    public NodeCleanerJenkins(
            final String                            dagName, 
            final String                            url, 
            final String                            token, 
            final Set<Map.Entry<String, String>>    params, 
            final String                            authorizationHeader,
            final Map<Scope, String>                nameTemplateMap,
            final Bumper                            bumper,
            final URI                               callback) {
        
        // handlebars helpers
        this.hb = new HB();
        
        final String dagPrefix;
        final String dagBranch;
        if (dagName != null &&  
            !dagName.trim().isEmpty()) {
            int lastDot = dagName.lastIndexOf('.');
            if (lastDot != -1) {
                dagPrefix = dagName.substring(0, lastDot);
                dagBranch = dagName.substring(lastDot+1);
            } else {
                dagPrefix = dagName;
                dagBranch = null;
            }
        } else {
            dagPrefix = dagName;
            dagBranch = null;
        }

        geppaequoConfig = CONFIG_REGISTRY.getConfig(GeppaequoConfig.class);

        
        final CloseableHttpClient httpClient;
        try {
            // @formatter:off
            httpClient = HttpClients
                            .custom()
                            .setSSLHostnameVerifier(new org.apache.http.conn.ssl.NoopHostnameVerifier())
                            .setSSLContext(new org.apache.http.ssl.SSLContextBuilder()
                                    .loadTrustMaterial(null, new TrustStrategy() {
                                         @Override
                                         public boolean isTrusted(X509Certificate[] arg0, String arg1) throws CertificateException {
                                             return true;
                                         }})
                                    .build())
                            .build();
            // @formatter:on
            this.executor = Executor.newInstance(httpClient);
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
        
        this.uri = URI.create(url);
        this.cleaningFunction = (tuple, node /* can be null */) -> {
            URI finalURL = null;
            final Scope     scope       = tuple.getKey();
            final Boolean   skipRelease = tuple.getValue(); // can be null
            try {
                // jobname
                String jobName;
                if (node == null) {
                    jobName = "~"+dagPrefix+"~";
                } else {
                    jobName = node.getName();
                    if (nameTemplateMap != null && nameTemplateMap.size() > 0) {
                        final String nameTemplate = nameTemplateMap.get(scope);
                        if (nameTemplate != null && !nameTemplate.isEmpty()) {
                            jobName = hb.applyTemplateAsString(node.getDag(), node, scope, nameTemplate);
                        }
                    }
                }
                
                // build url
                {
                    UriBuilder uriBuilder = UriBuilder.fromUri(this.uri).path("job").path(jobName);
                    // token
                    if (token != null && !token.trim().isEmpty()) {
                        uriBuilder.queryParam("token", token);
                    }
                    // params ?
                    if (params != null) {
                        uriBuilder.path("buildWithParameters");
                        // all 'untyped' parameters
                        for (Map.Entry<String, String> entry : params) {
                            if (entry.getKey()   != null && !entry.getKey().trim().isEmpty() &&
                                entry.getValue() != null && !entry.getValue().trim().isEmpty())
                            uriBuilder.queryParam(entry.getKey(), entry.getValue());
                        }
                        // release and next development versions
                        if (scope.equals(Scope.RELEASE) && node != null && node.getValue() != null && node.getValue().getGucrid() != null) {
                            if (skipRelease) {
                                uriBuilder.queryParam("releaseVersion", "");
                                uriBuilder.queryParam("developmentVersion", "");
                            } else {
                                Gucrid gucrid               = Gucrid.create(node.getValue().getGucrid());
                                String releaseVersion       = gucrid.getReleaseVersion().toString();
                                String developmentVersion   = gucrid.getNextDevelopmentVersion(bumper.fromNodeName(node.getName())).toString();
                                if (releaseVersion != null && !releaseVersion.trim().isEmpty()) {
                                    uriBuilder.queryParam("releaseVersion", releaseVersion);
                                }
                                if (developmentVersion != null && !developmentVersion.trim().isEmpty()) {
                                    uriBuilder.queryParam("developmentVersion", developmentVersion);
                                }
                            }
                        }
                        // statusCallbackURL
                        if (callback != null && node != null) {
                            uriBuilder.queryParam("statusCallbackURL", UriBuilder.fromUri(callback).path(
                                    "/api/dagr/v1" 
                                    + ResourceDags.__DAGS 
                                    + doubleEncode(node.getDag().getName())
                                    + ResourceDags.__BUSES 
                                    + scope
                                    + ResourceDags.__NODES 
                                    + doubleEncode(node.getName())
                                    + ResourceDags.__EVENTS));
                        }
                        // dagURL
                        if (dagName != null &&  
                            !dagName.trim().isEmpty()) {
                            uriBuilder.queryParam("dagURL", UriBuilder.fromUri(callback).path(
                                    "/api/dagr/v1" 
                                    + ResourceDags.__DAGS 
                                    + doubleEncode(dagName)));
                        }
                        // branch
                        if (node != null) {
                            if (node.getValue() != null && 
                                node.getValue().getBranch() != null && 
                                !node.getValue().getBranch().trim().isEmpty()) {
                                uriBuilder.queryParam("branch", node.getValue().getBranch().trim());
                            }
                        } else {
                            if (dagBranch != null &&  
                                !dagBranch.trim().isEmpty()) {
                                uriBuilder.queryParam("branch", dagBranch.trim());
                            }
                        }
                    } else {
                        uriBuilder.path("build");
                    }
                    finalURL = uriBuilder.build();
                }
                
                // build request
                Request request = Request.Post(finalURL).bodyForm( Form.form().build() );
                request.addHeader("Content-Type", "application/x-www-form-urlencoded; charset=UTF-8");
                
                {
                    if ( geppaequoConfig != null ) {
                        String proxyIfNeeded = geppaequoConfig.getProxyIfNeeded(this.uri.getHost());
                        if (proxyIfNeeded != null) {
                            request = request.viaProxy(proxyIfNeeded);
                        }
                    }
                }


                if (authorizationHeader != null && !authorizationHeader.trim().isEmpty()){
                    request.addHeader("Authorization", authorizationHeader);
                }
                
                org.apache.http.client.fluent.Response response = executor.execute(request);
                
                StatusLine statusLine = response.returnResponse().getStatusLine();
                
                LOG.debug("POST {} => status: {}", finalURL, statusLine);
                
                return NodeCleaner.NodeCleaningResult.from(scope, statusLine.getStatusCode(), statusLine.getReasonPhrase(), finalURL);
            } catch (Exception e) {
                throw new NodeCleaner.NodeCleanerException(finalURL, e);
            }
        };
    }
    
    public ThrowingBiFunction<Map.Entry<Scope, Boolean>, Node, NodeCleaner.NodeCleaningResult, NodeCleaner.NodeCleanerException>  getCleaningFunction() {
        return this.cleaningFunction;
    }

    static public class DagAndNode {
        final public Dag      dag;
        final public Dag.Node node;
        static private DagAndNode get(final Dag dag, final Node node) {
            return new DagAndNode(dag, node);
        }
        private DagAndNode(final Dag dag, final Node node) {
            this.dag   = dag;
            this.node  = node;
        }
        public Dag getDag() {
            return dag;
        }
        public Dag.Node getNode() {
            return node;
        }
    }
    
    private static String doubleEncode(final String name) {
        try {
            String utf8 = Charsets.UTF_8.name();
            return URLEncoder.encode(URLEncoder.encode(name, utf8), utf8);
        } catch (UnsupportedEncodingException e) {
            e.printStackTrace();
            return name;
        }
    }
    
}
