package net.aequologica.neo.dagr.jaxrs;

import static net.aequologica.neo.dagr.bus.BusEvent.Type.CLEAN_ORDER_ERROR;
import static net.aequologica.neo.dagr.bus.BusEvent.Type.CLEAN_ORDER_OK;
import static net.aequologica.neo.dagr.bus.BusEvent.Type.STATE_CHANGE_HACK;
import static net.aequologica.neo.dagr.jaxrs.ResourceDags.DagAndNode.dagAndNode;
import static net.aequologica.neo.dagr.jaxrs.ResourceDags.DagOnSteroidsAndNode.dagOnSteroidsAndNode;
import static net.aequologica.neo.dagr.jaxrs.ResourceDags.Error.error;
import static net.aequologica.neo.dagr.jaxrs.ResourceDags.Message.message;
import static javax.ws.rs.core.MediaType.APPLICATION_JSON;
import static javax.ws.rs.core.MediaType.APPLICATION_XML;
import static javax.ws.rs.core.MediaType.TEXT_HTML;
import static javax.ws.rs.core.MediaType.TEXT_PLAIN;

import java.io.IOException;
import java.net.DatagramSocket;
import java.net.InetAddress;
import java.net.URI;
import java.security.InvalidParameterException;
import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Observable;
import java.util.Observer;
import java.util.Set;
import java.util.SortedSet;
import java.util.stream.Collectors;

import javax.annotation.PostConstruct;
import javax.inject.Inject;
import javax.inject.Singleton;
import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.Consumes;
import javax.ws.rs.DELETE;
import javax.ws.rs.DefaultValue;
import javax.ws.rs.GET;
import javax.ws.rs.NotFoundException;
import javax.ws.rs.POST;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.WebApplicationException;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.Response.Status;
import javax.ws.rs.core.UriBuilder;

import org.glassfish.jersey.server.mvc.Viewable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.Function;
import com.google.common.base.Splitter;
import com.google.common.collect.Iterators;
import com.google.common.collect.Lists;
import net.aequologica.neo.dagr.DagOnSteroids;
import net.aequologica.neo.dagr.DagOnSteroids.DagCleaner;
import net.aequologica.neo.dagr.DagOnSteroids.NodeCleaner;
import net.aequologica.neo.dagr.DagOnSteroids.NodeCleaner.NodeState;
import net.aequologica.neo.dagr.DagOnSteroids.NodeNameVersion;
import net.aequologica.neo.dagr.DagOnSteroids.State;
import net.aequologica.neo.dagr.Dags;
import net.aequologica.neo.dagr.Scope;
import net.aequologica.neo.dagr.bus.Bus;
import net.aequologica.neo.dagr.bus.BusEvent;
import net.aequologica.neo.dagr.jaxrs.jenkins.NodeCleanerJenkins;
import net.aequologica.neo.dagr.jaxrs.travis.NodeCleanerTravis;
import net.aequologica.neo.dagr.model.Dag;
import net.aequologica.neo.dagr.model.Dag.Gucrid;
import net.aequologica.neo.dagr.model.Dag.Node;
import net.aequologica.neo.dagr.model.Dag.NodeValue;
import net.aequologica.neo.dagr.model.Pair;
import net.aequologica.neo.dagr.model.SubDagIncludes;

import net.thisptr.jackson.jq.exception.JsonQueryException;

@Singleton
@javax.ws.rs.Path("v1")
public class ResourceDags implements Observer {

    // ping
    final public static String _PING       = "_ping_";
    // views 
    final public static String _VIEWS      = "/views";
    // plurals
    final public static String __DAGS     = "/dags/";
    final public static String __BUSES    = "/buses/";
    final public static String __SUBS     = "/subs/";
    final public static String __NODES    = "/nodes/";
    final public static String __VERSIONS = "/versions/";
    final public static String __STATES   = "/states/";
    final public static String __EVENTS   = "/events/";
    // singulars
    final public static String __DESCRIPTION = "/description/";

    // actions on plural
    final public static String _RELOAD     = "reload";
    final public static String _SEARCH      = "search";
    // actions on singular 
    final public static String _INFO        = "/info";
    final public static String _JOURNAL     = "/journal";
    final public static String _RELOAD_ME   = "/"+_RELOAD;
    final public static String _RULES       = "/rules";
    final public static String _FILTER      = "/filter";
    final public static String _CURR_VER    = "/current";
    final public static String _NEXT_VER    = "/next";
    final public static String _TOPOLOGICAL = "/topological";
    final public static String _ISOMORPH    = "/isomorph";
    
    final private static Logger LOG        = LoggerFactory.getLogger(ResourceDags.class);
    final private static String SIMPLENAME = ResourceDags.class.getSimpleName();

    final private GaranceWrapper garanceWrapper = GaranceWrapper.THIS;

    @Inject
    private Dags dags;
    
    @Context
    private HttpServletRequest request;

    @Inject 
    public ResourceDags() throws IOException {
        super();
    }

    @PostConstruct
    public void postConstruct() {
        this.dags.addObserver(this);
        Collection<Map.Entry<String, Collection<Map.Entry<String, String>>>> exceptions = new ArrayList<>();
        update(this.dags, exceptions);
    }

    @Override
    /**
     * arg is one of:
     * 
     * Boolean.TRUE                                                          :  start reload all dags
     * String                                                                :  start reload a dag
     * Map.Entry<String,            Collection<Map.Entry<String, String>>>   :  results loading a dag
     * Collection<Map.Entry<String, Collection<Map.Entry<String, String>>>>  :  results loading all dags
     * 
     */
    public void update(Observable o, Object arg) {
        if (!(o instanceof Dags) || !this.dags.equals(o)) {
            return;
        }
        if (arg instanceof Boolean && (boolean)arg) {
            // loading of all dags has started, remove all subscriptions
            garanceWrapper._closeGaranceSubscriptions();
        } else if (arg instanceof String) {
            // loading of one dag has started, remove the attached subscription, if any
            String dagName = (String)arg;
            garanceWrapper._closeGaranceSubscription(dagName);
        } else if (arg instanceof Map.Entry<?,?>) {
            // results loading one dag
            @SuppressWarnings("unchecked")
            Map.Entry<String, Collection<Map.Entry<String, String>>> thisDagEntry  = (Map.Entry<String, Collection<Map.Entry<String, String>>>)arg;
            DagOnSteroids  dagOnSteroids = this.dags.getDagOnSteroids(thisDagEntry.getKey().toString());
            if (dagOnSteroids != null) {
                for (Entry<Scope, Bus<Node, Scope>> busEntry : dagOnSteroids.getBusEntries()) {
                    thisDagEntry.getValue().addAll(garanceWrapper._initGaranceSubscription(dagOnSteroids, busEntry));
                }
                thisDagEntry.getValue().addAll(_initProperties(dagOnSteroids));
            }
        } else if (arg instanceof Collection<?>) {
            // results loading all dags
            @SuppressWarnings("unchecked")
            Collection<Map.Entry<String, Collection<Map.Entry<String, String>>>> exceptions = (Collection<Map.Entry<String, Collection<Map.Entry<String, String>>>>)arg;
            Collection<DagOnSteroids> dagOnSteroidss = this.dags.getDagOnSteroidss();
            
            for ( DagOnSteroids dagOnSteroids: dagOnSteroidss) {
                for (Entry<Scope, Bus<Node, Scope>> busEntry : dagOnSteroids.getBusEntries()) {
                    exceptions.add(tuple(dagOnSteroids.getDag().getName(), garanceWrapper._initGaranceSubscription(dagOnSteroids, busEntry)));
                }
                exceptions.add(tuple(dagOnSteroids.getDag().getName(), _initProperties(dagOnSteroids)));
            }
        }
    }
    /*
    private Collection<Map.Entry<String, String>> _initGaranceSubscription(DagOnSteroids dagOnSteroids, String dagName, Entry<Scope, Bus<Node, Scope>> busEntry) {
        final Collection<Map.Entry<String, String>> exceptions = Lists.newArrayList();
        try {
            // register an observer on the bus that will save build durations to garance key/value store
            Disposable subscription = busEntry.getValue().toObservable()
                                           .filter   ( event -> event.getType().equals(BusEvent.Type.CLEAN_OK))
                                           .map      ( event -> event.get())
                                           .subscribe( node  -> {
                                                if (node                                                != null &&
                                                    node.getId()                                        != null &&
                                                    node.getDag()                                       != null &&
                                                    node.getDag().getName()                             != null &&
                                                    dagOnSteroids.getDagCleaner(busEntry.getKey()).getNodeCleaner(node)                  != null && 
                                                    dagOnSteroids.getDagCleaner(busEntry.getKey()).getNodeCleaner(node).getDuration()    != null) {

                                                    String key = node.getDag().getName() + "::" + busEntry.getKey() + "::" + node.getId();

                                                    Long duration = dagOnSteroids.getDagCleaner(busEntry.getKey()).getNodeCleaner(node).getDuration();
                                                    this.garanceSeries.put(key, duration).serialize();
                                                }
                                           });
            EnumMap<Scope, Disposable> dagSubscriptions = this.subscriptions.get(dagName);
            if (dagSubscriptions == null) {
                dagSubscriptions = new EnumMap<>(Scope.class);
                this.subscriptions.put(dagName, dagSubscriptions);
            }
            dagSubscriptions.put(busEntry.getKey(), subscription);
        } catch (Exception e) {
            exceptions.add(tuple(e.getClass().getSimpleName(), e.getMessage()));
        }
        return exceptions;
    }
    */
    private Collection<Map.Entry<String, String>> _initProperties(final DagOnSteroids dagOnSteroids) {
        final Map<String, String> properties = dagOnSteroids.getProperties();
        if (properties == null) {
            return Collections.emptyList();
        }
        final Collection<Map.Entry<String, String>> exceptions = Lists.newArrayList();
        try {
            final String dagCleanerType                = properties.get("type" );
            final String dagCleanerName                = properties.get("name" );
            final String dagCleanerUrl                 = properties.get("url"  );
            final String dagCleanerToken               = properties.get("token");
            final String dagCleanerAuthorizationHeader = properties.get("authorizationHeader");
            final String dagCleanerCallbackHost        = properties.get("callbackHost");
            
            final Map<Scope, String> dagCleanerNameTemplates = dagOnSteroids.getNameTemplates();

            if (dagOnSteroids   != null &&
                dagCleanerType  != null && 
                dagCleanerName  != null &&
                dagCleanerUrl   != null &&
                dagCleanerToken != null) {
                if (dagCleanerType.equals("jenkins")) {
                    Set<Map.Entry<String, String>> dagCleanerParams = _getBuildParamsIfAny(properties.get("params"), exceptions);
                    URI callback = null;
                    if (this.request != null) {
                        String callbackURIAsString = this.request.getRequestURL().toString().replace(
                            this.request.getRequestURI(), 
                            this.request.getContextPath());
                        UriBuilder uriBuilder = UriBuilder.fromUri(callbackURIAsString);
                        if (dagCleanerCallbackHost != null && !dagCleanerCallbackHost.trim().isEmpty()) {
                            uriBuilder = uriBuilder.host(dagCleanerCallbackHost);
                        } else if (this.request.getServerName().equals("localhost")) {
                            try (final DatagramSocket socket = new DatagramSocket()){
                                socket.connect(InetAddress.getByName("8.8.8.8"), 10002);
                                String ip = socket.getLocalAddress().getHostAddress();
                                uriBuilder = uriBuilder.host(ip);
                           }
                        }
                        callback = uriBuilder.build();
                        
                    }

                    dagOnSteroids.setCleaningFunction(
                        new NodeCleanerJenkins(
                                dagOnSteroids.getDag().getName(),
                                dagCleanerUrl, 
                                dagCleanerToken,
                                dagCleanerParams, 
                                dagCleanerAuthorizationHeader,
                                dagCleanerNameTemplates,
                                dagOnSteroids.bumper,
                                callback).getCleaningFunction()
                    );
                } else if (dagCleanerType.equals("travis")) {
                    final String dagCleanerUrlProxy = properties.get("urlProxy"  );
                    
                    dagOnSteroids.setCleaningFunction(
                        new NodeCleanerTravis(
                                dagCleanerUrlProxy!=null ? dagCleanerUrlProxy : dagCleanerUrl, 
                                dagCleanerToken).getCleaningFunction()
                    );
                }
            }
        } catch (Exception e) {
            exceptions.add(tuple(e.getClass().getSimpleName(), e.getMessage()));
        }
        return exceptions;
    }

    private Set<Map.Entry<String, String>> _getBuildParamsIfAny(final String dagCleanerParams, Collection<Map.Entry<String, String>> exceptions) {
        if (dagCleanerParams == null) {
            return null;
        }
        Set<Map.Entry<String, String>> params = Collections.emptySet();
        ObjectMapper objectMapper = new ObjectMapper();
        try {
            params = objectMapper.readValue(dagCleanerParams, new TypeReference<Set<Map.Entry<String, String>>>() {});
        } catch (IOException e) {
            exceptions.add(tuple("params="+dagCleanerParams, e.getClass().getSimpleName() + " - " + e.getMessage()));
        }
        return params;
    }

    /** 
     * <h4>verify that entry point is alive and kicking</h4>
     * 
     * @return "pong"
     */
    @GET
    @Path(_PING)
    @Produces(TEXT_PLAIN)
    public String ping_GET_TEXT() {
        return "_pong_";
    }

    /**
     * <h4>reload all dags</h4>
     * 
     * <p>
     * returns an array of objects; each object maps a dag name to an array of exceptions/warnings; array is empty when dag reload is ok. 
     * </p>
     * e.g.:
     * <pre>
     * [{
     *   "good dag name": []
     * },{
     *   "baad dag name": [{
     *     "Exception": "Tabernacle!"
     *   }]
     * }] 
     * </pre>
     * @return HTTP status code 200 OK 
     */
    @POST
    @Path(__DAGS+_RELOAD)
    @Produces(APPLICATION_JSON)
    public Response dagReloadAll_POST() {
        final Collection<Map.Entry<String, Collection<Map.Entry<String, String>>>> allExceptions = this.dags.loadDags();
        return Response.status(Response.Status.OK).entity(allExceptions).build();
    }

    /**
     * <h4>reload one dag</h4>
     * 
     * @param dagName the name of the dag
     * @return HTTP status code 404 not found,  
     * 500 Internal Server Error if exceptions are raised during load,  
     * or 204 No Content when successful
     */
    @POST
    @Path(__DAGS+"{dag}"+_RELOAD_ME)
    @Produces(APPLICATION_JSON)
    public Response dagReloadThis_POST(final @PathParam("dag") String dagName) {
        @SuppressWarnings("unused")
        final Dag dag = _getDag(dagName); // will throw 404 if not found
        final Map.Entry<String, Collection<Map.Entry<String, String>>> thisDagLoadExceptions = this.dags.loadDag(dagName);
        if (thisDagLoadExceptions.getValue().size() > 0) {
            return Response.status(Response.Status.INTERNAL_SERVER_ERROR).entity(thisDagLoadExceptions).build();
        } else {
            return Response.status(Response.Status.NO_CONTENT).build();
        }
    }

    /**
     * <h4>search all dags for nodes with a given name and/or a given attribute, if any.</h4>
     * 
     * <p>
     * returns an array of dag names. 
     * </p>
     * 
     * <p>
     * if node is null or empty, all nodes are searched for the matching attribute.
     * </p>
     * 
     * <p>
     * if node is null or empty, and one of key or value is null or empty, then an empty array is returned.
     * </p>
     * 
     * @return an array of dag names
     * @param key node attribute, e.g. "id" or "value.branch".
     * @param node search for dags that contains a node with this name.
     * @param value string or <a href="https://stedolan.github.io/jq/manual/#RegularexpressionsPCRE">regular expression</a> to match, e.g. "develop" or "dev.*".
     * @throws Exception 
     */
    @GET
    @Path(__DAGS+_SEARCH)
    @Consumes(TEXT_PLAIN)
    @Produces(APPLICATION_JSON)
    public List<String> dagSearch_GET(
            @QueryParam("node")  @DefaultValue("") String nodeName,
            @QueryParam("key")   @DefaultValue("") String key,
            @QueryParam("value") @DefaultValue("") String value) {
        // check parameters
        if (nodeName.isEmpty() && (key.isEmpty() || value.isEmpty())) {
            return Collections.emptyList();
        }
        List<String> ret = new ArrayList<>();
        // loop on all dags
        for (Dag dag : this.dags.getDAGs()){
            String filter = ".nodes[]";
            if (!nodeName.isEmpty()) {
                filter += "| select(.name | test(\""+nodeName+"\"))";
            }
            if (!key.isEmpty() && !value.isEmpty()) {
                filter += "| select(."+key+" | test(\""+value+"\"))";
            }
            
            final Object o;
            try {
                o = dag.filter(filter);
                if (o instanceof List) {
                    List<?> list = (List<?>)o;
                    if (list.size() > 0 ) {
                        ret.add(dag.getName());
                    }
                }
            } catch (JsonQueryException e) {
                LOG.warn("Exception applying filter\""+filter+"\" to dag /"+dag.getName()+": "+e.getMessage());
            }
        }
        return ret;
    }


    @GET
    @Path(__DAGS)
    @Produces(APPLICATION_JSON)
    public List<DagInfo> dagInfos_GET_JSON() throws IOException {
        return this.dags.getDagOnSteroidss().stream().map(d -> DagInfo.create(d)).collect(Collectors.toList());
    }
    
    /** <h4>retrieve one dag</h4> 
     *  
     * @param dagName
     * @return
     * @throws IOException
     */
    @GET
    @Path(__DAGS + "{dag}")
    @Produces(APPLICATION_JSON)
    public Dag dag_GET_JSON(final @PathParam("dag") String dagName) throws IOException {
        return _getDag(dagName);
    }

    /**
     * <h4>retrieve one dag metadata</h4>
     * 
     * @param dagName
     * @return
     * @throws Exception
     */
    @GET
    @Path(__DAGS + "{dag}" + _INFO)
    @Produces(APPLICATION_JSON)
    public DagInfo dagInfo_GET_JSON(final @PathParam("dag") String dagName) throws Exception {
        return _getDagInfo(dagName, null);
    }

    // filter
    /** 
     * <h4>slice and filter and map and transform dag structured data</h4>
     * 
     * <p>
     * using <a href="https://github.com/eiiches/jackson-jq">jackson-jq</a>, a java port of <a href="https://stedolan.github.io/jq/">jq</a> (try it <a href="https://jqplay.org">here</a>)
     * <p>
     * 
     * <p>
     * e.g.:
     * request body to select all SNAPSHOT nodes: 
     * <p>
     * </p>
     * <code>.nodes[] | select(.value.gucrid | test("SNAPSHOT$")) | .name</code>
     * </p>
     * 
     * <p>
     * if the request body is not semantically correct, method returns <a href="https://www.ietf.org/rfc/rfc4918.txt">422 Unprocessable Entity</a> with the description of the error.
     * </p>
     * 
     * e.g. <code>.nodes[]value</code> (missing point between [] and value) returns
     * <pre>{ "error": "Encountered <IDENTIFIER> 'value' at line 1, column 9. Was expecting one of:
     *     &lt;EOF&gt; 
     *     'and' ...
     *     'or' ...
     * (...)</pre>
     * 
     * <p>
     * Optional query parameter <code>returnSingleElementArray</code>: 
     * if returnSingleElementArray is false (default), and the return value would be an array containing only one element, then the element itself is returned instead of the array 
     * </p>
     * 
     * @param nodeNames json array of node names
     * @param dagName name of the dag
     * @param returnSingleElementArray default false     
     * @return a sub-dag key
     * @throws Exception json query compilation or execution error
     */
    @POST
    @Path(__DAGS + "{dag}" + _FILTER)
    @Consumes(TEXT_PLAIN)
    @Produces(APPLICATION_JSON)
    public Response dagFilter_POST(
            final String jsonQuery, 
            final @PathParam("dag") String dagName,
            final @QueryParam("returnSingleElementArray") @DefaultValue("false") Boolean returnSingleElementArray) throws Exception {
        final Dag dag = _getDag(dagName);
        if (dag == null) {
            throw new NotFoundException(Response.status(Status.NOT_FOUND).entity("dag /"+dagName+"/ not found").build());
        }
        Object filtered = dag.filter(jsonQuery);
        if (!returnSingleElementArray && filtered instanceof List<?>) {
            List<?> list = (List<?>)filtered;
            if (list.size() == 1) {
                filtered = list.get(0);
            }
        }
        return Response.ok().entity(filtered).build();
    }

    @GET
    @Path(__DAGS + "{dag}" + _CURR_VER)
    @Produces(APPLICATION_JSON)
    public Response dagCurrent_GET(
            final @PathParam("dag") String  dagName) throws Exception {
        final DagOnSteroids dagOnSteroids = _getDagOnSteroids(dagName);
        if (dagOnSteroids == null || dagOnSteroids.getDag() == null) {
            throw new NotFoundException(Response.status(Status.NOT_FOUND).entity("dag /"+dagName+"/ not found").build());
        }
        
        Collection<DagOnSteroids.NodeNameVersion> currents = dagOnSteroids.getCurrents(true);
        
        return Response.ok().entity(currents).build();
    }

    @GET
    @Path(__DAGS + "{dag}" + _NEXT_VER)
    @Produces(APPLICATION_JSON)
    public Response dagNext_GET(
            final @PathParam("dag")                                   String  dagName,
            final @QueryParam("allowSnapshots") @DefaultValue("true") Boolean allowSnapshots) throws Exception {
        final DagOnSteroids dagOnSteroids = _getDagOnSteroids(dagName);
        if (dagOnSteroids == null || dagOnSteroids.getDag() == null) {
            throw new NotFoundException(Response.status(Status.NOT_FOUND).entity("dag /"+dagName+"/ not found").build());
        }
        
        Collection<DagOnSteroids.NodeNameVersion> nexts = dagOnSteroids.getNexts(true, allowSnapshots);
        
        return Response.ok().entity(nexts).build();
    }

    // subdag resource ===============
    /** <h4>retrieve all sub-dags of one dag</h4>
     * 
     * @param dagName
     * @return a map of sub-dag keys to sub-dag contents 
     * @throws Exception
     */
    @GET
    @Path(__DAGS + "{dag}" + __SUBS)
    @Produces(APPLICATION_JSON)
    public Collection<Map.Entry<String, SortedSet<String>>> dagSubs_GET(final @PathParam("dag") String dagName) throws Exception {
        final Dag dag = _getDag(dagName);
        if (dag == null) {
            throw new NotFoundException(Response.status(Status.NOT_FOUND).entity("dag /"+dagName+"/ not found").build());
        }
        return dag.getSubDagIds();
    }

    /** <h4>create a sub-dag from a set of node names</h4>
     * <p> 
     * (optional) query parameter "<b>with</b>"<br/>one of "PREDECESSORS" or "SUCCESSORS", default is null)
     * <br/>&nbsp;&nbsp;&nbsp;&nbsp; 
     * add predecessors or successors nodes (and relevant links)
     * </p>
     * <p> 
     * (optional) query parameter "<b>depth</b>" (integer default is 0)
     * <br/> 
     * <table><tr><th>
     * depth</th><th>&nbsp;
     * </th></tr><tr><td>
     * 0</td><td>do not add any predecessors or successors
     * </td></tr><tr><td>
     * 1</td><td>if "with" query parameter is not null, add <i>only direct</i> predecessors or successors (and relevant links) to the result
     * </td></tr><tr><td>
     * -1</td><td>if "with" query parameter is not null, add <i>all transitive</i> predecessors or successors (and relevant links) to the result
     * </td></tr></table>
     * </p>
     * @param nodeNames
     * @param dagName
     * @param includes
     * @param depth
     * @return
     * @throws Exception
     */
    @POST
    @Path(__DAGS + "{dag}" + __SUBS)
    @Consumes(APPLICATION_JSON)
    @Produces(TEXT_PLAIN)
    public String dagSub_POST(
                                                            Set<String>     nodeNames, 
            final @PathParam("dag")                         String          dagName, 
            final @QueryParam("with")                       SubDagIncludes  includes, 
            final @QueryParam("depth")  @DefaultValue("0")  Integer         depth) throws Exception {
        final Dag dag = _getDag(dagName);
        if (dag == null) {
            throw new NotFoundException(Response.status(Status.NOT_FOUND).entity("dag /"+dagName+"/ not found").build());
        }
        if (includes != null || depth != 0) {
            nodeNames = dag.expandNodesNames(nodeNames, includes, depth);
        }
        return dag.registerSubDag(nodeNames);
    }

    /** <h4>retrieve a sub-dag</h4>
     * <p> 
     * (optional) query parameter "<b>with</b>"<br/>one of "PREDECESSORS" or "SUCCESSORS", default is null)
     * <br/>&nbsp;&nbsp;&nbsp;&nbsp; 
     * add predecessors or successors nodes (and relevant links)
     * </p>
     * <p> 
     * (optional) query parameter "<b>depth</b>" (integer default is 0)
     * <br/> 
     * <table><tr><th>
     * depth</th><th>&nbsp;
     * </th></tr><tr><td>
     * 0</td><td>do not add any predecessors or successors
     * </td></tr><tr><td>
     * 1</td><td>if "with" query parameter is not null, add <i>only direct</i> predecessors or successors (and relevant links) to the result
     * </td></tr><tr><td>
     * -1</td><td>if "with" query parameter is not null, add <i>all transitive</i> predecessors or successors (and relevant links) to the result
     * </td></tr></table>
     * </p>
     *  
     * @param dagName
     * @param subDagKey
     * @param includes
     * @param depth
     * @return
     * @throws Exception
     */
    @GET
    @Path(__DAGS + "{dag}" + __SUBS + "{sub}")
    @Produces(APPLICATION_JSON)
    public Dag dagSub_GET_JSON(
            final @PathParam("dag")                         String          dagName, 
            final @PathParam("sub")                         String          subDagKey, 
            final @QueryParam("with")                       SubDagIncludes  includes, 
            final @QueryParam("depth")  @DefaultValue("0")  Integer         depth) throws Exception {
        final Dag dag = _getDag(dagName);
        if (dag == null) {
            throw new NotFoundException(Response.status(Status.NOT_FOUND).entity(error("dag ["+dagName+"] not found")).build());
        }
        final Dag subdag = dag.getSubDag(subDagKey);
        if (subdag == null) {
            throw new NotFoundException(Response.status(Status.NOT_FOUND).entity(error("dag > sub ["+dagName+" > "+subDagKey+"] not found")).build());
        }
        if (includes == null || depth == 0) {
            return subdag;
        }
        Set<String> expandedNodesNames  = dag.expandNodesNames(subdag.getNodes().stream().map(n -> n.getName()).collect(Collectors.toSet()), includes, depth);
        String      subDagId            = dag.registerSubDag(expandedNodesNames);
        return dag.getSubDag(subDagId);
    }

    @POST
    @Path(__DAGS + "{dag}" + __SUBS + "{sub}" + __DESCRIPTION + "{description}" )
    @Produces(TEXT_PLAIN)
    public String dagSubSave_PUT_JSON(
            final @PathParam("dag")         String dagName, 
            final @PathParam("sub")         String subDagKey, 
            final @PathParam("description") String description) throws Exception {
        final Dag dag = _getDag(dagName);
        if (dag == null) {
            throw new NotFoundException(Response.status(Status.NOT_FOUND).entity(error("dag ["+dagName+"] not found")).build());
        }
        final Dag subdag = dag.getSubDag(subDagKey);
        if (subdag == null) {
            throw new NotFoundException(Response.status(Status.NOT_FOUND).entity(error("dag > sub ["+dagName+" > "+subDagKey+"] not found")).build());
        }
        java.nio.file.Path path = dags.saveSub(dag, subDagKey, description);
        
        return path.toString();
    }

    @DELETE
    @Path(__DAGS + "{dag}" + __SUBS + "{sub}")
    public void dagSub_DELETE(final @PathParam("dag") String dagName, final @PathParam("sub") String subDagKey) throws Exception {
        final Dag dag = _getDag(dagName);
        if (dag == null) {
            throw new NotFoundException(Response.status(Status.NOT_FOUND).entity(error("dag ["+dagName+"] not found")).build());
        }
        String removeSubDag = dag.unregisterSubDag(subDagKey);
        if (removeSubDag == null) {
            throw new NotFoundException(Response.status(Status.NOT_FOUND).entity(error("subgraph ["+subDagKey+"] for dag ["+dagName+"] not found")).build());
        }
    }

    /** <h4>check if two dags are isomorph</h4>  
     * 
     * <p>
     * cf. <a href="https://en.wikipedia.org/wiki/Graph_isomorphism">https://en.wikipedia.org/wiki/Graph_isomorphism</a>
     * </p
     * 
     * <p>
     * return an empty array if dags are not isomorph
     * </p>
     * 
     * <p>
     * is dag are isomorph, return a mapping of nodes between left and right dags, e.g.:
     * </p>
     * 
     * <pre>
     * [[{
     *   "left": {
     *     "id": "infinite",
     *     "name": "8",
     *   },
     *   "right": {
     *     "id": "fin",
     *     "name": "fin",
     *   }
     * }]
     * </pre>
     *  
     * @param dagName
     * @param otherDagName
     * @return if dags are not isomorph, an empty collection. If dags are isomorph, a mapping of their nodes.
     * @throws Exception
     */
    @GET
    @Path(__DAGS + "{dag}" + _ISOMORPH + "/{otherDag}")
    @Produces(APPLICATION_JSON)
    public Collection<List<Pair<Node, Node>>> isomorph_GET_JSON(
            final @PathParam("dag")      String dagName,
            final @PathParam("otherDag") String otherDagName
          ) throws Exception {
        final Dag dag   = _getDag(dagName);
        final Dag other = _getDag(otherDagName);
        return dag.getIsomorphism(other);
    }
    public static enum TopologicalReturnType {
        ID, NAME, GAV
    }

    /** <h4>get a topological sort of all nodes of one dag</h4> 
     * 
     * @param dagName
     * @param type
     * @return a topologically sorted list of nodes
     * @throws Exception
     */
    @GET
    @Path(__DAGS + "{dag}" + _TOPOLOGICAL)
    @Produces(APPLICATION_JSON)
    public List<String> topological_GET_JSON(
            final @PathParam("dag")                          String                dagName,
            final @QueryParam("type")  @DefaultValue("ID")  TopologicalReturnType type
          ) throws Exception {
        final Dag dag = _getDag(dagName);
        Iterator<Node> topologicalOrderIterator = dag.getTopologicalOrderIterator();
        final Function<Node, String> transform;
        if (type == null || type.equals(TopologicalReturnType.ID)) {
            transform = n -> n.getId();
        } else if (type.equals(TopologicalReturnType.NAME)) {
            transform = n -> n.getName();
        } else if (type.equals(TopologicalReturnType.GAV)) {
            transform = node-> {
                final NodeValue value = node.getValue();
                if (value == null) { 
                    return node.getId();
                }
                final String gucrid = value.getGucrid();
                if (gucrid == null || gucrid.length() == 0) {
                    return node.getId();
                }
                final Iterable<String> split = Splitter.on(':').split(gucrid);
                final Iterator<String> iterator = split.iterator();
                if (!iterator.hasNext()) {
                    return node.getId();
                }                
                @SuppressWarnings("unused")
                final String groupId = iterator.next();
                if (!iterator.hasNext()) {
                    return node.getId();
                }
                final String artifactId = iterator.next();
                if (artifactId == null || artifactId.toString().length() == 0) {
                    return node.getId();
                }
                return gucrid;
            };
        } else {
            throw new InvalidParameterException("invalid parameter 'type' ('"+type+"') in query string. must be one of "+TopologicalReturnType.values()+"; when not type parameter is given, type='"+TopologicalReturnType.ID+"' is the default.");
        }
        
        return Lists.<String>newArrayList(Iterators.transform(topologicalOrderIterator, transform));
    }

    /** <h4>get journal of cleaning process</h4> 
     * 
     * @param dagName
     * @param scope
     * @return
     * @throws Exception
     */
    @GET
    @Path(__DAGS + "{dag}" + __BUSES + "{bus}" + _JOURNAL)
    @Produces(TEXT_PLAIN)
    public String dagBusJournal_GET_TEXT_PLAIN(
            final @PathParam("dag") String  dagName,
            final @PathParam("bus") Scope   scope) throws Exception {
        final DagOnSteroids dagOnSteroids  = this.dags.getDagOnSteroids(dagName);
        if (dagOnSteroids == null) {
            throw new NotFoundException(Response.status(Status.NOT_FOUND).entity(error("dag ["+dagName+"] not found").valueOf(MediaType.TEXT_PLAIN_TYPE)).build());
        }
        return dagOnSteroids.getDagCleaner(scope).getJournalAsString();
    }

    /////////////////////////////////////////////////////
    /** <h4>cancel cleaning of one dag</h4>
     * 
     * @param dagName
     * @param scope
     * @return
     * @throws Exception
     */
    @DELETE
    @Path(__DAGS + "{dag}" + __BUSES + "{bus}")
    @Produces(APPLICATION_JSON)
    public Response dagBus_DELETE(
            final @PathParam("dag") String  dagName,
            final @PathParam("bus") Scope   scope) throws Exception {
        return dagBusSub_DELETE(dagName, scope, null);
    }
    
    /**
     * <h4>cancel cleaning of one sub-dag</h4>
     * 
     * @param dagName
     * @param scope
     * @param subDagKey
     * @return
     * @throws Exception
     */
    @DELETE
    @Path(__DAGS + "{dag}" + __BUSES + "{bus}" + __SUBS + "{sub}")
    @Produces(APPLICATION_JSON)
    public Response dagBusSub_DELETE(
            final @PathParam("dag") String  dagName,
            final @PathParam("bus") Scope   scope,
            final @PathParam("sub") String  subDagKey) throws Exception {
        final DagOnSteroids dagOnSteroids = _getDagOnSteroids(dagName);
        final DagCleaner    dagCleaner    = dagOnSteroids.getDagCleaner(scope);
        String info = null;
        State state = dagCleaner.getState();
        if (state== null || !state.equals(DagOnSteroids.State.RUNNING)) {
            info = "cleaning not running, nothing to cancel";
        } else {
            dagCleaner.cancel();
            if (!dagCleaner.getState().equals(DagOnSteroids.State.RUNNING)) {
                info = "cleaning cancelled (nb. node cleanings already ordered or currently running will continue until completion)";
            } else {
                info = "oooops, seems that cleaning could not be cancelled ... maybe retry ?";
            }
        }
        return Response.ok(message(info)).build();
    }
    /////////////////////////////////////////////////////

    /////////////////////////////////////////////////////
    @GET
    @Path(__DAGS + "{dag}" + __BUSES + "{bus}")
    @Produces(TEXT_PLAIN)
    public Response dagBus_GET(
            final @PathParam("dag")                         String  dagName,
            final @PathParam("bus")                         Scope   scope) throws Exception {
        DagOnSteroids.State state = null;
        final DagOnSteroids dagOnSteroids = _getDagOnSteroids(dagName);
        try {
            final DagCleaner dagCleaner = dagOnSteroids.getDagCleaner(scope);
            state = dagCleaner.getState();
        } catch (Exception e) {
            throw new WebApplicationException(Response.status(Status.INTERNAL_SERVER_ERROR).entity(error(e.getMessage())).build());            
        }
        return Response.ok().entity(state == null ? "null" : state.toString()).build();
    }

    /** <h4>trigger cleaning of one dag</h4>
     * 
     * @param dagName
     * @param scope
     * @return
     * @throws Exception
     */
    @PUT
    @Path(__DAGS + "{dag}" + __BUSES + "{bus}")
    @Produces(APPLICATION_JSON)
    public Response dagBus_PUT(
            final @PathParam("dag")                         String  dagName,
            final @PathParam("bus")                         Scope   scope) throws Exception {
        return privateDagBusPut(dagName, scope, null);
    }
    /** <h4>trigger cleaning of one dag</h4>
     * 
     * @deprecated use PUT instead
     * 
     * @param dagName
     * @param scope
     * @return
     * @throws Exception
     */
    @PATCH
    @Deprecated
    @Path(__DAGS + "{dag}" + __BUSES + "{bus}")
    @Produces(APPLICATION_JSON)
    public Response dagBus_PATCH(
            final @PathParam("dag")                         String  dagName,
            final @PathParam("bus")                         Scope   scope) throws Exception {
        return privateDagBusPut(dagName, scope, null);
    }

    /** <h4>trigger cleaning of one sub-dag</h4>
     * 
     * @param dagName
     * @param scope
     * @param subDagKey
     * @return
     * @throws Exception
     */
    @PUT
    @Path(__DAGS + "{dag}" + __BUSES + "{bus}" + __SUBS + "{sub}")
    @Produces(APPLICATION_JSON)
    public Response dagBusSub_PUT(
            final @PathParam("dag")                         String  dagName,
            final @PathParam("bus")                         Scope   scope,
            final @PathParam("sub")                         String  subDagKey) throws Exception {
        return privateDagBusPut(dagName, scope, subDagKey);
    }
    /** <h4>trigger cleaning of one sub-dag</h4>
     * 
     * @deprecated use PUT instead
     * 
     * @param dagName
     * @param scope
     * @param subDagKey
     * @return
     * @throws Exception
     */
    @PATCH
    @Deprecated
    @Path(__DAGS + "{dag}" + __BUSES + "{bus}" + __SUBS + "{sub}")
    @Produces(APPLICATION_JSON)
    public Response dagBusSub_PATCH(
            final @PathParam("dag")                         String  dagName,
            final @PathParam("bus")                         Scope   scope,
            final @PathParam("sub")                         String  subDagKey) throws Exception {
        return privateDagBusPut(dagName, scope, subDagKey);
    }

    private Response privateDagBusPut(final String dagName, final Scope scope, final String subDagKey) {
        final DagOnSteroids dagOnSteroids = _getDagOnSteroids(dagName);
        try {
            final DagCleaner dagCleaner = dagOnSteroids.getDagCleaner(scope);
            dagCleaner.cleanSubDag(subDagKey);
        } catch (Exception e) {
            throw new WebApplicationException(Response.status(Status.INTERNAL_SERVER_ERROR).entity(error(e.getMessage())).build());            
        }
        return Response.created(URI.create("journal")).status(Status.ACCEPTED).build();
    }

    //
    /////////////////////////////////////////////////////

    /** <h4>get all nodes of one dag</h4>
     *  
     * @param dagName
     * @return
     * @throws Exception
     */
    @GET
    @Path(__DAGS + "{dag}" + __NODES)
    @Produces(APPLICATION_JSON)
    public Collection<Node> dagNodes_GET_JSON(final @PathParam("dag") String dagName) throws Exception {
        final Dag dag = _getDag(dagName);
        return dag.getNodes();
    }

    /** <h4>get one node of one dag</h4>
     * 
     * @param dagName
     * @param node
     * @return
     * @throws Exception
     */
    @GET
    @Path(__DAGS + "{dag}" + __NODES + "{node}")
    @Produces(APPLICATION_JSON)
    public Node dagNode_GET_JSON(final @PathParam("dag") String dagName, 
                                 final @PathParam("node") String node) throws Exception {
        return _getDagAndNodeByName(dagName, node).node;
    }

    @GET
    @Path(__DAGS + "{dag}" + __NODES + "{node}" + __VERSIONS + "current")
    @Produces(APPLICATION_JSON)
    public String dagNodeCurrentVersion_GET_JSON(final @PathParam("dag") String dagName, 
                                               final @PathParam("node") String node) throws Exception {
        DagOnSteroidsAndNode _getDagAndNodeByName = _getDagOnSteroidsAndNodeByName(dagName, node);
        NodeNameVersion current = _getDagAndNodeByName.dag.getCurrent(_getDagAndNodeByName.node, false);
        return current.version;
    }

    @GET
    @Path(__DAGS + "{dag}" + __NODES + "{node}" + __VERSIONS + "next")
    @Produces(APPLICATION_JSON)
    public String dagNodeNextVersions_GET_JSON(final @PathParam("dag") String dagName, 
                                             final @PathParam("node") String node,
                                             final @QueryParam("allowSnapshots") @DefaultValue("true") Boolean allowSnapshots) throws Exception {
        DagOnSteroidsAndNode _getDagAndNodeByName = _getDagOnSteroidsAndNodeByName(dagName, node);
        NodeNameVersion current = _getDagAndNodeByName.dag.getNext(_getDagAndNodeByName.node, false, allowSnapshots, false);
        return current.version;
    }

    /** <h4>retrieve the state of one node</h4>
     *  
     * @param dagName
     * @param scope
     * @param nodeName
     * @return
     * @throws Exception
     */
    @GET
    @Path(__DAGS + "{dag}" + __BUSES + "{bus}" + __NODES + "{node}" + __STATES)
    @Produces(APPLICATION_JSON)
    public NodeState dagBusNodeState_GET_JSONfinal (final @PathParam("dag")  String dagName, 
                                                    final @PathParam("bus")  Scope  scope, 
                                                    final @PathParam("node") String nodeName) throws Exception {
        final DagOnSteroids dagOnSteroids = _getDagOnSteroids(dagName);
        final Node node = dagNode_GET_JSON(dagName, nodeName);
        return dagOnSteroids.getDagCleaner(scope).getNodeCleaner(node).getState();
    }

    /** <h4>trigger cleaning of one node</h4>
     *  
     * @param dagName
     * @param scope
     * @param nodeName
     * @param skipRelease
     * @return
     * @throws Exception
     */
    @PATCH
    @Deprecated
    @Path(__DAGS + "{dag}" + __BUSES + "{bus}" + __NODES + "{node}")
    @Produces(APPLICATION_JSON)
    public Response dagBusNode_PATCH(
            final @PathParam("dag")                                 String  dagName, 
            final @PathParam("bus")                                 Scope   scope, 
            final @PathParam("node")                                String  nodeName,
            final @PathParam("skipRelease") @DefaultValue("false")  Boolean skipRelease) throws Exception {
        return dagBusNode_PUT(dagName, scope, nodeName, skipRelease);
    }
    /** <h4>trigger cleaning of one node</h4>
     *  
     * @deprecated use PUT instead
     * 
     * @param dagName
     * @param scope
     * @param nodeName
     * @param skipRelease
     * @return
     * @throws Exception
     */
    @PUT
    @Path(__DAGS + "{dag}" + __BUSES + "{bus}" + __NODES + "{node}")
    @Produces(APPLICATION_JSON)
    public Response dagBusNode_PUT(
            final @PathParam("dag")                                 String  dagName, 
            final @PathParam("bus")                                 Scope   scope, 
            final @PathParam("node")                                String  nodeName,
            final @PathParam("skipRelease") @DefaultValue("false")  Boolean skipRelease) throws Exception {
        final DagAndNode        dagAndNode      = _getDagAndNodeByName(dagName, nodeName);
        final DagOnSteroids     dagOnSteroids   = _getDagOnSteroids(dagName);
        final DagCleaner        dagCleaner      = dagOnSteroids.getDagCleaner(scope); 
        final Bus<Node, Scope>  bus             = dagCleaner.getBus();
        
        final NodeCleaner.NodeCleaningResult cleanResult = dagOnSteroids.clean(scope, dagAndNode.node, skipRelease);
        if (cleanResult!= null && cleanResult.asBoolean()) {
            _sendBusEvent(bus, CLEAN_ORDER_OK, dagAndNode.node);
            return Response.accepted().build();
        } else {
            _sendBusEvent(bus, CLEAN_ORDER_ERROR, dagAndNode.node);
            return Response.serverError().entity(cleanResult).build();
        }
    }

    /** <h4>force state of one node</h4>
     *  
     * @param dagName
     * @param scope
     * @param nodeName
     * @param newState
     * @return
     * @throws Exception
     */
    @PATCH
    @Deprecated
    @Path(__DAGS + "{dag}" + __BUSES + "{bus}" + __NODES + "{node}" + __STATES + "{state}")
    @Produces(APPLICATION_JSON)
    public Response dagBusNodeState_PATCH_JSON(
            final @PathParam("dag")      String     dagName,
            final @PathParam("bus")      Scope      scope, 
            final @PathParam("node")     String     nodeName, 
            final @PathParam("state")    NodeState  newState) throws Exception {
        return dagBusNodeState_PUT_JSON(dagName, scope, nodeName, newState);
    }
    /** <h4>force state of one node</h4>
     *  
     * @deprecated use PUT instead
     * 
     * @param dagName
     * @param scope
     * @param nodeName
     * @param newState
     * @return
     * @throws Exception
     */
    @PUT
    @Path(__DAGS + "{dag}" + __BUSES + "{bus}" + __NODES + "{node}" + __STATES + "{state}")
    @Produces(APPLICATION_JSON)
    public Response dagBusNodeState_PUT_JSON(
            final @PathParam("dag")      String     dagkey,
            final @PathParam("bus")      Scope      scope, 
            final @PathParam("node")     String     nodename, 
            final @PathParam("state")    NodeState  newState) throws Exception {
        try {
            final DagOnSteroids dagOnSteroids   = _getDagOnSteroids(dagkey);
            final DagCleaner        dagCleaner      = dagOnSteroids.getDagCleaner(scope);
            final Bus<Node, Scope>  bus             = dagCleaner.getBus();
            final Node              node            = dagNode_GET_JSON(dagkey, nodename);
            final NodeCleaner       nodeCleaner   = dagCleaner.getNodeCleaner(node);
            final NodeState         currentState  = nodeCleaner.getState();
            if ((currentState == null && newState == null) || 
                (currentState != null && newState != null && newState.equals(currentState))) {
                // 
                // http://serverfault.com/a/168348
                // The 304 response MUST NOT contain a message-body, and thus is always terminated by the first empty line after the header fields. 
                // 
                return Response.status(Response.Status.NOT_MODIFIED).build();
            }
            nodeCleaner.setState(newState);
            _sendBusEvent(bus, STATE_CHANGE_HACK, node);
            return Response.status(Response.Status.OK).build();
        } catch (Exception e) {
            LOG.error("[dagr "+SIMPLENAME+"] exception /{}/ logged and re-thrown", e.getMessage());
            throw e;
        } finally {
        }
    }

    // get one dag node version rules 
    @GET
    @Path(__DAGS + "{dag}" + _RULES)
    @Produces(APPLICATION_XML)
    public Response dagBusNodeRules_GET_JSONfinal(final @PathParam("dag")  String dagkey, @QueryParam("withArtifact") @DefaultValue("false") boolean withArtifact) throws Exception {
        final Dag           dag    = _getDag(dagkey);
        final StringBuilder entity = new StringBuilder();
        entity.append(
                "<ruleset comparisonMethod=\"maven\"\n" + 
                "         xmlns=\"http://mojo.codehaus.org/versions-maven-plugin/rule/2.0.0\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\"\n" + 
                "         xsi:schemaLocation=\"http://mojo.codehaus.org/versions-maven-plugin/rule/2.0.0 http://mojo.codehaus.org/versions-maven-plugin/xsd/rule-2.0.0.xsd\">\n" +
                "  <rules>\n" 
        );
        for (Node node : dag.getNodes()) {
            final Gucrid gucrid = Gucrid.create(node.getValue().getGucrid());

            if (gucrid == null || !gucrid.getGroupId().isPresent() || !gucrid.getArtifactId().isPresent() || !gucrid.getSemanticVersion().isPresent()) {
                entity.append(
                        "    <!-- cannot parse groupId or artifactId or semantic version from gucrid=["+node.getValue().getGucrid()+"] -->" 
                );
            } else {
                StringBuilder forbiddenVersions = new StringBuilder();
                int major = gucrid.getSemanticVersion().get().getMajor();
                for (int i = major+1; i<10; i++) {
                    forbiddenVersions.append(String.valueOf(i));
                }
                entity.append("    <rule groupId=\""+gucrid.getGroupId().get()+"\"");
                if (withArtifact) {
                    entity.append(" artifactId=\""+gucrid.getArtifactId().get()+"\" ");
                }
                entity.append(" comparisonMethod=\"maven\">\n" + 
                "      <ignoreVersions>\n" + 
                "        <ignoreVersion type=\"regex\">^[" + forbiddenVersions.toString() + "]\\..*</ignoreVersion>\n" + 
                "      </ignoreVersions>\n" + 
                "    </rule>\n" +
                "    "); 
            }
        }
        entity.append(
                "  </rules>\n" +
                "</ruleset>"
        );
        return Response.status(Response.Status.OK).entity(entity.toString()).build();
    }

    /** 
     * <h4>send an event about the cleaning process (i.e. on a specific bus) of one node of one dag</h4>
     * 
     * @return
     */
    @PUT
    @Path(__DAGS + "{dag}" + __BUSES + "{bus}" + __NODES + "{node}" + __EVENTS + "{event}")
    @Consumes(TEXT_PLAIN)
    @Produces(APPLICATION_JSON)
    public Response dagBusNodeEvent_PUT(   final                           String        cleanerId, 
                                           final @PathParam("dag")         String        dagName,
                                           final @PathParam("bus")         Scope         scope, 
                                           final @PathParam("node")        String        nodeName, 
                                           final @PathParam("event")       BusEvent.Type eventType)  {
        final DagOnSteroids     dagOnSteroids   = _getDagOnSteroids(dagName);
        final DagCleaner        dagCleaner      = dagOnSteroids.getDagCleaner(scope);
        final Bus<Node, Scope>  bus             = dagCleaner.getBus();
        final DagAndNode        dagAndNode      = _getDagAndNodeByName(dagName, nodeName);
        final NodeCleaner       nodeCleaner     = dagCleaner.getNodeCleaner(dagAndNode.node);
        
        if (eventType.equals(BusEvent.Type.CLEAN_STARTED)) {
            if (cleanerId != null && cleanerId.length()>0) {
                nodeCleaner.setId(cleanerId);
            }
        } else if (eventType.equals(BusEvent.Type.CLEAN_OK      ) || 
                   eventType.equals(BusEvent.Type.CLEAN_ERROR   ) || 
                   eventType.equals(BusEvent.Type.CLEAN_ABORTED )) {
            nodeCleaner.setId(null);
        }
            
        _sendBusEvent(bus, eventType, dagAndNode.node);
        
        return Response.status(Response.Status.OK).build();
    }

    @PATCH
    @Deprecated
    @Path(__DAGS + "{dag}" + __BUSES + "{bus}" + __NODES + "{node}" + __EVENTS + "{event}")
    @Consumes(TEXT_PLAIN)
    @Produces(APPLICATION_JSON)
    public Response dagBusNodeEvent_PATCH( final                           String        cleanerId, 
                                           final @PathParam("dag")         String        dagName,
                                           final @PathParam("bus")         Scope         scope, 
                                           final @PathParam("node")        String        nodeName, 
                                           final @PathParam("event")       BusEvent.Type eventType)  {
        return dagBusNodeEvent_PUT( cleanerId, 
                dagName,
                scope, 
                nodeName, 
                eventType);
    }
    //////////////////////////////////////////////////////////////////////////////////

    //////////// VIEWs
    // utility class
    static public class Model {
        final List<DagInfo> infos;
        final String        warning;
        final Exception     exception;
        public Model(final List<DagInfo> infos, final String warning, final Exception exception) {
            super();
            this.infos     = infos == null ? Collections.emptyList() : infos;
            this.warning   = warning;
            this.exception = exception;
        }
        public List<DagInfo> getInfos() {
            return infos;
        }
        public String getWarning() {
            return warning;
        }
        public Exception getException() {
            return exception;
        }
    }
    // go to dag-list JSP 
    @GET
    @Path(_VIEWS + __DAGS)
    @Produces(TEXT_HTML)
    public Viewable dagList_GET_HTML() throws IOException {
        return new Viewable("/WEB-INF/dagr/dag-list");
    }
    // go to one dag JSP
    @GET
    @Path(_VIEWS + __DAGS + "{dag}")
    @Produces(TEXT_HTML)
    public Viewable dag_GET_HTML(final @PathParam("dag") String dagName) {
        return dagSub_GET_HTML(dagName, null);
    }
    // go to one subdag JSP
    @GET
    @Path(_VIEWS + __DAGS + "{dag}" + __SUBS + "{sub}")
    @Produces(TEXT_HTML)
    public Viewable dagSub_GET_HTML(final @PathParam("dag") String dagName, final @PathParam("sub") String subkey) {
        DagInfo info;
        try {
            info = _getDagInfo(dagName, subkey);
            final String warning; {
                if (subkey != null && !subkey.isEmpty() && info.getSubDagId() == null) {
                    warning = "sub \"" + subkey + "\" not found in dag \"" + dagName + "\"";
                } else {
                    warning = null;
                }
            }
            return new Viewable("/WEB-INF/dagr/dag", new Model(Arrays.asList(info), warning, null));
        } catch (NotFoundException e) {
            return new Viewable("/WEB-INF/dagr/dag", new Model(null, "dag \"" + dagName + "\" not found", e));
        }
    }
    //////////// end of VIEWs
    
    private Dag _getDag(final String dagkey) throws NotFoundException {
        String[] both = DagOnSteroids.parseName(dagkey);
        final Dag dag = this.dags.getDAG(both[0]);
        if (dag == null) {
            throw new NotFoundException(Response.status(Status.NOT_FOUND).entity(error("dag ["+dagkey+"] not found").valueOf(MediaType.APPLICATION_JSON_TYPE)).build());
        }
        return dag;
    }

    private DagInfo _getDagInfo(final String dagkey, final String subDagId) throws NotFoundException {
        final DagOnSteroids dagOnSteroids  = _getDagOnSteroids(dagkey);
        dagOnSteroids.setSubDagId(subDagId);
        final DagInfo info = DagInfo.create(dagOnSteroids);
        if (info == null) {
            throw new NotFoundException(Response.status(Status.NOT_FOUND).entity(error("info for dag ["+dagkey+"] not found").valueOf(MediaType.APPLICATION_JSON_TYPE)).build());
        }
        return info;
    }

    private DagOnSteroids _getDagOnSteroids(final String dagkey) throws NotFoundException {
        final DagOnSteroids dagOnSteroids  = this.dags.getDagOnSteroids(dagkey);
        if (dagOnSteroids == null) {
            throw new NotFoundException(Response.status(Status.NOT_FOUND).entity(error("dag ["+dagkey+"] not found").valueOf(MediaType.APPLICATION_JSON_TYPE)).build());
        }
        return dagOnSteroids;
    }

    static class DagAndNode {
        final Dag  dag;
        final Node node;
        static DagAndNode dagAndNode(final Dag dag,final  Node node) {
            return new DagAndNode(dag, node);
        }
        private DagAndNode(final Dag dag, final Node node) {
            this.dag  = dag;
            this.node = node;
        }
    }

    static class DagOnSteroidsAndNode {
        final DagOnSteroids  dag;
        final Node node;
        static DagOnSteroidsAndNode dagOnSteroidsAndNode(final DagOnSteroids dag,final  Node node) {
            return new DagOnSteroidsAndNode(dag, node);
        }
        private DagOnSteroidsAndNode(final DagOnSteroids dag, final Node node) {
            this.dag  = dag;
            this.node = node;
        }
    }

    private DagAndNode _getDagAndNodeByName(final String dagkey, final String nodeName) throws NotFoundException {
        final Dag        dag   = _getDag(dagkey);
        final Collection<Node> nodes = dag.getNodesFromName(nodeName);
        if (nodes.size() == 0) {
            throw new NotFoundException(Response.status(Status.NOT_FOUND).entity(error("dag ["+dagkey+"] node ["+nodeName+"] not found").valueOf(MediaType.APPLICATION_JSON_TYPE)).build());
        } else if (nodes.size() > 1) {
            throw new WebApplicationException(Response.status(Status.NOT_IMPLEMENTED).entity("application error. Found multiple nodes with the same name ["+nodeName+"] in dag ["+ dagkey+ "]. This is not supported").build());
        }
        return dagAndNode(dag, nodes.stream().findFirst().get());
    }

    private DagOnSteroidsAndNode _getDagOnSteroidsAndNodeByName(final String dagkey, final String nodeName) throws NotFoundException {
        final DagOnSteroids    dag   = _getDagOnSteroids(dagkey);
        final Collection<Node> nodes = dag.getDag().getNodesFromName(nodeName);
        if (nodes.size() == 0) {
            throw new NotFoundException(Response.status(Status.NOT_FOUND).entity(error("dag ["+dagkey+"] node ["+nodeName+"] not found").valueOf(MediaType.APPLICATION_JSON_TYPE)).build());
        } else if (nodes.size() > 1) {
            throw new WebApplicationException(Response.status(Status.NOT_IMPLEMENTED).entity("application error. Found multiple nodes with the same name ["+nodeName+"] in dag ["+ dagkey+ "]. This is not supported").build());
        }
        return dagOnSteroidsAndNode(dag, nodes.stream().findFirst().get());
    }

    // helper to send an event on the bus
    private void _sendBusEvent(final Bus<Node, Scope> bus, final BusEvent.Type eventType, final Node node) {
        if (bus == null) {
            throw new RuntimeException("no bus");
        }
        List<Node> listOfNodes;
        if (node == null) {
            listOfNodes = Collections.emptyList();
        } else {
            listOfNodes = Collections.singletonList(node);
        }
        LOG.debug("[dagr {}] sending event {}, node='{}' to the bus", SIMPLENAME, eventType, node == null ? "∅" : node.getName());
        
        // balance dans le bus
        bus.send(eventType, listOfNodes, request.getRequestURI().toString());
    }

    @JsonIgnoreProperties
    static public class Error {
        @JsonProperty
        final String error;

        static public Error error(final String error) {
            return new Error(error);
        }

        public Error(String error) {
            this.error = error;
        }

        public Object valueOf(MediaType mediaType) {
            if (mediaType.equals(MediaType.TEXT_PLAIN_TYPE)) {
                return error;
            } else if (mediaType.equals(MediaType.APPLICATION_JSON_TYPE)) {
                return this;
            } else {
                return error;
            }
        }
    }

    @JsonIgnoreProperties
    static public class Message {
        @JsonProperty
        final String message;

        static public Message message(final String message) {
            return new Message(message);
        }

        public Message(String message) {
            this.message = message;
        }
        
    }
    
    static <T> Map.Entry<String, T> tuple(final String a, final T b) {
        return new AbstractMap.SimpleImmutableEntry<>(a, b);
    }
}
