package net.aequologica.neo.dagr.jaxrs;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.util.AbstractMap;
import java.util.Base64;
import java.util.Collection;
import java.util.EnumMap;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;

import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;

import com.google.common.collect.Lists;
import net.aequologica.neo.dagr.DagOnSteroids;
import net.aequologica.neo.dagr.Scope;
import net.aequologica.neo.dagr.bus.Bus;
import net.aequologica.neo.dagr.bus.BusEvent;
import net.aequologica.neo.dagr.config.DagrConfig;
import net.aequologica.neo.dagr.model.Dag.Node;
import net.aequologica.neo.geppaequo.config.ConfigRegistry;

import io.reactivex.disposables.Disposable;

enum GaranceWrapper {
    THIS;

    final Map<String, EnumMap<Scope, Disposable>> garanceSubscriptions = new HashMap<>();
    
    final private Base64.Encoder base64encoder = Base64.getEncoder(); 
    final private String         garanceEndPoint; 
    final private String         garanceApi; 

    GaranceWrapper() {
        DagrConfig dagrConfig = ConfigRegistry.CONFIG_REGISTRY.getConfig(DagrConfig.class);

        if (dagrConfig.getGaranceEndPoint().contains("localhost")) {
            String applicationUrl = System.getenv("HC_APPLICATION_URL");
            if (applicationUrl != null && !applicationUrl.isEmpty()) {
                garanceEndPoint = applicationUrl;
            } else {
                garanceEndPoint = dagrConfig.getGaranceEndPoint();
            }
        } else {
            garanceEndPoint = dagrConfig.getGaranceEndPoint();
        }
        
        garanceApi = "/api/garance/v1";
    }

    int put2garance(final String key, final Long value) throws UnsupportedEncodingException {
        if (key == null || value == null) {
            return -1;
        }
        CloseableHttpClient httpClient = HttpClientBuilder.create().build();
        HttpPost postRequest = new HttpPost(garanceEndPoint
                                           + garanceApi
                                           + "/series/" 
                                           + base64encoder.encodeToString(key.getBytes("UTF-8")) 
                                           + "/values/" 
                                           + String.valueOf(value.doubleValue()));
     
        try (CloseableHttpResponse httpResponse = httpClient.execute(postRequest)) {
            httpResponse.getEntity();
            int statusCode = httpResponse.getStatusLine().getStatusCode();
            return statusCode;
        } catch (IOException e) {
            e.printStackTrace();
        }
        return -1;
    }
    
    Collection<Map.Entry<String, String>> _initGaranceSubscription(
            final DagOnSteroids                  dagOnSteroids, 
            final Entry<Scope, Bus<Node, Scope>> busEntry) {
        final Collection<Map.Entry<String, String>> exceptions = Lists.newArrayList();
        try {
            // register an observer on the bus that will save build durations to garance key/value store
            Disposable subscription = busEntry.getValue().toObservable()
                                           .filter   ( event -> event.getType().equals(BusEvent.Type.CLEAN_OK))
                                           .map      ( event -> event.get())
                                           .subscribe( node  -> {
                                                if (busEntry                                            != null &&
                                                    busEntry.getKey()                                   != null &&
                                                    node                                                != null &&
                                                    node.getId()                                        != null &&
                                                    node.getDag()                                       != null &&
                                                    node.getDag().getName()                             != null ) {

                                                    String key = node.getDag().getName() + "::" + busEntry.getKey() + "::" + node.getName();
                                                    
                                                    if (dagOnSteroids                                                                        != null && 
                                                        dagOnSteroids.getDagCleaner(busEntry.getKey())                                       != null && 
                                                        dagOnSteroids.getDagCleaner(busEntry.getKey()).getNodeCleaner(node)                  != null) {
                                                        Long duration = dagOnSteroids.getDagCleaner(busEntry.getKey()).getNodeCleaner(node).getDuration();
                                                        put2garance(key, duration);
                                                    }

                                                }
                                           });
            final String dagName = dagOnSteroids.getDag().getName();
            EnumMap<Scope, Disposable> dagSubscriptions = this.garanceSubscriptions.get(dagName);
            if (dagSubscriptions == null) {
                dagSubscriptions = new EnumMap<>(Scope.class);
                this.garanceSubscriptions.put(dagName, dagSubscriptions);
            }
            dagSubscriptions.put(busEntry.getKey(), subscription);
        } catch (Exception e) {
            exceptions.add(tuple(e.getClass().getSimpleName(), e.getMessage()));
        }
        return exceptions;
    }
    
    void _closeGaranceSubscriptions() {
        for (EnumMap<Scope, Disposable> dagSubscriptions : this.garanceSubscriptions.values()) {
            for (Disposable aSubscription : dagSubscriptions.values()) {
                aSubscription.dispose();
            }
        }
        this.garanceSubscriptions.clear();
    }

    void _closeGaranceSubscription(final String dagName) {
        EnumMap<Scope, Disposable> dagSubscriptions = this.garanceSubscriptions.get(dagName);
        if (dagSubscriptions != null) {
            for (Disposable aSubscription : dagSubscriptions.values()) {
                if (aSubscription != null) {
                    aSubscription.dispose();
                    this.garanceSubscriptions.remove(dagName);
                }
            }
        }
    }

    static <T> Map.Entry<String, T> tuple(final String a, final T b) {
        return new AbstractMap.SimpleImmutableEntry<>(a, b);
    }
}
