package net.aequologica.neo.dagr.websocket;

import java.io.IOException;
import java.util.Collection;

import org.atmosphere.config.service.Disconnect;
import org.atmosphere.config.service.ManagedService;
import org.atmosphere.config.service.Message;
import org.atmosphere.config.service.PathParam;
import org.atmosphere.config.service.Ready;
import org.atmosphere.cpr.AtmosphereResource;
import org.atmosphere.cpr.AtmosphereResourceEvent;
import org.atmosphere.cpr.Broadcaster;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.aequologica.neo.dagr.DagOnSteroids;
import net.aequologica.neo.dagr.DagsSingleton;
import net.aequologica.neo.dagr.DagsSingleton.Dags;
import net.aequologica.neo.dagr.bus.Bus;
import net.aequologica.neo.dagr.bus.Bus.Scope;
import net.aequologica.neo.dagr.model.Dag.Node;
import rx.Subscription;

@ManagedService(path = "/patata/{dagName}/patati/{scope}") 
public final class DagSocket {
    private static final Logger LOG = LoggerFactory.getLogger(DagSocket.class);
    
    private Dags dags = DagsSingleton.INSTANCE.dags;

    @PathParam("dagName")
    private String dagName;
    
    @PathParam("scope")
    private String scopeAsAString;
    private Scope scope;
    
    private DagOnSteroids dagOnSteroids;
    private Subscription subscription;
    private final Object lock = new Object();
    
    @Ready
    public final void onReady(final AtmosphereResource r){
        if (dagName == null) {
            throw new RuntimeException("@PathParam(\"dagName\") is " + dagName);
        }

        if (scopeAsAString == null) {
            throw new RuntimeException("@PathParam(\"scope\") is " + scopeAsAString);
        }

        String[] both = DagOnSteroids.parseName(dagName);
        dagOnSteroids = this.dags.getDagOnSteroids(both[0]);
        if (dagOnSteroids == null) {
            throw new RuntimeException("dag /"+dagName+"/ not found");
        }
        scope = Scope.valueOf(scopeAsAString);

        synchronized (lock) {
            if (subscription == null) {
                final Bus<Node> bus = dags.getBus(dagName, scope);
                if (bus == null) {
                    LOG.warn("[websocket {}] no bus for dag {} {}", scopeAsAString, dagName, dagOnSteroids);
                } else {
                    subscription = bus.toObservable()
                            .map(nodeevent -> nodeevent.get())
                            .subscribe(node -> broadcast(r.getBroadcaster(), node));
                    
                    LOG.debug("[websocket {}] [broadcaster:#{}] subscribed - browser {} connected to dag {}", scopeAsAString, r.getBroadcaster().getID(), r.uuid(), dagOnSteroids );
                }
            }
        }
    }

    @Disconnect
    public final void onDisconnect(final AtmosphereResourceEvent event){
        Collection<AtmosphereResource> atmosphereResources = event.getResource().getBroadcaster().getAtmosphereResources();
        if (atmosphereResources.size() < 2 && subscription != null) {
            synchronized (subscription) {
                subscription.unsubscribe();
                subscription = null;
            }
            LOG.debug("[websocket {}] [broadcaster:#{}] unsubscribed !", scopeAsAString, event.getResource().getBroadcaster().getID());
        }
        
        if (event.isCancelled()) {
            LOG.info("[websocket {}] Client {} cancelled the connection", scopeAsAString, event.getResource().uuid());
        } else if(event.isClosedByClient()) {
            LOG.info("[websocket {}] Client {} closed the connection", scopeAsAString, event.getResource().uuid());
        }
    }

    @Message(encoders = {DagSocketMessageEncoderDecoder.class}, decoders = {DagSocketMessageEncoderDecoder.class})
    public final DagSocketMessage onMessage(final DagSocketMessage message) throws IOException{
        LOG.info("[websocket {}] in dag /{}/, node /{}/ has state /{}/ ", scopeAsAString, dagName, message.getNodeCleaner().getNode().getName(), message.getNodeCleaner().getState());
        dagOnSteroids.getDagCleaner(scope).getNodeCleaner(message.getNodeCleaner().getNode()).setState(message.getNodeCleaner().getState());
        return message;
    }
    
    private Node broadcast(Broadcaster broadcaster, Node node) {
        try {
            LOG.info("[websocket {}] received node '{}' from the bus", scopeAsAString, node.getName());
            
            DagSocketMessage dagSocketMessage = new DagSocketMessage();
            dagSocketMessage.setNodeCleaner(dagOnSteroids.getDagCleaner(scope).getNodeCleaner(node));
            DagSocketMessageEncoderDecoder codec = new DagSocketMessageEncoderDecoder();
            String encoded = codec.encode(dagSocketMessage);
            LOG.debug("[websocket {}] [broadcaster:#{}] about to broadcast: |{}|", scopeAsAString, broadcaster.hashCode(), encoded);
            broadcaster.broadcast(encoded);

            return node;
        } catch (Exception e) {
            LOG.error("[websocket {}] exception /{}/ logged and re-thrown", scopeAsAString, e.getMessage());
            throw e;
        } finally {
        }
    }

}