package net.aequologica.neo.dagr.jaxrs.travis;

import static com.fasterxml.jackson.annotation.JsonInclude.Include.NON_NULL;

import java.net.URI;
import java.net.URLEncoder;

import org.apache.http.Header;
import org.apache.http.HttpEntity;
import org.apache.http.StatusLine;
import org.apache.http.client.fluent.Request;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.message.BasicHeader;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;

import net.aequologica.neo.dagr.DagOnSteroids.NodeCleaner;
import net.aequologica.neo.dagr.bus.Bus.Scope;
import net.aequologica.neo.dagr.jaxrs.travis.NodeCleanerTravis.Body;
import net.aequologica.neo.dagr.model.Dag.Node;
import net.aequologica.neo.geppaequo.config.ConfigRegistry;
import net.aequologica.neo.geppaequo.config.geppaequo.GeppaequoConfig;
import pl.touk.throwing.ThrowingBiFunction;

/*
cf. https://docs.travis-ci.com/user/triggering-builds

e.g.

POST
https://api.travis-ci.org/repo/repo_owner%2Frepo_name/requests

HEADERS
Authorization:token "<your travis token>" 
Content-Type:application/json
Accept:application/json
Travis-API-Version:3

BODY
{"request": {"branch":"develop"}}

*/
public class NodeCleanerTravis {
    
    private final static Logger log = LoggerFactory.getLogger(NodeCleanerTravis.class);
    
    final URI                       uri;
    final ThrowingBiFunction<Scope, Node, NodeCleaner.NodeCleaningResult, NodeCleaner.NodeCleanerException> cleaningFunction;
    
    public NodeCleanerTravis(final String url, final String token) {
        this.uri     = URI.create(url);
        this.cleaningFunction = (scope, node) -> {
            final Header[] headers = new Header[] {
                    new BasicHeader("Authorization"     , "token \""+token+"\""),
                    new BasicHeader("Content-Type"      , "application/json"),
                    new BasicHeader("Accept"            , "application/json"),
                    new BasicHeader("Travis-API-Version", "3"),
            };
            
            final String path = calcPath(node);
            
            String branch = "master";
            if ( node.getValue().getBranch() != null && 
                !node.getValue().getBranch().isEmpty()) {
                branch = node.getValue().getBranch();
            }
            
            String finalURL = this.uri + path;
            try {
                Request request = Request
                        .Post(finalURL)
                        .setHeaders(headers)
                        .bodyString(Body.string(Body.Request.request(branch)), ContentType.APPLICATION_JSON);
                
                // usual proxy weariness 
                {
                    GeppaequoConfig config = ConfigRegistry.CONFIG_REGISTRY.getConfig(GeppaequoConfig.class);
                    String proxyIfNeeded = config.getProxyIfNeeded(this.uri.getHost());
                    if (proxyIfNeeded != null) {
                        request = request.viaProxy(proxyIfNeeded);
                    }
                }
                
                org.apache.http.client.fluent.Response response = request.execute();
                StatusLine statusLine = response.returnResponse().getStatusLine();
                log.debug("POST {} => status: {}", this.uri + path, statusLine);
                return NodeCleaner.NodeCleaningResult.from(scope, statusLine.getStatusCode(), statusLine.getReasonPhrase()).source(finalURL);
                
            } catch (Throwable t) {
                throw new NodeCleaner.NodeCleanerException(finalURL, t);
            }
            
        };
    }

    private String calcPath(Node node) {
        if (node                     == null || 
            node.getValue()          == null || 
            node.getValue().getScm() == null || 
            node.getValue().getScm().isEmpty()) {
            throw new RuntimeException("no scm");
        }
        
        String slug = getSlug(node.getValue().getScm());
        if (slug == null || 
            slug.isEmpty()) {
            throw new RuntimeException("no slug");
        }
        return "/repo/"+slug+"/requests";
    }

    public ThrowingBiFunction<Scope, Node,NodeCleaner.NodeCleaningResult, NodeCleaner.NodeCleanerException> getCleaningFunction() {
        return this.cleaningFunction;
    }
    
    static String getSlug(String scm) {
        try {
            URI uri = URI.create(scm);
            String path = uri.getPath();
            path = path.replaceAll("^/(.*)\\.git", "$1");
            path = URLEncoder.encode(path, "UTF8");
            return path;
        } catch (Exception e) {
            log.debug("exception trying to get slug from {}: {}", scm, e.getMessage());
            return null;
        }
    }
    
    @JsonIgnoreProperties
    static class Body {
        final static ObjectMapper mapper;

        static {
            mapper = new ObjectMapper();
            mapper.setSerializationInclusion(NON_NULL);
        }
        
        @JsonProperty
        final Request request;

        Body(final Request request) {
            this.request = request;
        }
        
        @Override
        public String toString() {
            try {
                return mapper.writeValueAsString(this);
            } catch (JsonProcessingException e) {
                log.error(e.getMessage());
            }
            return null;
        }

        static Body body(final Request request) {
            return new Body(request);
        }

        static String string(final Request request) {
            return body(request).toString();
        }
        
        static HttpEntity entity(final Request request) {
            return new StringEntity(string(request), ContentType.APPLICATION_JSON);
        }

        @JsonIgnoreProperties
        static class Request {
            @JsonProperty
            final String branch;

            Request(final String branch) {
                this.branch = branch;
            }

            static Request request(final String branch) {
                return new Request(branch);
            }
        }
    }

}
