package net.aequologica.neo.dagr.jaxrs;

import static javax.ws.rs.core.MediaType.APPLICATION_JSON;
import static javax.ws.rs.core.MediaType.TEXT_HTML;
import static javax.ws.rs.core.MediaType.TEXT_PLAIN;
import static net.aequologica.neo.dagr.bus.BusEvent.Type.CLEAN_ORDER_ERROR;
import static net.aequologica.neo.dagr.bus.BusEvent.Type.CLEAN_ORDER_OK;
import static net.aequologica.neo.dagr.bus.BusEvent.Type.STATE_CHANGE_HACK;
import static net.aequologica.neo.dagr.jaxrs.ResourceDags.DagAndNode.dagAndNode;
import static net.aequologica.neo.dagr.jaxrs.ResourceDags.Error.error;
import static net.aequologica.neo.dagr.jaxrs.ResourceDags.Message.message;
import static net.aequologica.neo.geppaequo.config.ConfigRegistry.CONFIG_REGISTRY;

import java.io.IOException;
import java.net.URI;
import java.security.InvalidParameterException;
import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.EnumMap;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Observable;
import java.util.Observer;
import java.util.Set;
import java.util.SortedSet;
import java.util.stream.Collectors;

import javax.annotation.PostConstruct;
import javax.inject.Inject;
import javax.inject.Singleton;
import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.Consumes;
import javax.ws.rs.DELETE;
import javax.ws.rs.DefaultValue;
import javax.ws.rs.GET;
import javax.ws.rs.MatrixParam;
import javax.ws.rs.NotFoundException;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.WebApplicationException;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.Response.Status;
import javax.ws.rs.core.UriBuilder;

import org.glassfish.jersey.server.mvc.Viewable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.Function;
import com.google.common.base.Splitter;
import com.google.common.collect.Iterators;
import com.google.common.collect.Lists;

import net.aequologica.neo.dagr.DagOnSteroids;
import net.aequologica.neo.dagr.DagOnSteroids.DagCleaner;
import net.aequologica.neo.dagr.DagOnSteroids.NodeCleaner;
import net.aequologica.neo.dagr.DagOnSteroids.NodeCleaner.NodeState;
import net.aequologica.neo.dagr.DagsSingleton;
import net.aequologica.neo.dagr.DagsSingleton.Dags;
import net.aequologica.neo.dagr.bus.Bus;
import net.aequologica.neo.dagr.bus.Bus.Scope;
import net.aequologica.neo.dagr.bus.BusEvent;
import net.aequologica.neo.dagr.jaxrs.jenkins.NodeCleanerJenkins;
import net.aequologica.neo.dagr.jaxrs.travis.NodeCleanerTravis;
import net.aequologica.neo.dagr.model.Dag;
import net.aequologica.neo.dagr.model.Dag.Node;
import net.aequologica.neo.dagr.model.Dag.NodeValue;
import net.aequologica.neo.dagr.model.Dag.SubDag;
import net.aequologica.neo.garance.SeriesSet;
import net.aequologica.neo.geppaequo.config.geppaequo.GeppaequoConfig;
import rx.Subscription;

@Singleton
@javax.ws.rs.Path(ResourceDags.RESOURCE_PATH)
public class ResourceDags implements Observer {

    final public static String RESOURCE_PATH = "/v1";
    // plurals
    final public static String __DAGS     = "/dags/";
    final public static String __BUSES    = "/buses/";
    final public static String __SUBS     = "/subs/";
    final public static String __NODES    = "/nodes/";
    final public static String __STATES   = "/states/";
    final public static String __EVENTS   = "/events/";
    // singular ?
    final public static String _INFO        = "/info";
    final public static String _JOURNAL     = "/journal";
    final public static String _RELOAD      = "/reload";
    final public static String _TOPOLOGICAL = "/topological";

    final private static Logger LOG        = LoggerFactory.getLogger(ResourceDags.class);
    final private static String SIMPLENAME = ResourceDags.class.getSimpleName();
    
    final private GeppaequoConfig                           geppaequoConfig = CONFIG_REGISTRY.getConfig(GeppaequoConfig.class);
    final private Dags                                      dags            = DagsSingleton.INSTANCE.dags;
    final private Map<String, EnumMap<Scope, Subscription>> subscriptions   = new HashMap<>();

    @Context
    private HttpServletRequest request;

    @Inject 
    private SeriesSet garanceSeries;
    
    public ResourceDags() throws IOException {
        super();
        
        this.dags.addObserver(this);
    }
    
    @PostConstruct
    public void postConstruct() {
        Collection<Map.Entry<String, Collection<Map.Entry<String, String>>>> exceptions = new ArrayList<>();
        update(this.dags, exceptions);
    }
    
    @Override
    /**
     * arg is one of:
     * 
     * Boolean.TRUE                                                          :  start reload all dags
     * String                                                                :  start reload a dag
     * Map.Entry<String,            Collection<Map.Entry<String, String>>>   :  results loading a dag
     * Collection<Map.Entry<String, Collection<Map.Entry<String, String>>>>  :  results loading all dags
     * 
     */
    public void update(Observable o, Object arg) {
        if (!(o instanceof Dags) || !this.dags.equals(o)) {
            return;
        }
        if (arg instanceof Boolean && (boolean)arg) {
            // loading of all dags has started, remove all subscriptions
            for (EnumMap<Scope, Subscription> dagSubscriptions : this.subscriptions.values()) {
                for (Subscription thisSubscription : dagSubscriptions.values()) {
                    thisSubscription.unsubscribe();
                }
            }
            this.subscriptions.clear();
        } else if (arg instanceof String) {
            // loading of one dag has started, remove the attached subscription, if any
            String dagName = (String)arg;
            EnumMap<Scope, Subscription> dagSubscriptions = this.subscriptions.get(dagName);
            for (Subscription thisSubscription : dagSubscriptions.values()) {
                if (thisSubscription != null) {
                    thisSubscription.unsubscribe();
                    this.subscriptions.remove(dagName);
                }
            }
        } else if (arg instanceof Map.Entry<?,?>) {
            // results loading one dag
            @SuppressWarnings("unchecked")
            Map.Entry<String, Collection<Map.Entry<String, String>>> thisDagEntry  = (Map.Entry<String, Collection<Map.Entry<String, String>>>)arg;
            DagOnSteroids  dagOnSteroids = this.dags.getDagOnSteroids(thisDagEntry.getKey().toString());
            if (dagOnSteroids != null) {
                for (Entry<Scope, Bus<Node>> busEntry : dagOnSteroids.getBusEntries()) {
                    thisDagEntry.getValue().addAll(_initGaranceSubscription(dagOnSteroids, dagOnSteroids.getDag().getName(), busEntry));
                }
                thisDagEntry.getValue().addAll(_initProperties(dagOnSteroids));
            }
            
        } else if (arg instanceof Collection<?>) {
            // results loading all dags
            @SuppressWarnings("unchecked")
            Collection<Map.Entry<String, Collection<Map.Entry<String, String>>>> exceptions = (Collection<Map.Entry<String, Collection<Map.Entry<String, String>>>>)arg;
            Collection<DagOnSteroids> dagOnSteroidss = this.dags.getDagOnSteroidss();
            
            for ( DagOnSteroids dagOnSteroids: dagOnSteroidss) {
                for (Entry<Scope, Bus<Node>> busEntry : dagOnSteroids.getBusEntries()) {
                    exceptions.add(tuple(dagOnSteroids.getDag().getName(), _initGaranceSubscription(dagOnSteroids, dagOnSteroids.getDag().getName(), busEntry)));
                }
                exceptions.add(tuple(dagOnSteroids.getDag().getName(), _initProperties(dagOnSteroids)));
            }
        }
    }

    private Collection<Map.Entry<String, String>> _initGaranceSubscription(DagOnSteroids dagOnSteroids, String dagName, Entry<Scope, Bus<Node>> busEntry) {
        final Collection<Map.Entry<String, String>> exceptions = Lists.newArrayList();
        try {
            // register an observer on the bus that will save build durations to garance key/value store
            Subscription subscription = busEntry.getValue().toObservable()
                                           .filter   ( event -> event.getType().equals(BusEvent.Type.CLEAN_OK))
                                           .map      ( event -> event.get())
                                           .subscribe( node  -> {
                                                if (node                                                != null &&
                                                    node.getId()                                        != null &&
                                                    node.getDag()                                       != null &&
                                                    node.getDag().getName()                             != null &&
                                                    dagOnSteroids.getDagCleaner(busEntry.getKey()).getNodeCleaner(node)                  != null && 
                                                    dagOnSteroids.getDagCleaner(busEntry.getKey()).getNodeCleaner(node).getDuration()    != null) {

                                                    String key = /* geppaequoConfig.getLocalhostName() + "::" + */ node.getDag().getName() + "::" + busEntry.getKey() + "::" + node.getId();

                                                    Long duration = dagOnSteroids.getDagCleaner(busEntry.getKey()).getNodeCleaner(node).getDuration();
                                                    this.garanceSeries.put(key, duration).serialize();
                                                }
                                           });
            EnumMap<Scope, Subscription> dagSubscriptions = this.subscriptions.get(dagName);
            if (dagSubscriptions == null) {
                dagSubscriptions = new EnumMap<>(Scope.class);
                this.subscriptions.put(dagName, dagSubscriptions);
            }
            dagSubscriptions.put(busEntry.getKey(), subscription);
        } catch (Exception e) {
            exceptions.add(tuple(e.getClass().getSimpleName(), e.getMessage()));
        }
        return exceptions;
    }

    private Collection<Map.Entry<String, String>> _initProperties(final DagOnSteroids dagOnSteroids) {
        final Map<String, String> properties = dagOnSteroids.getProperties();
        if (properties == null) {
            return Collections.emptyList();
        }
        final Collection<Map.Entry<String, String>> exceptions = Lists.newArrayList();
        try {
            final String dagCleanerType                = properties.get("type" );
            final String dagCleanerName                = properties.get("name" );
            final String dagCleanerUrl                 = properties.get("url"  );
            final String dagCleanerToken               = properties.get("token");
            final String dagCleanerAuthorizationHeader = properties.get("authorizationHeader");
            final String dagCleanerJobNameTemplate     = properties.get("jobNameTemplate");
            final String dagCleanerReleaseNameTemplate = properties.get("releaseNameTemplate");
            final String dagCleanerCallbackHost        = properties.get("callbackHost");
            
            if (dagOnSteroids   != null &&
                dagCleanerType  != null && 
                dagCleanerName  != null &&
                dagCleanerUrl   != null &&
                dagCleanerToken != null) {
                if (dagCleanerType.equals("jenkins")) {
                    Set<Map.Entry<String, String>> dagCleanerParams = _getBuildParamsIfAny(properties.get("params"), exceptions);
                    URI callback = null;
                    if (this.request != null) {
                        String callbackURIAsString = this.request.getRequestURL().toString().replace(
                            this.request.getRequestURI(), 
                            this.request.getContextPath());
                        UriBuilder uriBuilder = UriBuilder.fromUri(callbackURIAsString);
                        if (dagCleanerCallbackHost != null && !dagCleanerCallbackHost.trim().isEmpty()) {
                            uriBuilder = uriBuilder.host(dagCleanerCallbackHost);
                        }
                        callback = uriBuilder.build();
                    }
                    dagOnSteroids.setCleaningFunction(
                        new NodeCleanerJenkins(
                                dagCleanerUrl, 
                                dagCleanerToken,       
                                dagCleanerParams, 
                                dagCleanerAuthorizationHeader,
                                dagCleanerJobNameTemplate,
                                dagCleanerReleaseNameTemplate,
                                callback).getCleaningFunction()
                    );
                } else if (dagCleanerType.equals("travis")) {
                    final String dagCleanerUrlProxy   = properties.get("urlProxy"  );
                    
                    dagOnSteroids.setCleaningFunction(
                        new NodeCleanerTravis(dagCleanerUrlProxy!=null?dagCleanerUrlProxy:dagCleanerUrl, dagCleanerToken).getCleaningFunction()
                    );
                }
            }
        } catch (Exception e) {
            exceptions.add(tuple(e.getClass().getSimpleName(), e.getMessage()));
        }
        return exceptions;
    }

    private Set<Map.Entry<String, String>> _getBuildParamsIfAny(final String dagCleanerParams, Collection<Map.Entry<String, String>> exceptions) {
        if (dagCleanerParams == null) {
            return null;
        }
        Set<Map.Entry<String, String>> params = Collections.emptySet();
        ObjectMapper objectMapper = new ObjectMapper();
        try {
            params = objectMapper.readValue(dagCleanerParams, new TypeReference<Set<Map.Entry<String, String>>>() {});
        } catch (IOException e) {
            exceptions.add(tuple("params="+dagCleanerParams, e.getClass().getSimpleName() + " - " + e.getMessage()));
        }
        return params;
    }

    // ping service
    @GET
    @Path("")
    @Produces(TEXT_PLAIN)
    public String pingGET() {
        return "pong";
    }

    // reload dags
    @POST
    @Path(_RELOAD)
    @Produces(APPLICATION_JSON)
    public Response dagReloadAllPOST() {
        final Collection<Map.Entry<String, Collection<Map.Entry<String, String>>>> allExceptions = this.dags.loadDags();
        return Response.status(Response.Status.OK).entity(allExceptions).build();
    }

    // reload this DAG
    @POST
    @Path(_RELOAD + "/{dag}")
    @Produces(APPLICATION_JSON)
    public Response dagReloadThisPOST(final @PathParam("dag") String dagkey) {
        @SuppressWarnings("unused")
        final Dag dag = _getDag(dagkey); // will throw 404 if not found
        final Map.Entry<String, Collection<Map.Entry<String, String>>> thisDagLoadExceptions = this.dags.loadDag(dagkey);
        if (thisDagLoadExceptions.getValue().size() > 0) {
            return Response.status(Response.Status.INTERNAL_SERVER_ERROR).entity(thisDagLoadExceptions).build();
        } else {
            return Response.status(Response.Status.NO_CONTENT).build();
        }
    }

    // go to dag-list JSP 
    @GET
    @Path("/dag-list")
    @Produces(TEXT_HTML)
    public Viewable dagListGET_HTML() throws IOException {
        return new Viewable("/WEB-INF/dagr/dag-list");
    }

    // get DAGs
    @GET
    @Path(__DAGS)
    @Produces(APPLICATION_JSON)
    public List<DagInfo> dagInfosGET_JSON() throws IOException {
        return this.dags.getDagOnSteroidss().stream().map(d -> DagInfo.create(d)).collect(Collectors.toList());
    }
    
    // current dag JSP
    @GET
    @Path(__DAGS)
    @Produces(TEXT_HTML)
    public Viewable dagsGET_HTML() throws IOException {
        return new Viewable("/WEB-INF/dagr/dag", dagInfosGET_JSON());
    }

    // one dag as json 
    @GET
    @Path(__DAGS + "{dag}")
    @Produces(APPLICATION_JSON)
    public Dag dagGET_JSON(final @PathParam("dag") String dagkey) throws IOException {
        return _getDag(dagkey);
    }

    // one dagInfo as json 
    @GET
    @Path(__DAGS + "{dag}" + _INFO)
    @Produces(APPLICATION_JSON)
    public DagInfo dagInfoGET_JSON(final @PathParam("dag") String dagkey) throws Exception {
        return _getDagInfo(dagkey, null);
    }

    // one dag as html
    @GET
    @Path(__DAGS + "{dag}")
    @Produces(TEXT_HTML)
    public Viewable dagGET_HTML(final @PathParam("dag") String dagkey) {
        DagInfo info;
        try {
            info = _getDagInfo(dagkey, null);
            return new Viewable("/WEB-INF/dagr/dag", Arrays.asList(info));
        } catch (NotFoundException e) {
            return new Viewable("/WEB-INF/dagr/dag", Collections.EMPTY_LIST);
        }
    }

    // subdag resource ===============
    // get all subdags
    @GET
    @Path(__DAGS + "{dag}" + __SUBS)
    @Produces(APPLICATION_JSON)
    public Collection<SubDag> subDagsGET(final @PathParam("dag") String dagkey) throws Exception {
        final Dag dag = _getDag(dagkey);
        if (dag == null) {
            throw new NotFoundException(Response.status(Status.NOT_FOUND).entity("dag /"+dagkey+"/ not found").build());
        }
        return dag.getSubDags();
    }

    // create subdag
    @POST
    @Path(__DAGS + "{dag}" + __SUBS)
    @Consumes(APPLICATION_JSON)
    @Produces(TEXT_PLAIN)
    public String subDagPOST(final Set<String> nodeNames, final @PathParam("dag") String dagkey) throws Exception {
        final Dag dag = _getDag(dagkey);
        if (dag == null) {
            throw new NotFoundException(Response.status(Status.NOT_FOUND).entity("dag /"+dagkey+"/ not found").build());
        }
        SubDag subDag = new SubDag(nodeNames);
        dag.addSubDag(subDag);
        return subDag.getId();
    }

    // one subdag as html
    @GET
    @Path(__DAGS + "{dag}" + __SUBS + "{sub}")
    @Produces(TEXT_HTML)
    public Viewable subDagGET_HTML(final @PathParam("dag") String dagkey, final @PathParam("sub") String subkey, final @QueryParam("order") String order) throws Exception {
        final DagOnSteroids dagOnSteroids = _getDagOnSteroids(dagkey);
        final DagInfo info = _getDagInfo(dagkey, subkey);
        dagOnSteroids.setSubDagId(subkey);
        return new Viewable("/WEB-INF/dagr/dag", Arrays.asList(info));
    }

    // one subdag as json
    @GET
    @Path(__DAGS + "{dag}" + __SUBS + "{sub}")
    @Produces(APPLICATION_JSON)
    public Dag subDagGET_JSON(final @PathParam("dag") String dagkey, final @PathParam("sub") String subkey, final @QueryParam("order") String order) throws Exception {
        final Dag dag = _getDag(dagkey);
        if (dag == null) {
            throw new NotFoundException(Response.status(Status.NOT_FOUND).entity(error("dag ["+dagkey+"] not found")).build());
        }
        SubDag subDag = dag.getSubDag(subkey);
        if (subDag == null) {
            throw new NotFoundException(Response.status(Status.NOT_FOUND).entity(error("dag > sub ["+dagkey+" > "+subkey+"] not found")).build());
        }
        final SortedSet<String> nodeNames = subDag.getNodeNames();
        
        if (order != null) {
            if (order.equals("preds")) {
                return dag.predecessorsDag(nodeNames);
            } else if (order.equals("succs")) {
                return dag.successorsDag(nodeNames);
            }
        }
        return dag.sub(nodeNames);
    }

    // delete subdag 
    @DELETE
    @Path(__DAGS + "{dag}" + __SUBS + "{sub}")
    public void subDagDELETE(final @PathParam("dag") String dagkey, final @PathParam("sub") String subkey) throws Exception {
        final Dag dag = _getDag(dagkey);
        if (dag == null) {
            throw new NotFoundException(Response.status(Status.NOT_FOUND).entity(error("dag ["+dagkey+"] not found")).build());
        }
        SubDag removeSubDag = dag.removeSubDag(subkey);
        if (removeSubDag == null) {
            throw new NotFoundException(Response.status(Status.NOT_FOUND).entity(error("subgraph ["+subkey+"] for dag ["+dagkey+"] not found")).build());
        }
    }

    // dag topological
    @GET
    @Path(__DAGS + "{dag}" + _TOPOLOGICAL)
    @Produces(APPLICATION_JSON)
    public List<String> topologicalGET_JSON(
            final @PathParam("dag")                         String dagkey,
            final @QueryParam("type")  @DefaultValue("id")  String type
          ) throws Exception {
        final Dag dag = _getDag(dagkey);
        Iterator<Node> topologicalOrderIterator = dag.getTopologicalOrderIterator();
        final Function<Node, String> transform;
        if (type == null || type.equals("id")) {
            transform = n -> n.getId();
        } else if (type.equals("name")) {
            transform = n -> n.getName();
        } else if (type.equals("gav")) {
            transform = node-> {
                final NodeValue value = node.getValue();
                if (value == null) { 
                    return node.getId();
                }
                final String gucrid = value.getGucrid();
                if (gucrid == null || gucrid.length() == 0) {
                    return node.getId();
                }
                final Iterable<String> split = Splitter.on(':').split(gucrid);
                final Iterator<String> iterator = split.iterator();
                if (!iterator.hasNext()) {
                    return node.getId();
                }                
                @SuppressWarnings("unused")
                final String groupId = iterator.next();
                if (!iterator.hasNext()) {
                    return node.getId();
                }
                final String artifactId = iterator.next();
                if (artifactId == null || artifactId.toString().length() == 0) {
                    return node.getId();
                }
                return artifactId;
            };
        } else {
            throw new InvalidParameterException("invalid parameter 'type' ('"+type+"') in query string. must be one of ['id' | 'name' | 'gav']; when not type parameter is given, type='id' is the default.");
        }
        
        return Lists.<String>newArrayList(Iterators.transform(topologicalOrderIterator, transform));
    }

    // get journal of rebuild all nodes
    @GET
    @Path(__DAGS + "{dag}" + _JOURNAL)
    @Produces(TEXT_PLAIN)
    public String rebuildJournalDag_GET_TEXT_PLAIN(
            final @PathParam("dag")                         String  dagkey,
            final @MatrixParam("bus") @DefaultValue("JOB")  Scope   scope) throws Exception {
        return rebuildJournalDagBus_GET_TEXT_PLAIN(dagkey, scope);
    }
    @GET
    @Path(__DAGS + "{dag}" + __BUSES + "{bus}" + _JOURNAL)
    @Produces(TEXT_PLAIN)
    public String rebuildJournalDagBus_GET_TEXT_PLAIN(
            final @PathParam("dag")                         String  dagkey,
            final @PathParam("bus")                         Scope   scope) throws Exception {
        final DagOnSteroids dagOnSteroids  = this.dags.getDagOnSteroids(dagkey);
        if (dagOnSteroids == null) {
            throw new NotFoundException(Response.status(Status.NOT_FOUND).entity(error("dag ["+dagkey+"] not found").valueOf(MediaType.TEXT_PLAIN_TYPE)).build());
        }
        return dagOnSteroids.getDagCleaner(scope).getJournalAsString();
    }

    /////////////////////////////////////////////////////
    // cancel rebuild all nodes
    @DELETE
    @Path(__DAGS + "{dag}")
    @Produces(APPLICATION_JSON)
    public Response rebuildCancelDagDELETE(
            final @PathParam("dag")                         String   dagkey,
            final @MatrixParam("bus") @DefaultValue("JOB")  Scope scope) throws Exception {
        return rebuildCancelDagSubBusDELETE(dagkey, scope, null);
    }
    @DELETE
    @Path(__DAGS + "{dag}" + __BUSES + "{bus}")
    @Produces(APPLICATION_JSON)
    public Response rebuildCancelDagBusDELETE(
            final @PathParam("dag")                         String      dagkey,
            final @PathParam("bus")                         Scope    scope) throws Exception {
        return rebuildCancelDagSubBusDELETE(dagkey, scope, null);
    }
    @DELETE
    @Path(__DAGS + "{dag}" + __SUBS + "{sub}")
    @Produces(APPLICATION_JSON)
    public Response rebuildCancelDagSubDELETE(
            final @PathParam("dag")                         String   dagkey,
            final @PathParam("sub")                         String   subkey, 
            final @MatrixParam("bus") @DefaultValue("JOB")  Scope scope) throws Exception {
        return rebuildCancelDagSubBusDELETE(dagkey, scope, subkey);
    }
    @DELETE
    @Path(__DAGS + "{dag}" + __BUSES + "{bus}" + __SUBS + "{sub}")
    @Produces(APPLICATION_JSON)
    public Response rebuildCancelDagSubBusDELETE(
            final @PathParam("dag")                         String   dagkey,
            final @PathParam("bus")                         Scope scope,
            final @PathParam("sub")                         String   subkey) throws Exception {
        final DagOnSteroids dagOnSteroids = _getDagOnSteroids(dagkey);
        final DagCleaner    dagCleaner    = dagOnSteroids.getDagCleaner(scope);
        String info = null;
        if (!dagCleaner.isRunning()) {
            info = "cleaning not running, nothing to cancel";
        } else {
            dagCleaner.cancel();
            if (!dagCleaner.isRunning()) {
                info = "cleaning cancelled (nb. node cleanings already ordered or currently running will continue until completion)";
            } else {
                info = "oooops, seems that cleaning could not be cancelled ... maybe retry ?";
            }
        }
        return Response.ok(message(info)).build();
    }
    /////////////////////////////////////////////////////

    /////////////////////////////////////////////////////
    // rebuild all nodes
    //
    @PATCH
    @Path(__DAGS + "{dag}")
    @Produces(APPLICATION_JSON)
    public Response rebuildDag_PATCH(
            final @PathParam("dag")                         String      dagkey,
            final @MatrixParam("bus") @DefaultValue("JOB")  Scope    scope,
            final @QueryParam("onlyThisNodeAndSuccessors")  String      onlyThisNodeAndSuccessors) throws Exception {
        return rebuildDagBusSub_PATCH(dagkey, scope, null, onlyThisNodeAndSuccessors);
    }
    @PATCH
    @Path(__DAGS + "{dag}" + __BUSES + "{bus}")
    @Produces(APPLICATION_JSON)
    public Response rebuildDagBus_PATCH(
            final @PathParam("dag")                         String      dagkey,
            final @PathParam("bus")                         Scope    scope, 
            final @QueryParam("onlyThisNodeAndSuccessors")  String      onlyThisNodeAndSuccessors) throws Exception {
        return rebuildDagBusSub_PATCH(dagkey, scope, null, onlyThisNodeAndSuccessors);
    }
    // rebuild all sub dag nodes
    @PATCH
    @Path(__DAGS + "{dag}" + __SUBS + "{sub}")
    @Produces(APPLICATION_JSON)
    public Response rebuildDagSub_PATCH(
            final @PathParam("dag")                         String      dagkey,
            final @PathParam("sub")                         String      subkey, 
            final @QueryParam("onlyThisNodeAndSuccessors")  String      onlyThisNodeAndSuccessors,
            final @MatrixParam("bus") @DefaultValue("JOB")  Scope    scope) throws Exception {
        return rebuildDagBusSub_PATCH(dagkey, scope, subkey, onlyThisNodeAndSuccessors);
    }
    @PATCH
    @Path(__DAGS + "{dag}" + __BUSES + "{bus}" + __SUBS + "{sub}")
    @Produces(APPLICATION_JSON)
    public Response rebuildDagBusSub_PATCH(
            final @PathParam("dag")                         String      dagkey,
            final @PathParam("bus")                         Scope    scope,
            final @PathParam("sub")                         String      subkey, 
            final @QueryParam("onlyThisNodeAndSuccessors")  String      onlyThisNodeAndSuccessors) throws Exception {
        final DagOnSteroids dagOnSteroids = _getDagOnSteroids(dagkey);
        try {
            final DagCleaner       dagCleaner         = dagOnSteroids.getDagCleaner(scope);
            dagCleaner.cleanFromInclusive(onlyThisNodeAndSuccessors, subkey);
            
        } catch (Exception e) {
            throw new WebApplicationException(Response.status(Status.INTERNAL_SERVER_ERROR).entity(error(e.getMessage())).build());            
        }
        return Response.created(URI.create("journal")).status(Status.ACCEPTED).build();
    }
    //
    /////////////////////////////////////////////////////

    // dag nodes as json 
    @GET
    @Path(__DAGS + "{dag}" + __NODES)
    @Produces(APPLICATION_JSON)
    public Collection<Node> dagNodesGET_JSON(final @PathParam("dag") String dagkey) throws Exception {
        final Dag dag = _getDag(dagkey);
        return dag.getNodes();
    }

    // one dag node as json
    @GET
    @Path(__DAGS + "{dag}" + __NODES + "{node}")
    @Produces(APPLICATION_JSON)
    public Node dagNodeGET_JSON(final @PathParam("dag") String dagkey, 
                                final @PathParam("node") String node) throws Exception {
        return _getDagAndNodeByName(dagkey, node).node;
    }

    // get one dag node state 
    @GET
    @Path(__DAGS + "{dag}" + __NODES + "{node}" + __STATES)
    @Produces(APPLICATION_JSON)
    public NodeState dagNodeStateGET_JSONfinal (final @PathParam("dag")  String dagkey, 
                                                final @PathParam("node") String nodename,
                                                final @MatrixParam("bus") @DefaultValue("JOB") Scope scope) throws Exception {
        return dagNodeStateGET_JSONfinal(dagkey, scope, nodename);
    }
    @GET
    @Path(__DAGS + "{dag}" + __BUSES + "{bus}" + __NODES + "{node}" + __STATES)
    @Produces(APPLICATION_JSON)
    public NodeState dagNodeStateGET_JSONfinal (final @PathParam("dag")  String dagkey, 
                                                final @PathParam("bus")  Scope scope, 
                                                final @PathParam("node") String nodename) throws Exception {
        final DagOnSteroids dagOnSteroids = _getDagOnSteroids(dagkey);
        final Node node = dagNodeGET_JSON(dagkey, nodename);
        return dagOnSteroids.getDagCleaner(scope).getNodeCleaner(node).getState();
    }

    // clean this dag node 
    @PATCH
    @Path(__DAGS + "{dag}" + __NODES + "{node}")
    @Produces(APPLICATION_JSON)
    public Response dagNodeCleanPOST(
            final @PathParam("dag")                        String   dagkey, 
            final @PathParam("node")                       String   node,
            final @MatrixParam("bus") @DefaultValue("JOB") Scope scope) throws Exception {
        return dagNodeCleanBusPOST( dagkey,  scope, node);
    }
    @PATCH
    @Path(__DAGS + "{dag}" + __BUSES + "{bus}" + __NODES + "{node}")
    @Produces(APPLICATION_JSON)
    public Response dagNodeCleanBusPOST(
            final @PathParam("dag")         String      dagkey, 
            final @PathParam("bus")         Scope    scope, 
            final @PathParam("node")        String      node) throws Exception {
        final DagAndNode    dagAndNode      = _getDagAndNodeByName(dagkey, node);
        final DagOnSteroids dagOnSteroids   = _getDagOnSteroids(dagkey);
        final DagCleaner    dagCleaner      = dagOnSteroids.getDagCleaner(scope); 
        final Bus<Node>     bus             = dagCleaner.getBus();
        
        final NodeCleaner.NodeCleaningResult cleanResult = dagOnSteroids.clean(scope, dagAndNode.node);
        if (cleanResult!= null && cleanResult.asBoolean()) {
            _sendBusEvent(bus, CLEAN_ORDER_OK, dagAndNode.node);
            return Response.accepted().build();
        } else {
            _sendBusEvent(bus, CLEAN_ORDER_ERROR, dagAndNode.node);
            return Response.serverError().entity(cleanResult).build();
        }
    }

    // set one dag node state 
    @PATCH
    @Path(__DAGS + "{dag}" + __NODES + "{node}" + __STATES + "{state}")
    @Produces(APPLICATION_JSON)
    public Response dagNodeStatePATCH_JSON(
            final @PathParam("dag")                         String      dagkey, 
            final @PathParam("node")                        String      node, 
            final @PathParam("state")                       NodeState   state,
            final @MatrixParam("bus") @DefaultValue("JOB")  Scope    scope) throws Exception {
        return dagNodeStateBusPATCH_JSON(dagkey, scope, node, state);
    }
    @PATCH
    @Path(__DAGS + "{dag}" + __BUSES + "{bus}" + __NODES + "{node}" + __STATES + "{state}")
    @Produces(APPLICATION_JSON)
    public Response dagNodeStateBusPATCH_JSON(
            final @PathParam("dag")      String     dagkey,
            final @PathParam("bus")      Scope   scope, 
            final @PathParam("node")     String     nodename, 
            final @PathParam("state")    NodeState  newState) throws Exception {
        try {
            final DagOnSteroids dagOnSteroids   = _getDagOnSteroids(dagkey);
            final DagCleaner    dagCleaner      = dagOnSteroids.getDagCleaner(scope);
            final Bus<Node>     bus             = dagCleaner.getBus();
            final Node          node            = dagNodeGET_JSON(dagkey, nodename);

            NodeState currentState = dagOnSteroids.getDagCleaner(scope).getNodeCleaner(node).getState();
            if ((currentState == null && newState == null) || 
                (currentState != null && newState != null && newState.equals(currentState))) {
                // 
                // http://serverfault.com/a/168348
                // The 304 response MUST NOT contain a message-body, and thus is always terminated by the first empty line after the header fields. 
                // 
                return Response.status(Response.Status.NOT_MODIFIED).build();
            }
            dagOnSteroids.getDagCleaner(scope).getNodeCleaner(node).setState(newState);
            _sendBusEvent(bus, STATE_CHANGE_HACK, node);
            return Response.status(Response.Status.OK).build();
        } catch (Exception e) {
            LOG.error("[dagr "+SIMPLENAME+"] exception /{}/ logged and re-thrown", e.getMessage());
            throw e;
        } finally {
        }
    }

    // send an event to the bus
    @PATCH
    @Path(__DAGS + "{dag}" + __NODES + "{node}" + __EVENTS + "{event}")
    @Consumes(TEXT_PLAIN)
    @Produces(APPLICATION_JSON)
    public Response dagNodeEventSendPATCH(  final                           String        cleanerId, 
                                            final @PathParam("dag")         String        dagkey, 
                                            final @PathParam("node")        String        node, 
                                            final @PathParam("event")       BusEvent.Type eventType,
                                            final @MatrixParam("bus")       @DefaultValue("JOB") Scope scope)  {
        return dagNodeEventBusSendPATCH(cleanerId, dagkey, scope, node, eventType);
    }
    @PATCH
    @Path(__DAGS + "{dag}" + __BUSES + "{bus}" + __NODES + "{node}" + __EVENTS + "{event}")
    @Consumes(TEXT_PLAIN)
    @Produces(APPLICATION_JSON)
    public Response dagNodeEventBusSendPATCH(   final                           String        cleanerId, 
                                                final @PathParam("dag")         String        dagkey,
                                                final @PathParam("bus")         Scope      scope, 
                                                final @PathParam("node")        String        node, 
                                                final @PathParam("event")       BusEvent.Type eventType)  {
        final DagOnSteroids dagOnSteroids   = _getDagOnSteroids(dagkey);
        final DagCleaner    dagCleaner      = dagOnSteroids.getDagCleaner(scope);
        final Bus<Node>     bus             = dagCleaner.getBus();
        final DagAndNode    dagAndNode      = _getDagAndNodeByName(dagkey, node);
        
        if (eventType.equals(BusEvent.Type.CLEAN_STARTED)) {
            if (cleanerId != null && cleanerId.length()>0) {
                dagOnSteroids.getDagCleaner(scope).getNodeCleaner(dagAndNode.node).setId(cleanerId);
            }
        } else if (eventType.equals(BusEvent.Type.CLEAN_OK   ) || 
                eventType.equals(BusEvent.Type.CLEAN_ERROR   ) || 
                eventType.equals(BusEvent.Type.CLEAN_ABORTED )) {
            dagOnSteroids.getDagCleaner(scope).getNodeCleaner(dagAndNode.node).setId(null);
        }
            
        _sendBusEvent(bus, eventType, dagAndNode.node);
        
        return Response.status(Response.Status.OK).build();
    }
    
    //////////////////////////////////////////////////////////////////////////////////

    private Dag _getDag(final String dagkey) throws NotFoundException {
        String[] both = DagOnSteroids.parseName(dagkey);
        final Dag dag = this.dags.getDAG(both[0]);
        if (dag == null) {
            throw new NotFoundException(Response.status(Status.NOT_FOUND).entity(error("dag ["+dagkey+"] not found").valueOf(MediaType.APPLICATION_JSON_TYPE)).build());
        }
        return dag;
    }

    private DagInfo _getDagInfo(final String dagkey, final String subDagId) throws NotFoundException {
        final DagOnSteroids dagOnSteroids  = _getDagOnSteroids(dagkey);
        dagOnSteroids.setSubDagId(subDagId);
        final DagInfo info = DagInfo.create(dagOnSteroids);
        if (info == null) {
            throw new NotFoundException(Response.status(Status.NOT_FOUND).entity(error("info for dag ["+dagkey+"] not found").valueOf(MediaType.APPLICATION_JSON_TYPE)).build());
        }
        return info;
    }

    private DagOnSteroids _getDagOnSteroids(final String dagkey) throws NotFoundException {
        final DagOnSteroids dagOnSteroids  = this.dags.getDagOnSteroids(dagkey);
        if (dagOnSteroids == null) {
            throw new NotFoundException(Response.status(Status.NOT_FOUND).entity(error("dag ["+dagkey+"] not found").valueOf(MediaType.APPLICATION_JSON_TYPE)).build());
        }
        return dagOnSteroids;
    }

    
    private DagAndNode _getDagAndNodeByName(final String dagkey, final String nodeName) throws NotFoundException {
        final Dag dag = _getDag(dagkey);
        final List<Node> nodes = dag.getNodesFromName(nodeName);
        if (nodes.size() == 0) {
            throw new NotFoundException(Response.status(Status.NOT_FOUND).entity(error("dag ["+dagkey+"] node ["+nodeName+"] not found").valueOf(MediaType.APPLICATION_JSON_TYPE)).build());
        } else if (nodes.size() > 1) {
            throw new WebApplicationException(Response.status(Status.NOT_IMPLEMENTED).entity("application error. Found multiple nodes with the same name ["+nodeName+"] in dag ["+ dagkey+ "]. This is not supported").build());
        }
        return dagAndNode(dag, nodes.get(0));
    }

    // helper to send an event on the bus
    private void _sendBusEvent(final Bus<Node> bus, final BusEvent.Type eventType, final Node node) {
        if (bus == null) {
            throw new RuntimeException("no bus");
        }
        LOG.debug("[dagr "+SIMPLENAME+"] sending event {}, owner_name/name='{}', branch='{}' to the bus", eventType, node.getName(), node.getValue() != null ? node.getValue().getBranch() : null);
        
        // balance dans le bus
        bus.send(eventType, node.getName(), node.getValue() != null ? node.getValue().getBranch() : null, request.getRequestURI().toString());
    }

    static class DagAndNode {
        final Dag dag;
        final Node node;
        static DagAndNode dagAndNode(final Dag dag,final  Node node) {
            return new DagAndNode(dag, node);
        }
        private DagAndNode(final Dag dag, final Node node) {
            super();
            this.dag = dag;
            this.node = node;
        }
    }

    @JsonIgnoreProperties
    static public class Error {
        @JsonProperty
        final String error;

        static public Error error(final String error) {
            return new Error(error);
        }

        public Error(String error) {
            this.error = error;
        }
        
        public Object valueOf(MediaType mediaType) {
            if (mediaType.equals(MediaType.TEXT_PLAIN)) {
                return error;
            } else if (mediaType.equals(MediaType.APPLICATION_JSON)) {
                return this;
            } else {
                return error;
            }
        }
    }

    @JsonIgnoreProperties
    static public class Message {
        @JsonProperty
        final String message;

        static public Message message(final String message) {
            return new Message(message);
        }

        public Message(String message) {
            this.message = message;
        }
        
    }
    
    static private <T> Map.Entry<String, T> tuple(final String a, final T b) {
        return new AbstractMap.SimpleImmutableEntry<>(a, b);
    }
    
}
