package net.aequologica.neo.dagr;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.file.FileSystem;
import java.nio.file.FileSystems;
import java.nio.file.Path;
import java.nio.file.PathMatcher;
import java.nio.file.Paths;
import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Observable;
import java.util.Properties;
import java.util.Set;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.aequologica.neo.dagr.DagOnSteroids.DagCleaner;
import net.aequologica.neo.dagr.bus.Bus;
import net.aequologica.neo.dagr.model.Dag;
import net.aequologica.neo.dagr.model.Dag.Node;
import net.aequologica.neo.dagr.model.DagDocumentSerializer;
import net.aequologica.neo.geppaequo.document.DocumentHelper;

public enum DagsSingleton {
    
    INSTANCE;
    
    public final Dags dags = new Dags();

    public static class Dags extends Observable {

        private final static Logger                 log             = LoggerFactory.getLogger(Dags.class);
        private final static FileSystem             fileSystem      = FileSystems.getDefault();
        private final static DagDocumentSerializer  serializer      = new DagDocumentSerializer();
        private final static String                 DAGS_DIRECTORY  = "/dags";
        
        private final Lock                          lockOnIO;
        private final Map<String, DagOnSteroids>    map;
        
        Dags() {
            this.lockOnIO   = new ReentrantLock();
            this.map        = new HashMap<>();
            
            loadDags();
        }

        /////////////////////////////
        // collections getters
        public Set<String> getDagNames() {
            return this.map.keySet();
        }
        public Collection<DagOnSteroids> getDagOnSteroidss() {
            return this.map.values();
        }
        public List<Dag> getDAGs() {
            return this.map.values().stream().map(b -> b.getDag()).collect(Collectors.toList());
        }
        
        /////////////////////////////
        // elements getters
        public DagOnSteroids getDagOnSteroids(final String dagkey) {
            return this.map.get(dagkey);
        } 
        public Dag getDAG(final String dagkey) {
            DagOnSteroids dagOnSteroids = this.map.get(dagkey); return dagOnSteroids == null ? null : dagOnSteroids.getDag();
        }
        public Bus<Node, Scope> getBus(final String dagkey, final Scope scope) {
            final DagOnSteroids dagOnSteroids = this.map.get(dagkey); 
            if (dagOnSteroids == null) {
                return null;
            }
            final DagCleaner dagCleaner = dagOnSteroids.getDagCleaner(scope);
            if (dagCleaner == null) {
                return null;
            }
            return dagCleaner.getBus();
        }
        
        /////////////////////////////
        // loaders
        public Collection<Map.Entry<String, Collection<Map.Entry<String, String>>>> loadDags() {
            
            this.lockOnIO.lock(); // block until condition holds
            try {
                final Collection<Map.Entry<String, Collection<Map.Entry<String, String>>>> allExceptions = new ArrayList<>();

                // notify before clear
                setChanged();
                notifyObservers(Boolean.TRUE);
                this.map.clear();

                List<Path> sources;
                try {
                    sources = privateGetDagSources();
                } catch (Exception e) {
                    Collection<Entry<String, String>> bigError = new ArrayList<>();
                    bigError.add(tuple(e.getClass().getSimpleName(), e.getMessage()));
                    allExceptions.add(tuple("", bigError));
                    return allExceptions;
                }

                for (Path source : sources) {
                    PathMatcher matcher = fileSystem.getPathMatcher("regex:^.*\\.(properties|yml|hbs|gitmodules|xml|pom)$");
                    if (matcher.matches(source.getFileName())) {
                        continue;
                    }
                    allExceptions.add(privateLoadDag(null, source));
                }

                // notify after puts
                setChanged();
                notifyObservers(allExceptions);
                return allExceptions;
            } finally {
                this.lockOnIO.unlock();
            }
        }

        public Map.Entry<String, Collection<Map.Entry<String, String>>> loadDag(String dagName) {
            this.lockOnIO.lock(); // block until condition holds
            try {
                // notify before clear
                setChanged();
                notifyObservers(dagName);
                
                final Dag  dag = getDAG(dagName);
                final Path source;
                if (dag != null) {
                    source = Paths.get(dag.getSource());
                } else {
                    try {
                        source = privateLocateSourceWithDagName(dagName);
                    } catch (IOException e) {
                        Collection<Map.Entry<String, String>> exceptions = new ArrayList<>();
                        exceptions.add(tuple(e.getClass().getSimpleName(), e.getMessage()));
                        return tuple(dagName, exceptions);
                    }
                }
                
                final Map.Entry<String, Collection<Map.Entry<String, String>>> thisDagExceptions = privateLoadDag(dagName, source);
                        
                // notify after puts
                setChanged();
                notifyObservers(thisDagExceptions);
                return thisDagExceptions;
            } finally {
                this.lockOnIO.unlock();
            }
        }

        private class PropertiesAndExceptions {
            Map<String, String> properties;
            final Collection<Map.Entry<String, String>> exceptions = new ArrayList<>();
            
            void load(Path propertiesPath) {
                
               // properties is optional, not an issue if it does not exists, simply return null
                try (final InputStream inputStream = DocumentHelper.getInputStream(propertiesPath);) {
                    if (inputStream == null) {
                        throw new IOException("no ["+ propertiesPath + "] in document service");
                    }
        
                    try {
                        final Properties properties = new Properties();
                        properties.load(new InputStreamReader(inputStream, "UTF-8"));
                        this.properties = properties.entrySet().stream().collect(Collectors.toMap(e -> e.getKey().toString(), e -> e.getValue().toString()));
                    } catch (Exception accumulated) {
                        // if properties exists, and there is an error loading / java8streamingCollecting it, please report the error
                        exceptions.add(tuple(propertiesPath.getFileName().toString(), accumulated.getClass().getSimpleName() + " - " + accumulated.getMessage()));
                        this.properties = null;
                    }
                    
                } catch (Exception ignored) {
                    log.info("ignored exception {} {}", ignored.getClass().getSimpleName(), ignored.getMessage());
                    this.properties = null;
                }
            }
        }


        private Map.Entry<String, Collection<Map.Entry<String, String>>> privateLoadDag(String dagName, final Path source) {
            if (source == null) {
                throw new IllegalArgumentException("source must be not null");
            } 
            
            final Collection<Map.Entry<String, String>> exceptions = new ArrayList<>();

            try {
                final Dag dag = serializer.read(source);
                
                if (dagName == null) {
                    dagName = dag.getName();
                }
                
                this.map.remove(dagName);
                
                final Path sourceWithoutJSONExtensionIFAny;
                int indexJsonExtension = source.toString().lastIndexOf(".json");
                if (0 < indexJsonExtension) {
                    sourceWithoutJSONExtensionIFAny = Paths.get(source.toString().substring(0, indexJsonExtension));
                } else {
                    sourceWithoutJSONExtensionIFAny = source;
                }
                
                // aliases
                PropertiesAndExceptions aliasesPAE = new PropertiesAndExceptions();
                aliasesPAE.load(Paths.get(sourceWithoutJSONExtensionIFAny+".aliases.properties"));
                if (aliasesPAE.exceptions != null && aliasesPAE.exceptions.size() > 0) {
                    exceptions.addAll(aliasesPAE.exceptions);
                }
                
                // cleaner
                PropertiesAndExceptions cleanerPAE = new PropertiesAndExceptions();
                cleanerPAE.load(Paths.get(sourceWithoutJSONExtensionIFAny+".cleaner.properties"));
                if (cleanerPAE.exceptions != null && cleanerPAE.exceptions.size() > 0) {
                    exceptions.addAll(cleanerPAE.exceptions);
                }
                
                map.put(dag.getName(), new DagOnSteroids(dag, aliasesPAE.properties, cleanerPAE.properties));

            } catch (Exception e) {
                exceptions.add(tuple(e.getClass().getSimpleName(), e.getMessage()));
            }
            
            return tuple(dagName!=null ? dagName : source.getFileName().toString(), exceptions);
        }
        
        static void packageDumpDag(Path path, Dag dag) throws IOException {
            serializer.write(path, dag);
        }

        static private <T> Map.Entry<String, T> tuple(final String a, final T b) {
            return new AbstractMap.SimpleImmutableEntry<>(a, b);
        }
        
        private Path privateLocateSourceWithDagName(final String dagName) throws IOException {
            final List<Path> sources = privateGetDagSources();
            for (final Path source : sources) {
                final Dag dag = serializer.read(source);
                if (dag.getName().equals(dagName)) {
                    return source;
                }
            }
            List<String> collect = sources
                .stream()
                .map(s -> s.getFileName().toString())
                .collect(Collectors.toList()); 
            throw new IOException("no dag with name /" + dagName + "/ found in " + collect);
        }
        
        private List<Path> privateGetDagSources() throws IOException {
            final List<Path>  potentials = DocumentHelper.list(Paths.get(DAGS_DIRECTORY));
            final List<Path>  sources    = new ArrayList<>();
            final PathMatcher matcher    = fileSystem.getPathMatcher("regex:^.*\\.(properties|yml|hbs|gitmodules|xml|pom)$");
            for (final Path source : potentials) {
                if (matcher.matches(source.getFileName())) {
                    continue;
                }
                sources.add(source);
            }
            return sources;
        }

    }
}
