package net.aequologica.neo.dagr;

import static net.aequologica.neo.dagr.DagOnSteroids.NodeCleaner.NodeState.ABORTED;
import static net.aequologica.neo.dagr.DagOnSteroids.NodeCleaner.NodeState.BEING_CLEANED;
import static net.aequologica.neo.dagr.DagOnSteroids.NodeCleaner.NodeState.CLEAN;
import static net.aequologica.neo.dagr.DagOnSteroids.NodeCleaner.NodeState.CLEANED;
import static net.aequologica.neo.dagr.DagOnSteroids.NodeCleaner.NodeState.CLEANING_ORDERED;
import static net.aequologica.neo.dagr.DagOnSteroids.NodeCleaner.NodeState.DIRTY;
import static net.aequologica.neo.dagr.DagOnSteroids.NodeCleaner.NodeState.FAIL;
import static net.aequologica.neo.dagr.DagOnSteroids.NodeCleaner.NodeState.UNCLEANABLE;
import static net.aequologica.neo.dagr.bus.BusEvent.Type.CLEAN_ABORTED;
import static net.aequologica.neo.dagr.bus.BusEvent.Type.CLEAN_ERROR;
import static net.aequologica.neo.dagr.bus.BusEvent.Type.CLEAN_OK;
import static net.aequologica.neo.dagr.bus.BusEvent.Type.CLEAN_ORDER_ERROR;
import static net.aequologica.neo.dagr.bus.BusEvent.Type.CLEAN_ORDER_OK;
import static net.aequologica.neo.dagr.bus.BusEvent.Type.CLEAN_STARTED;
import static net.aequologica.neo.dagr.bus.BusEvent.Type.MAGIC_CLEAN;
import static net.aequologica.neo.dagr.bus.BusEvent.Type.SMUDGE;

import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.aequologica.neo.dagr.DagOnSteroids.NodeCleaner.NodeState;
import net.aequologica.neo.dagr.bus.Bus;
import net.aequologica.neo.dagr.bus.BusEvent;
import net.aequologica.neo.dagr.model.Dag.Node;
import rx.Observable;
import rx.subjects.PublishSubject;
import rx.subjects.SerializedSubject;
import rx.subjects.Subject;

class NodeBus implements Bus<Node> {

    private final static Logger LOG = LoggerFactory.getLogger(NodeBus.class);

    // cf. http://nerds.weddingpartyapp.com/tech/2014/12/24/implementing-an-event-nodeBus-with-rxjava-rxbus/
    final private Subject<BusEvent<Node>, BusEvent<Node>>   nodeBus;
    final private DagOnSteroids                             dagOnSteroids;
    final private Scope                                     scope;
    
    NodeBus(DagOnSteroids dagOnSteroids, Scope scope) {
        this.nodeBus        = new SerializedSubject<>(PublishSubject.create());
        this.dagOnSteroids  = dagOnSteroids;
        this.scope          = scope;
        LOG.debug("[bus {}] created bus for dag {}", scope, dagOnSteroids.getDag().getName());
    }
    
    @Override
    public Scope getScope() {
        return scope;
    }

    @Override
    public void send(final BusEvent.Type type, final String nodeName, final String branch, final String source) {
        try {
            // the state machine
            final NodeState newState;
            // no switch on enum value in java ... sob
            if (       type.equals( MAGIC_CLEAN       )) { newState = CLEAN             ; 
            } else if (type.equals( SMUDGE            )) { newState = DIRTY             ; 
            } else if (type.equals( CLEAN_ORDER_OK    )) { newState = CLEANING_ORDERED  ; 
            } else if (type.equals( CLEAN_ORDER_ERROR )) { newState = UNCLEANABLE       ; 
            } else if (type.equals( CLEAN_STARTED     )) { newState = BEING_CLEANED     ; 
            } else if (type.equals( CLEAN_ERROR       )) { newState = FAIL              ; 
            } else if (type.equals( CLEAN_ABORTED     )) { newState = ABORTED           ; 
            } else if (type.equals( CLEAN_OK          )) { newState = CLEANED           ; 
            } else {
                newState = null; 
            }
    
            List<Node> nodes = DagOnSteroids.getNodesFromNamedAndBranchContains(dagOnSteroids.getDag(), nodeName, branch);
    
            LOG.debug("[bus {}] found {} nodes with nodeName='{}' and branch='{}' in dag '{}'", scope, nodes.size(), nodeName, branch, dagOnSteroids.getDag().getName());

            for (final Node node : nodes) {
                NodeState previousState = dagOnSteroids.getDagCleaner(scope).getNodeCleaner(node).getState();
                if (newState != null && (previousState == null || !previousState.equals(newState))) {
                    dagOnSteroids.getDagCleaner(scope).getNodeCleaner(node).setState(newState);
                }
            
                LOG.debug("[bus {}] about to notify observers that state of node {} ({}) changed from {} to {}", scope, node.getName(), node.getId(), previousState, dagOnSteroids.getDagCleaner(scope).getNodeCleaner(node).getState());

                this.nodeBus.onNext(new BusEvent<Node>() {
                    @Override public Type   getType()     { return type;    }
                    @Override public String getSource()   { return source;  }
                    @Override public Node   get()         { return node;    }
                });
            }
        } catch (Exception e) {
            LOG.error("[bus {}] exception {} logged and re-thrown", scope, e.getMessage());
            throw e;
        } finally {
        }
    }

    @Override
    public Observable<BusEvent<Node>> toObservable() {
      return this.nodeBus;
    }

}
