package net.aequologica.neo.dagr;

import static net.aequologica.neo.dagr.DagOnSteroids.NodeCleaner.NodeState.ABORTED;
import static net.aequologica.neo.dagr.DagOnSteroids.NodeCleaner.NodeState.CLEAN;
import static net.aequologica.neo.dagr.DagOnSteroids.NodeCleaner.NodeState.CLEANED;
import static net.aequologica.neo.dagr.DagOnSteroids.NodeCleaner.NodeState.DIRTY;
import static net.aequologica.neo.dagr.DagOnSteroids.NodeCleaner.NodeState.FAIL;
import static net.aequologica.neo.dagr.DagOnSteroids.NodeCleaner.NodeState.UNCLEANABLE;
import static net.aequologica.neo.dagr.bus.BusEvent.Type.CLEAN_ABORTED;
import static net.aequologica.neo.dagr.bus.BusEvent.Type.CLEAN_ERROR;
import static net.aequologica.neo.dagr.bus.BusEvent.Type.CLEAN_OK;
import static net.aequologica.neo.dagr.bus.BusEvent.Type.CLEAN_ORDER_ERROR;
import static net.aequologica.neo.dagr.bus.BusEvent.Type.CLEAN_ORDER_OK;
import static net.aequologica.neo.dagr.bus.BusEvent.Type.CLEAN_STARTED;
import static net.aequologica.neo.dagr.bus.BusEvent.Type.MAGIC_CLEAN;
import static net.aequologica.neo.dagr.bus.BusEvent.Type.SMUDGE;

import java.io.BufferedWriter;
import java.io.ByteArrayOutputStream;
import java.io.Closeable;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.text.SimpleDateFormat;
import java.util.AbstractMap;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.EnumMap;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.UUID;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.google.common.base.Joiner;

import net.aequologica.neo.dagr.DagOnSteroids.NodeCleaner.NodeCleanerException;
import net.aequologica.neo.dagr.DagOnSteroids.NodeCleaner.NodeCleaningResult;
import net.aequologica.neo.dagr.DagOnSteroids.NodeCleaner.NodeState;
import net.aequologica.neo.dagr.bus.Bus;
import net.aequologica.neo.dagr.bus.Bus.Scope;
import net.aequologica.neo.dagr.model.Dag;
import net.aequologica.neo.dagr.model.Dag.Node;
import pl.touk.throwing.ThrowingBiFunction;
import rx.Subscription;

public class DagOnSteroids {

    private static final Logger           LOG        = LoggerFactory.getLogger(DagOnSteroids.class);
    private static final SimpleDateFormat FMT        = new SimpleDateFormat("yyyy.MM.dd HH:mm:ss");
    private static final String           SIMPLENAME = DagOnSteroids.class.getSimpleName();
    
    ThrowingBiFunction<Scope, Node, NodeCleaner.NodeCleaningResult, NodeCleaner.NodeCleanerException> cleaningFunction = (b, n) -> NodeCleaner.NodeCleaningResult.undefined(b);

    final private Dag                         dag;
    final private Map<String, String>         properties;
    final private EnumMap<Scope, DagCleaner>  dagCleaners;
    
    private Optional<String> subDagId;

    private enum State {
        RUNNING, DONE, CANCELLED;
    }

    public DagOnSteroids(final Dag dag, Map<String, String> nodeAliases, Map<String, String> properties) {
        this.dag         = dag;
        this.dagCleaners = new EnumMap<Scope, DagCleaner>(Scope.class);
        
        this.properties = properties != null ? properties : Collections.emptyMap();

        for (Scope scope : Scope.values()) {
            this.dagCleaners.put(scope, new DagCleaner(new NodeBus(this, scope)));
        }

        if (nodeAliases != null) {
            for (Node node : dag.getNodes()) {
                node.setAlias(nodeAliases.get(node.getName()));
            }
        }

        this.subDagId = Optional.ofNullable(null);

    }

    public Dag getDag() {
        return this.dag;
    }

    public Map<String, String> getProperties() {
        return this.properties;
    }

    public String getSubDagId() {
        return subDagId.orElse(null);
    }

    public void setSubDagId(String subDagId) {
        this.subDagId = Optional.ofNullable(subDagId);
    }

    public List<Map.Entry<Scope, Bus<Node>>> getBusEntries() {
        return this.dagCleaners.entrySet().stream().map(e -> new AbstractMap.SimpleImmutableEntry<>(e.getKey(), e.getValue().getBus())).collect(Collectors.toList());
    }

    public EnumMap<Scope, DagCleaner> getDagCleaners() {
        return this.dagCleaners;
    }
    
    public DagCleaner getDagCleaner(Scope scope) {
        return this.dagCleaners.get(scope);
    }
    
    public ThrowingBiFunction<Scope, Node, NodeCleaner.NodeCleaningResult, NodeCleaner.NodeCleanerException> getCleaningFunction() {
        return this.cleaningFunction;
    }

    public void setCleaningFunction(ThrowingBiFunction<Scope, Node, NodeCleaner.NodeCleaningResult, NodeCleaner.NodeCleanerException> cleaningFunction) {
        this.cleaningFunction = cleaningFunction;
    }
    
    public NodeCleaningResult clean(Scope scope, Node node) throws NodeCleanerException {
        if (node == null) {
            throw new NodeCleanerException("no enclosing node");
        }
        if (cleaningFunction == null) {
            throw new NodeCleanerException("no cleaningFunction defined in enclosing dag");
        }
        return cleaningFunction.apply(scope, node);
    }

    final public class DagCleaner {
        final private Bus<Node>              bus;
        final private boolean                failFast;
        final private boolean                treatErrorsAsWarnings;
        final private Map<Node, NodeCleaner> nodeCleanerMap;

        private Journal         journal;
        private Subscription    subscription;

        public DagCleaner(Bus<Node> bus) {
            this.bus = bus;

            this.failFast               = properties.get("failFast")              == null ? true  : Boolean.valueOf(properties.get("failFast"));
            this.treatErrorsAsWarnings  = properties.get("treatErrorsAsWarnings") == null ? false : Boolean.valueOf(properties.get("treatErrorsAsWarnings"));
            
            this.nodeCleanerMap = new HashMap<>();
            for (Node node : dag.getNodes()) {
                nodeCleanerMap.put(node, new NodeCleaner(this.bus.getScope(), node, NodeState.UNKNOWN));
            }
        }

        public Bus<Node> getBus() {
            return this.bus;
        }

        public NodeCleaner getNodeCleaner(Node node) {
            return nodeCleanerMap.get(node);
        }

        public void cleanAll() {
            cleanFromInclusive(null);
        }

        public void cleanFromInclusive(String onlyThisNodeAndSuccessors) {
            cleanFromInclusive(onlyThisNodeAndSuccessors, null);
        }

        public void cleanFromInclusive(String onlyThisNodeAndSuccessors, String subDagKey) {
            if (getCleaningFunction() == null) {
                throw new RuntimeException("No cleaner defined. Check configuration.");
            }
            if (this.journal == null) {
                this.journal = new Journal();
            } else {
                if (this.journal.isRunning()) {
                    throw new RuntimeException("Cleaning already running. Wait for completion or cancel it before retrying.");
                }
            }
            final Set<String> nodesToInclude = dag.getNodes(subDagKey).stream().map(n -> n.getName()).collect(Collectors.toSet());
            synchronized (journal) {
                this.journal.raz();
                this.journal.write("start first round");

                Node startNode = null;
                if (onlyThisNodeAndSuccessors != null) {
                    List<Node> nodesNamedAs = DagOnSteroids.this.dag.getNodesFromName(onlyThisNodeAndSuccessors);
                    if (nodesNamedAs.size() == 0) {
                        throw new RuntimeException("Cannot start cleaning: node /" + onlyThisNodeAndSuccessors + "/ not found");
                    } else if (nodesNamedAs.size() == 1) {
                        startNode = nodesNamedAs.get(0);
                    } else {
                        throw new RuntimeException("Cannot start cleaning: more than one node found with name /" + onlyThisNodeAndSuccessors + "/");
                    }
                }

                // smudge nodes after startNode (or all nodes, if no startNode)
                // (or all subdag nodes, if subDagKey)
                // DO NOT smudge non-SNAPSHOT nodes
                SortedSet<String> successorsOf = null;
                for (Node node : dag.getTopologicalNodes()) {
                    final String branch = node.getValue() != null ? node.getValue().getBranch() : null;
                    if (startNode != null) {
                        
                        if (successorsOf == null) { // not found yet
                            if (node.getName().equals(startNode.getName())) { // found !
                                successorsOf = dag.successorsOf(startNode)
                                                                .stream()
                                                                .map(n -> n.getName())
                                                                .collect(Collectors.toCollection(() -> new TreeSet<String>()));
                            } else {
                                bus.send(MAGIC_CLEAN, node.getName(), branch, SIMPLENAME);
                                explainMagicClean(node, "topologically before start node", "(startnode=/" + startNode.getName() + "/)");
                                continue;
                            }
                        }
                    }
                    if ( node.getValue()             != null    && 
                         node.getValue().getGucrid() != null    && 
                         !node.getValue().getGucrid().contains("SNAPSHOT")) {
                        // force CLEAN state if not a snapshot
                        bus.send(MAGIC_CLEAN, node.getName(), branch, SIMPLENAME);
                        explainMagicClean(node, "not a snapshot", node.getValue().getGucrid().toString());
                    } else if (!nodesToInclude.contains(node.getName())){
                        // force CLEAN state if node is not included (e.g. sub dags)
                        bus.send(MAGIC_CLEAN, node.getName(), branch, SIMPLENAME);
                        explainMagicClean(node, "not in sub dag", nodesToInclude.toString());
                    } else if (startNode    != null                         &&
                               successorsOf != null                         &&
                               !startNode.getName().equals(node.getName())  &&
                               !successorsOf.contains(node.getName())) {
                        // force CLEAN state if not a successor of start node
                        bus.send(MAGIC_CLEAN, node.getName(), branch, SIMPLENAME);
                        explainMagicClean(node, "not start node or one of its successors", "(startnode=/" + startNode.getName() + "/)");
                    } else {
                        // SMUDGE everything else
                        bus.send(SMUDGE, node.getName(), branch, SIMPLENAME);
                    }
                    journal.write(String.format("set node /%s/ state to %s",
                                                node.getName(),
                                                getNodeCleaner(node).getState()
                    ));
                }

                bus.toObservable().filter(
                        event -> event.getType().equals(CLEAN_STARTED)  || 
                                 event.getType().equals(CLEAN_OK)       || 
                                 event.getType().equals(CLEAN_ERROR)    || 
                                 event.getType().equals(CLEAN_ABORTED)).subscribe(event -> {
                            Node node = event.get();
                            if (journal != null) {
                                journal.write(String.format("received event %s on node /%s/ from [ %s ]",
                                        event.getType(),
                                        node.getName(),
                                        event.getSource()
                                ));
                                if (!journal.state.equals(State.RUNNING)) {
                                    journal.write(String.format("Cleaning is not running (state=%s), doing nothing.",
                                            journal.state.toString()
                                    ));
                                    return;
                                }
                            }

                            if (event.getType().equals(CLEAN_OK)) {

                                queue();

                            } else if (event.getType().equals(CLEAN_ERROR) || event.getType().equals(CLEAN_ABORTED)) {

                                error(event.get());

                                if (treatErrorsAsWarnings) {
                                    queue();
                                }
                            }

                            ifAllNodesAreCleanThenDone();
                        });

                queue();

                this.journal.write("first round done");
            }
        }

        private void explainMagicClean(Node node, String reason, String optional) {
            journal.write(String.format("set node /%s/ state to %s: %s%s",                     
                          node.getName(),                   
                          CLEAN,
                          reason,                           
                          optional != null ? (" - " + optional) : ""
            ));
        }

        public boolean isRunning() {
            if (journal == null) {
                return false;
            } else {
                return journal.isRunning();
            }
        }

        public String getJournalAsString() {
            if (journal != null) {
                return journal.toString();
            } else {
                return "nothing to read";
            }
        }

        public void done() {
            if (subscription != null) {
                subscription.unsubscribe();
                subscription = null;
            }
            if (journal != null) {
                journal.done();
            }
        }

        public void error(Node error) {
            if (failFast) {
                if (subscription != null) {
                    subscription.unsubscribe();
                    subscription = null;
                }
            }
            if (journal != null) {
                journal.error(error);
                if (failFast) {
                    journal.close();
                }
            }
        }

        public void cancel() {
            if (subscription != null) {
                subscription.unsubscribe();
                subscription = null;
            }
            if (journal != null) {
                journal.cancel();
            }
        }

        private void ifAllNodesAreCleanThenDone() {
            if (journal == null) {
                return;
            }
            if (!journal.state.equals(State.RUNNING)) {
                return;
            }
            int doneCount = 0;
            Collection<Node> nodesSubSet = dag.getNodes(getSubDagId());
            for (Node node : nodesSubSet) {
                NodeState state = getNodeCleaner(node).getState();
                if (state != null) {
                    if (state.oneOf(CLEAN, CLEANED)) {
                        doneCount++;
                    } else if (treatErrorsAsWarnings && (state.oneOf(FAIL, ABORTED, UNCLEANABLE))) {
                        doneCount++;
                    }
                }
            }
            if (doneCount == nodesSubSet.size()) {
                done();
            }
        }

        private void orderCleaningOfNode(Node node) {

            NodeCleaner.NodeCleaningResult result;
            try {
                result = clean(bus.getScope(), node);
            } catch (Exception e) {
                LOG.error("ordering cleaning of node /" + node.getName()
                        + "/ failed. (caught exception is not rethrown but cleanNode function will return not zero + exception message)", e);
                result = NodeCleaner.NodeCleaningResult.fromException(bus.getScope(), e);
            }

            final String branch = node.getValue() != null ? node.getValue().getBranch() : null;
            if (result != null && result.asBoolean()) {
                bus.send(CLEAN_ORDER_OK, node.getName(), branch, SIMPLENAME);
                if (journal != null) {
                    journal.write(String.format("ordered cleaning of node /%s/", node.getName()));
                }
            } else {
                String error = (result == null ? "null result" : result.getStatus() + " " + result.getReason());
                bus.send(CLEAN_ORDER_ERROR, node.getName(), branch, error);
                if (journal != null) { 
                    journal.write(String.format("cannot order cleaning of node /%s/: %s [%s]",
                            node.getName(),
                            (result == null ? "null result" : result.getStatus() + " " + result.getReason()),
                            result.getSource()
                    ));            
                }
            }
        }

        private void queue() {
            if (journal != null)
                journal.write("start inpecting queue");
            for (Node node : dag.getTopologicalNodes()) {
                final NodeState thisState = getNodeCleaner(node).getState();
                // check only dirty nodes
                if (!thisState.oneOf(DIRTY)) {
                    continue;
                }
                // inspect predecessors
                List<Node> predecessors = dag.predecessorsOf(node);
                int doneCount = 0;
                int errorCount = 0;
                for (Node pred : predecessors) {
                    final NodeState predState = getNodeCleaner(pred).getState();
                    if (predState != null) {
                        if (predState.oneOf(CLEAN, CLEANED)) {
                            doneCount++;
                        } else if (treatErrorsAsWarnings && (predState.oneOf(FAIL, ABORTED, UNCLEANABLE))) {
                            doneCount++;
                            errorCount++;
                        }
                    }
                }
                // all predecessors are clean
                if (doneCount == predecessors.size()) {
                    if (journal != null) {
                        journal.write(String.format("node /%s/ is %s and all its predecessors (%d) are DONE (with %d error(s)) : time to clean it, isn't it?",
                                node.getName(),
                                thisState,
                                doneCount,
                                errorCount
                        ));
                    }
                    orderCleaningOfNode(node);
                }
            }
            if (journal != null)
                journal.write("done inpecting queue");
        }

        class Journal implements Closeable {

            private State state;
            private ByteArrayOutputStream baos;
            private BufferedWriter writer;
            private String results;

            Journal() {
                raz();
            }

            void raz() {
                this.state = State.RUNNING;
                this.results = null;
                this.baos = new ByteArrayOutputStream();
                this.writer = new BufferedWriter(new OutputStreamWriter(baos, StandardCharsets.UTF_8));
            }

            void write(String entry) {
                try {
                    LOG.debug(entry);

                    writer.newLine();
                    writer.write(FMT.format(new Date()));
                    writer.write(" (");
                    writer.write(bus.getScope().toString());
                    writer.write(")");
                    writer.write(" [");
                    writer.write(Thread.currentThread().getName());
                    writer.write("] ");
                    writer.write(entry);
                    writer.flush();
                } catch (IOException e) {
                    LOG.warn("ignored exception {}", e.getMessage());
                }
            }

            boolean isRunning() {
                return state.equals(State.RUNNING);
            }

            boolean isDone() {
                return state.equals(State.DONE);
            }

            void done() {
                write("done!");
                this.state = State.DONE;
                close();
            }

            boolean isCanceled() {
                return state.equals(State.CANCELLED);
            }

            void cancel() {
                write("cancelled by external action");
                this.state = State.CANCELLED;
                close();
            }

            void error(Node node) {
                write("node [ " + node.getName() + " ] in error");
            }

            @Override
            public void close() {
                for (Node node : dag.getTopologicalNodes()) {
                    write("node /" + node.getName() + "/ state is " + getNodeCleaner(node).toString());
                }
                this.results = toString();
                try {
                    this.writer.close();
                } catch (IOException e) {
                    LOG.warn("ignored exception {}", e.getMessage());
                }
            }

            @Override
            public String toString() {
                if (results != null) {
                    return results;
                }
                return this.state.toString() + new String(baos.toByteArray(), StandardCharsets.UTF_8);
            }
        }

    }

    static public List<Node> getNodesFromNamedAndBranchContains(final Dag dag, final String nodeName, final String branchSubstring) {
        final List<Node> ret = new LinkedList<>();
        
        for (Node node : dag.getNodesFromName(nodeName)) {
            if (!containsBranchSubstring(node, branchSubstring)) {
                continue;
            }
            ret.add(node);
        }
        return ret;
    }

    static private boolean containsBranchSubstring(final Node node, final String branchSubstring) {
        final String nodeBranch;
        
        if (node                        != null &&
            node.getValue()             != null &&
            node.getValue().getBranch() != null) {
            nodeBranch = node.getValue().getBranch(); 
        } else {
            nodeBranch = null;
        }
        
        if (nodeBranch == null && branchSubstring == null) {
            return true;
        }
        
        if (nodeBranch == null && branchSubstring == null) {
            return false;
        }
        
        return nodeBranch.contains(branchSubstring);
    }

    static final private Pattern DAGNAMECONTAINSSUB = Pattern.compile("(.*)%2Fsubs%2(.*)");
    static public String[] parseName(String dagName) {
        // e.g. development.neo.ondemand.com%2Fsubs%2F124966568
        // to test regexps : http://www.regexplanet.com/advanced/java/index.html
        Matcher dagNameContainsSub = DAGNAMECONTAINSSUB.matcher(dagName);
        if (dagNameContainsSub.matches()) {
            return new String[] {dagNameContainsSub.group(1), dagNameContainsSub.group(2)};
        } else {
            return new String[] {dagName, null};
        }
    }

    @JsonIgnoreProperties(ignoreUnknown=true)
    public static class NodeCleaner {
        
        @JsonProperty
        private UUID         id;

        @JsonProperty
        final private Node   node;
        @JsonProperty
        final private Scope  scope;
        
        @JsonProperty
        private NodeState    state;
        
        @JsonIgnore
        private Long         start;
        @JsonIgnore
        private Long         stop;
        
        @JsonCreator
        public NodeCleaner(
                @JsonProperty(value="scope") final Scope scope,
                @JsonProperty(value="node")  final Node  node,
                @JsonProperty(value="state")  final NodeState state) {
            this.node  = node;
            this.scope = scope;
            this.state = state;
        }

        public String getId() {
            return this.id == null ? null : this.id.toString();
        }

        public void setId(String id) {
            if (id == null) {
                this.id = null;
                if (this.start != null) {
                    this.stop = new Date().getTime();
                }
            } else {
                this.start = new Date().getTime();
                this.stop  = null;
                if (id.matches("[0-9a-f]{8}-[0-9a-f]{4}-[1-5][0-9a-f]{3}-[89ab][0-9a-f]{3}-[0-9a-f]{12}")) {
                    this.id = UUID.fromString(id);
                }
            }
        }
        
        @JsonIgnore
        public Node getNode() {
            return node;
        }

        public NodeState getState() {
            return state;
        }

        public void setState(NodeState state) {
            this.state = state;
        }
        
        public Long getDuration() {
            if (start == null) {
                return null;
            } else if (stop == null) {
                return new Date().getTime() - start;
            }
            return stop - start;
        }
        
        public enum NodeState {
            // at start
            @JsonProperty("CLEAN")              UNKNOWN, 
            @JsonProperty("CLEAN")              CLEAN, 
            @JsonProperty("DIRTY")              DIRTY, 
            // doing something 
            @JsonProperty("CLEANING_ORDERED")   CLEANING_ORDERED,
            @JsonProperty("BEING_CLEANED")      BEING_CLEANED,
            // at end
            @JsonProperty("UNCLEANABLE")        UNCLEANABLE,
            @JsonProperty("CLEANED")            CLEANED,
            @JsonProperty("FAIL")               FAIL,
            @JsonProperty("ABORTED")            ABORTED;
            
            public static String gimmethestuff() {
                return Joiner.on(" ").join(values()).toString();
            }
        
            public boolean oneOf(NodeState ... others) {
                for (NodeState other : others) {
                    if (this.equals(other)) {
                        return true;
                    }
                }
                return false;
            }
        }

        public static class NodeCleanerException extends Exception {
        
            private static final long serialVersionUID = -3468108554682441131L;
            
            final public URI source;
        
            public NodeCleanerException() {
                this((URI)null);
            }
        
            public NodeCleanerException(String arg0, Throwable arg1, boolean arg2, boolean arg3) {
                this((URI)null, arg0, arg1, arg2, arg3);
            }
        
            public NodeCleanerException(String arg0, Throwable arg1) {
                this((URI)null, arg0, arg1);
            }
        
            public NodeCleanerException(String arg0) {
                this((URI)null, arg0);
            }
        
            public NodeCleanerException(Throwable arg0) {
                this((URI)null, arg0);
            }
        
            public NodeCleanerException(URI source, String arg0, Throwable arg1, boolean arg2, boolean arg3) {
                super(addSourceToMessage(arg0, source), arg1, arg2, arg3);
                this.source = source;
            }
        
            public NodeCleanerException(URI source, String arg0, Throwable arg1) {
                super(addSourceToMessage(arg0, source), arg1);
                this.source = source;
            }
        
            public NodeCleanerException(URI source, String arg0) {
                super(addSourceToMessage(arg0, source));
                this.source = source;
            }
        
            public NodeCleanerException(URI source, Throwable arg0) {
                super(sourceAsMessage(source), arg0);
                this.source = source;
            }
        
            public NodeCleanerException(URI source) {
                super(sourceAsMessage(source));
                this.source = source;
            }
            
            static private final String addSourceToMessage(String message, URI source) {
                return message + (source!=null ? " - source: " + source.toString() : "");
            }
            
            static private final String sourceAsMessage(URI source) {
                return (source!=null ? "source: " + source.toString() : "");
            }
        
        }

        @JsonIgnoreProperties(ignoreUnknown=true)
        public static class NodeCleaningResult {
            private final Scope  scope;
            private final int    status;
            private final String reason;
            
            private String source;
            
            @JsonCreator
            public NodeCleaningResult(
                    @JsonProperty(value="bus")    Scope scope, 
                    @JsonProperty(value="status") int status, 
                    @JsonProperty(value="reason") String reason) {
                this.scope  = scope;
                this.status = status;
                this.reason = reason;
            }
            public Scope getScope() {
                return scope;
            }
            public int getStatus() {
                return status;
            }
            public String getReason() {
                return reason;
            }
            public String getSource() {
                return source;
            }
            public void setSource(String source) {
                this.source = source;
            }
            @JsonIgnore
            public NodeCleaningResult source(String source) {
                setSource(source);
                return this;
            }
            @JsonIgnore
            public boolean asBoolean() {
                return (this.status == 0 || this.status/100 == 2);
            }
            
            static public NodeCleaningResult undefined(Scope c) {
                return new NodeCleaningResult(c, -1, "undefined");
            }
            public static NodeCleaningResult fromException(Scope c, Exception e) {
                return new NodeCleaningResult(c, -1, e.getMessage());
            }
            public static NodeCleaningResult ok(Scope c) {
                return new NodeCleaningResult(c, 0, "");
            }
            public static NodeCleaningResult from(Scope c, int status, String reason) {
                return new NodeCleaningResult(c, status, reason);
            }
        }

    }
    
}