package net.aequologica.neo.dagr;

import static com.fasterxml.jackson.annotation.JsonInclude.Include.NON_NULL;
import static com.fasterxml.jackson.databind.SerializationFeature.INDENT_OUTPUT;
import static com.google.common.net.MediaType.JSON_UTF_8;

import java.io.ByteArrayInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

import javax.inject.Singleton;

import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.collect.HashMultimap;
import com.google.common.collect.Lists;
import com.google.common.collect.SetMultimap;

import net.aequologica.neo.dagr.jgrapht.DagJGraphT;
import net.aequologica.neo.dagr.model.Dag;
import net.aequologica.neo.dagr.model.Dag.Node;
import net.aequologica.neo.dagr.model.DagDocumentSerializer;
import net.aequologica.neo.geppaequo.cmis.ECMHelper.Stream;
import net.aequologica.neo.geppaequo.document.DocumentHelper;

@Singleton
public class Dags {

    private static DagDocumentSerializer serializer = new DagDocumentSerializer();

    private final Map<String, Dag>         dagmap;
    private final Map<String, DagJGraphT>  graphmap;

    private final SetMultimap<String, String> user2dags;
    private final SetMultimap<String, String> dag2users;

    private final Path         userDagTuplesPath;
    private final ObjectMapper mapper;
    
    private final Lock lockOnIO = new ReentrantLock();

    public Dags() {
        this.userDagTuplesPath = Paths.get("/.dagr/userdagtuples.json");
        this.mapper            = new ObjectMapper();
                
        this.mapper.enable(INDENT_OUTPUT);
        this.mapper.setSerializationInclusion(NON_NULL);
        
        this.dagmap    = new HashMap<>();
        this.graphmap  = new HashMap<>();
        this.user2dags = HashMultimap.create();
        this.dag2users = HashMultimap.create();
        
        load();
    }
    /////////////////////////////

    public Collection<Dag> getDAGs() {
        return this.dagmap.values();
    }

    public Set<String> getDAGKeys() {
        return this.dagmap.keySet();
    }

    public Dag getDAG(final String dagkey) {
        return this.dagmap.get(dagkey);
    }

    public DagJGraphT getDagJGraphT(final String dagkey) {
        return this.graphmap.get(dagkey);
    }
    
    public List<Dag> getUserDAGs(final String username) {
        Set<String> thisUserDags = user2dags.get(username);
        final List<Dag> ret = new ArrayList<>(thisUserDags.size());
        for (String dagkey : thisUserDags) {
            final Dag dag = this.dagmap.get(dagkey);
            if (dag != null) {
                ret.add(dag);
            }
        }
        return ret;
    }
    
    public List<Node> getNodesByScmAndBranch(String scm, String branch) {
        List<Node> ret = new LinkedList<>();
        
        Collection<Dag> dags = this.dagmap.values();
        for (Dag dag : dags) {
        
            List<Node> nodes = dag.getNodes();
            for (Node node : nodes) {
                if (node.getValue() == null ||
                    node.getValue().getScm() == null ||
                    node.getValue().getBranch() == null) {
                    continue;
                }
                if (-1 == node.getValue().getScm().indexOf(scm)) {
                    continue;
                }
                if (-1 == node.getValue().getBranch().indexOf(branch)) {
                    continue;
                }
                ret.add(node);
             }
        }
        return ret;
    }

    public List<Map.Entry<String, String>> load() {
        
        this.lockOnIO.lock(); // block until condition holds
        try {
            final Path path = Paths.get("/dags");
            final List<Map.Entry<String, String>> exceptions = Lists.newArrayList();

            this.dagmap.clear();
            this.graphmap.clear();

            List<java.nio.file.Path> sources;
            try {
                sources = DocumentHelper.list(path);
            } catch (Exception e) {
                exceptions.add(new AbstractMap.SimpleImmutableEntry<String, String>(path.toString(), e.getClass().getSimpleName()+" - " +e.getMessage()));
                return exceptions;
            }

            for (java.nio.file.Path source : sources) {
                try {
                    String lowerCase = source.getFileName().toString().toLowerCase();
                    if (lowerCase.endsWith(userDagTuplesPath.getFileName().toString())) {
                        continue;
                    }
                    Dag tmpDag = serializer.read(source);
                    if (tmpDag == null) {
                        continue;
                    }
                    Dag dag = new DagJGraphT(tmpDag).detectAndFlagTransitiveEdges();
                    dag.setSource(source.toString());
                    String dagkey = source.getFileName().toString();
                    dag.setKey(dagkey);

                    dagmap.put(dagkey, dag);
                    graphmap.put(dagkey, new DagJGraphT(dag));

                } catch (Exception e) {
                    exceptions.add(new AbstractMap.SimpleImmutableEntry<String, String>(source.toString(), e.getClass().getSimpleName()+" - " +e.getMessage()));
                }
            }

            return exceptions;
        } finally {
            this.lockOnIO.unlock();
        }

    }

    public static void dumpDag(Path path, Dag dag) throws IOException {
        serializer.write(path, dag);
    }


}
