package net.aequologica.neo.dagr;

import static com.fasterxml.jackson.annotation.JsonInclude.Include.NON_NULL;
import static com.fasterxml.jackson.databind.SerializationFeature.INDENT_OUTPUT;
import static com.google.common.net.MediaType.JSON_UTF_8;

import java.io.ByteArrayInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.collect.HashMultimap;
import com.google.common.collect.Lists;
import com.google.common.collect.SetMultimap;

import net.aequologica.neo.dagr.jgrapht.DagJGraphT;
import net.aequologica.neo.dagr.model.Dag;
import net.aequologica.neo.dagr.model.Dag.Node;
import net.aequologica.neo.dagr.model.DagDocumentSerializer;
import net.aequologica.neo.geppaequo.cmis.ECMHelper.Stream;
import net.aequologica.neo.geppaequo.document.DocumentHelper;

public class Dags {

    private static DagDocumentSerializer serializer = new DagDocumentSerializer();

    private final Map<String, Dag>         dagmap;
    private final Map<String, DagJGraphT>  graphmap;

    private final SetMultimap<String, String> user2dags;
    private final SetMultimap<String, String> dag2users;

    private final Path         userDagTuplesPath;
    private final ObjectMapper mapper;
    
    private final Lock lockOnIO = new ReentrantLock();

    /////////////////////////////
    // hand-made singleton
    // cf. https://en.wikipedia.org/wiki/Initialization-on-demand_holder_idiom for the singleton pattern used    
    private static class LazyHolder {
        private static final Dags INSTANCE = new Dags();
        static {
            INSTANCE.load();
        }
    }

    public static Dags getInstance() {
        return LazyHolder.INSTANCE;
    }

    private Dags() {
        super();
        this.userDagTuplesPath = Paths.get("/.dagr/userdagtuples.json");
        this.mapper            = new ObjectMapper();
                
        this.mapper.enable(INDENT_OUTPUT);
        this.mapper.setSerializationInclusion(NON_NULL);
        
        this.dagmap    = new HashMap<>();
        this.graphmap  = new HashMap<>();
        this.user2dags = HashMultimap.create();
        this.dag2users = HashMultimap.create();

    }
    /////////////////////////////

    public Collection<Dag> getDAGs() {
        return this.dagmap.values();
    }

    public Set<String> getDAGKeys() {
        return this.dagmap.keySet();
    }

    public Dag getDAG(final String dagkey) {
        return this.dagmap.get(dagkey);
    }

    public DagJGraphT getDagJGraphT(final String dagkey) {
        return this.graphmap.get(dagkey);
    }
    
    public List<Dag> getUserDAGs(final String username) {
        Set<String> thisUserDags = user2dags.get(username);
        final List<Dag> ret = new ArrayList<>(thisUserDags.size());
        for (String dagkey : thisUserDags) {
            final Dag dag = this.dagmap.get(dagkey);
            if (dag != null) {
                ret.add(dag);
            }
        }
        return ret;
    }
    
    public List<Node> getNodesByRepoFullnameAndRef(String repo_fullname, String ref) {
        List<Node> ret = new LinkedList<>();
        
        Collection<Dag> dags = this.dagmap.values();
        for (Dag dag : dags) {
        
            List<Node> nodes = dag.getNodes();
            for (Node node : nodes) {
                if (node.getValue() == null ||
                    node.getValue().getScm() == null ||
                    node.getValue().getBranch() == null) {
                    continue;
                }
                if (-1 == node.getValue().getScm().indexOf(repo_fullname)) {
                    continue;
                }
                if (-1 == node.getValue().getBranch().indexOf(ref)) {
                    continue;
                }
                ret.add(node);
             }
        }
        return ret;
    }

    public List<Map.Entry<String, String>> load() {
        
        this.lockOnIO.lock(); // block until condition holds
        try {
            final Path path = Paths.get("/dags");
            final List<Map.Entry<String, String>> exceptions = Lists.newArrayList();

            this.dagmap.clear();
            this.graphmap.clear();

            List<java.nio.file.Path> sources;
            try {
                sources = DocumentHelper.list(path);
            } catch (Exception e) {
                exceptions.add(new AbstractMap.SimpleImmutableEntry<String, String>(path.toString(), e.getClass().getSimpleName()+" - " +e.getMessage()));
                return exceptions;
            }

            for (java.nio.file.Path source : sources) {
                try {
                    String lowerCase = source.getFileName().toString().toLowerCase();
                    if (lowerCase.endsWith(userDagTuplesPath.getFileName().toString())) {
                        continue;
                    }
                    Dag tmpDag = serializer.read(source);
                    if (tmpDag == null) {
                        continue;
                    }
                    Dag dag = new DagJGraphT(tmpDag).detectAndFlagTransitiveEdges();
                    dag.setSource(source.toString());
                    String dagkey = source.getFileName().toString();
                    dag.setKey(dagkey);

                    dagmap.put(dagkey, dag);
                    graphmap.put(dagkey, new DagJGraphT(dag));

                } catch (Exception e) {
                    exceptions.add(new AbstractMap.SimpleImmutableEntry<String, String>(source.toString(), e.getClass().getSimpleName()+" - " +e.getMessage()));
                }
            }
            
            try {
                readUserDagTuples(userDagTuplesPath);
            } catch (IOException e) {
                exceptions.add(new AbstractMap.SimpleImmutableEntry<String, String>(userDagTuplesPath.toString(), e.getClass().getSimpleName()+" - " +e.getMessage()));
            }
            
            return exceptions;
        } finally {
            this.lockOnIO.unlock();
        }

    }

    public boolean subscribe(String dagkey, String username) throws IOException {
        Dag dag = this.dagmap.get(dagkey);
        if (dag == null) {
            throw new IOException("["+dagkey+"] not found");
        }
        
        synchronized (this.user2dags) {
            synchronized (this.dag2users) {
                user2dags.put(username, dagkey);
                dag2users.put(dagkey, username);
            }
        }

        writeUserDagTuples(userDagTuplesPath);

        return true;
    }

    public boolean unsubscribe(String dagkey, String username) throws IOException {
        Dag dag = dagmap.get(dagkey);
        if (dag == null) {
            throw new IOException("["+dagkey+"] not found");
        }
        
        synchronized (this.user2dags) {
            synchronized (this.dag2users) {
                user2dags.remove(username, dagkey);
                dag2users.remove(dagkey, username);
            }
        }
        
        writeUserDagTuples(userDagTuplesPath);

        return true;
    }

    public Boolean isUserSubscribed(String dagkey, String username) {
        Collection<String> existingUsers = dag2users.get(dagkey);
        return existingUsers.contains(username);
    }

    private void writeUserDagTuples(Path path) throws JsonProcessingException, IOException {
        this.lockOnIO.lock();
        try {
            List<UserDagTuple> userdags = new ArrayList<>();
            Collection<Entry<String, String>> entries = user2dags.entries();
            for (Entry<String, String> entry : entries) {
                userdags.add(new UserDagTuple(entry.getKey(), entry.getValue()));
            }
            final byte[] bytes = this.mapper.writeValueAsBytes(userdags); // "Encoding used will be UTF-8."
            DocumentHelper.write(path, new Stream() {
                @Override public String      getMimeType()      { return JSON_UTF_8.toString(); }
                @Override public long        getLength()        { return bytes.length; }
                @Override public InputStream getInputStream()   { return new ByteArrayInputStream(bytes); }
            });
        } finally {
            this.lockOnIO.unlock();
        }
    }

    private void readUserDagTuples(final Path path) throws IOException {
        this.lockOnIO.lock();
        try {
            this.user2dags.clear();
            this.dag2users.clear();

            final InputStream inputStream = DocumentHelper.getInputStream(path);
            if (inputStream == null) {
                throw new FileNotFoundException(path.toString());
            }
            final List<UserDagTuple> userdags = this.mapper.readValue(inputStream, new TypeReference<List<UserDagTuple>>() {});
            
            for (UserDagTuple tuple : userdags) {
                user2dags.put(tuple.user, tuple.dag);
                dag2users.put(tuple.dag, tuple.user);
            }
        } finally {
            this.lockOnIO.unlock();
        }
    } 

    private static class UserDagTuple {
        @JsonProperty
        String user;
        @JsonProperty
        String dag;
        
        private UserDagTuple() {
        }
        
        private UserDagTuple(final String user, final String dag) {
            this.user = user;
            this.dag = dag;
        }
    }

    public static void dumpDag(Path path, Dag dag) throws IOException {
        serializer.write(path, dag);
    }


}
