package net.abstractfactory.plum.view.component.containers.window;

import java.util.List;
import java.util.concurrent.BlockingDeque;
import java.util.concurrent.Callable;
import java.util.concurrent.LinkedBlockingDeque;

import net.abstractfactory.common.ApplicationContextUtils;
import net.abstractfactory.common.Context;
import net.abstractfactory.plum.repository.biz.TransactionExecutor;
import net.abstractfactory.plum.view.context.ViewSessionContextUtils;
import net.abstractfactory.plum.view.event.ComponentEvent;
import net.abstractfactory.plum.view.event.Event;
import net.abstractfactory.plum.view.event.StopThreadEvent;
import net.abstractfactory.plum.view.event.TimeToRefreshViewEvent;

import org.apache.log4j.Logger;

/**
 * execute tasks in queue.
 * 
 * 
 * @author jackding
 * 
 */
public class WindowEventThread extends Thread {
	private final Logger logger = Logger.getLogger(WindowEventThread.class);

	private Window window;
	private EventThreadContext context;

	private BlockingDeque<Event> queue = new LinkedBlockingDeque<Event>();

	/**
	 * the suspending thread, which is waiting for this thread to end.
	 */
	private WindowEventThread previousEventThread;

	public WindowEventThread(Window window,
			WindowEventThread previousEventThread) {
		this.window = window;

		this.previousEventThread = previousEventThread;

		Context parentContext = null;
		if (previousEventThread != null)
			parentContext = previousEventThread.getContext();

		context = new EventThreadContext(parentContext);

		setName("EventThread for '" + window.getTitle() + "'");

	}

	@Override
	public void run() {
		while (true) {

			Event event;
			try {
				event = queue.take();
				if (event instanceof ComponentEvent) {

					final ComponentEvent e = (ComponentEvent) event;

					if (logger.isDebugEnabled())
						logger.debug(String.format(
								"start execute event action [%s] %s.%s", e
										.getComponent().getClass()
										.getSimpleName(), e.getComponent()
										.getName(), e.getAction().getName()));

					TransactionExecutor transactionExecutor = ApplicationContextUtils
							.getTransactionExecutor();

					// an do-nothing executor.
					if (transactionExecutor == null)
						transactionExecutor = new TransactionExecutor();

					transactionExecutor.execute(new Callable<Void>() {
						@Override
						public Void call() throws Exception {

							try {
								e.execute();
							} catch (Throwable e) {
								e.printStackTrace();
								logger.error(e);
							}
							return null;
						}
					});

					if (logger.isDebugEnabled())
						logger.debug(String.format(
								"end execute event action [%s] %s.%s", e
										.getComponent().getClass()
										.getSimpleName(), e.getComponent()
										.getName(), e.getAction().getName()));

				} else if (event instanceof TimeToRefreshViewEvent) {

					if (window.isVisible()) {
						Context sessionContext = EventThreadContext
								.getCurrentThreadContext().getSessionContext();
						
						ViewRefreshTask refreshViewFutureTask = ViewSessionContextUtils
								.takeRefreshViewFutureTask(sessionContext);

						if (refreshViewFutureTask != null) {
							logger.debug(String.format(
									"%s thread execute refresh view task.",
									window.getTitle()));

							refreshViewFutureTask.run();
							
						}
						else
							logger.debug("skip refresh view because there is no ViewRefreshTask in session.");
							
					}else
					{
						logger.debug("skip refresh view because current window is not visible.");
					}
				} else if (event instanceof StopThreadEvent) {
					logger.info("stop thread: " + getName());
					break;
				}  
			} catch (InterruptedException e) {
				logger.debug("window thread exiting...");
				break;
			} catch (Exception e) {
				logger.error(e);
			}

		}
	}

	public synchronized void addEvent(Event e) {
		queue.add(e);
	}

	public synchronized void addEvents(List<Event> events) {
		queue.addAll(events);
	}

	/**
	 * stop the thread.
	 */
	public synchronized void stopThread() {
		queue.add(new StopThreadEvent());
	}

	public Window getWindow() {
		return window;
	}

	public EventThreadContext getContext() {
		return context;
	}

	public WindowEventThread getPreviousEventThread() {
		return previousEventThread;
	}
}
