package net.abstractfactory.plum.view.component.containers.window;

import java.util.ArrayList;
import java.util.List;

import net.abstractfactory.plum.view.Visitor;
import net.abstractfactory.plum.view.component.Button;
import net.abstractfactory.plum.view.component.ClickEventListener;
import net.abstractfactory.plum.view.component.Component;
import net.abstractfactory.plum.view.component.Label;
import net.abstractfactory.plum.view.component.containers.Panel;

import org.apache.log4j.Logger;

/**
 * Window is a window, it has its own Z index, its title. <br>
 * window could be a dialog, or MDI child/parent.
 * 
 * @author HZ00260
 * 
 */
public class Window extends Component {
	private static Logger logger = Logger.getLogger(Window.class);

	public static final String NAME_TITLE_PANEL = "titlePanel";
	public static final String NAME_TITLE_LABEL = "titleLabel";
	public static final String NAME_CLOSE_BUTTON = "closeButton";
	public static final String NAME_CONTENT_PANEL = "contentPanel";
	/**
	 * Window Tree are another tree, not same with the component tree; Windows
	 * display in its own Z axis.<br>
	 * in the HTML render, Z index is useful since it can not display Real modal
	 * dialog. It only display the most font window.
	 */
	protected Window parentWindow;
	protected List<Window> childWindows;

	// title panel
	protected String title;
	protected Label titleLabel;
	protected Button closeButton;
	protected Panel titlePanel;
	/**
	 * A window has only one child component, that is the contentPanel. any
	 * other component should be added as the contentPanel's child.
	 */
	protected Panel contentPanel;

	// a reference to the window manager
	protected WindowManager windowManager;

	protected WindowEventThread eventThread;

	public Window() {
		
		visible = false;

		childWindows = new ArrayList<Window>();

		init();
	
	}

	private void init() {
		titlePanel = new Panel();
		titlePanel.setName(NAME_TITLE_PANEL);

		titleLabel = new Label();
		titleLabel.setName(NAME_TITLE_LABEL);
		titleLabel.setText(title);
		titlePanel.addChild(titleLabel);

		closeButton = new Button();
		closeButton.setName(NAME_CLOSE_BUTTON);
		closeButton.setCaption("X");
		closeButton.addClickListener(new ClickEventListener() {

			@Override
			public void onClick(Component source) {
				closeWindow();
			}
		});

		titlePanel.addChild(closeButton);

		contentPanel = new Panel();
		contentPanel.setName("contentPanel");

		addChild(titlePanel);
		addChild(contentPanel);
	}

	/**
	 * close, detach, free window.
	 */
	public void closeWindow() {
		hide();
		getEventThread().stopThread();

		Window parentWindow = getParentWindow();
		if (parentWindow != null)
			parentWindow.removeChildWindow(this);

		destroy();
	}

	public Window getParentWindow() {
		return parentWindow;
	}

	public void setParentWindow(Window parentWindow) {
		// remove old parent
		if (this.parentWindow != null)
			this.parentWindow.removeChildWindow(this);

		this.parentWindow = parentWindow;

		// add new child
		if (parentWindow != null)
			parentWindow.addChildWindow(this);
	}

	public void addChildWindow(Window child) {

		child.parentWindow = this;
		childWindows.add(child);

		// register to WindowManager
		windowManager.register(child);
		child.setWindowManager(windowManager);

		// dialog has its own thread
		if (child.getEventThread() == null)
			child.setEventThread(eventThread);

		// modal dialog
		if (child instanceof Dialog) {
			Dialog dlg = (Dialog) child;
			if (dlg.isModal()) {
				windowManager.bringFront(dlg);

			}
		}
	}

	public void removeChildWindow(Window child) {
		child.parentWindow = null;
		childWindows.remove(child);

		// unregister from WindowManager
		windowManager.unregister(child);

		logger.debug("now font window is "
				+ windowManager.getVisibleFontWindow().title);
	}

	public List<Window> getChildWindows() {
		return childWindows;

	}

	public Window getRootWindow() {
		if (parentWindow == null)
			return this;
		else
			return parentWindow.getRootWindow();

	}

	public String getTitle() {
		return title;
	}

	public void setTitle(String title) {
		this.title = title;

		titleLabel.setText(title);

		// update title to thread
		if (eventThread != null)
			eventThread.setName("EventThread for '" + getTitle() + "'");

	}

	public Panel getTitlePanel() {
		return titlePanel;
	}

	public Panel getContentPanel() {
		return contentPanel;
	}

	public void setContentPanel(Panel contentPanel) {
		this.contentPanel = contentPanel;

		removeAllChildren();
		addChild(contentPanel);
	}	

	public WindowManager getWindowManager() {
		return windowManager;
	}

	public void setWindowManager(WindowManager windowManager) {
		this.windowManager = windowManager;
	}

	public void setEventThread(WindowEventThread eventThread) {
		this.eventThread = eventThread;
	}

	public WindowEventThread getEventThread() {
		return eventThread;
	}

	@Override
	public Object accept(Visitor visitor) {
		return visitor.visit(this);

	}
}
