package net.abstractfactory.plum.view.component;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import net.abstractfactory.common.TreeNode;
import net.abstractfactory.plum.view.Visitor;
import net.abstractfactory.plum.view.component.containers.window.Window;
import net.abstractfactory.plum.view.component.window.PackingCase;
import net.abstractfactory.plum.view.event.EventListener;
import net.abstractfactory.plum.view.misc.Color;
import net.abstractfactory.plum.view.misc.Font;

/**
 * Any component is a container by default. But it should not always be a
 * container.
 * 
 * 
 * @author HZ00260
 * 
 */
public abstract class Component extends TreeNode {
	/**
	 * refresh only sync abstract view to (remote) native view.
	 */
	public static final String EVENT_BEFORE_REFRESH_VIEW = "beforeRefresh";
	/**
	 * usually view should load data from model.
	 */
	public static final String EVENT_BEFORE_UPDATE_VIEW = "beforeUpdateView";
	/**
	 * any part of the view changes that the Controller may interested.
	 */
	public static final String EVENT_STATE_CHANGE = "stateChange";
	/**
	 * usually used to update model.
	 */
	public static final String EVENT_VALUE_CHANGE = "valueChange";

	/**
	 * it is unique
	 */
	protected String id;

	protected String name;
	protected String caption;

	protected Font font;
	protected Color color;
	protected Color backgroundColor;

	protected boolean disabled = false;
	protected boolean readOnly = false;
	protected boolean visible = true;

	/**
	 * extra attributes used by custom view builder. such as WebViewBuilder, it
	 * may need specific attributes to tweak GUI effect.
	 */
	private Map<String, Object> extraAttributes = new HashMap<String, Object>();

	private Map<String, List<EventListener>> eventListeners = new HashMap<String, List<EventListener>>();
	/**
	 * optional model behind the view object.
	 * 
	 * 
	 */
	private Object model;

	public Component() {

	}

	public String getName() {
		return name;
	}

	public void setName(String name) {
		this.name = name;
	}

	public String getCaption() {
		return caption;
	}

	public void setCaption(String caption) {
		this.caption = caption;
	}

	public boolean isDisabled() {
		return disabled;
	}

	public void setDisabled(boolean disabled) {
		this.disabled = disabled;
	}

	public boolean isReadOnly() {
		return readOnly;
	}

	public void setReadOnly(boolean readOnly) {
		this.readOnly = readOnly;
	}

	public boolean isVisible() {
		return visible;
	}

	public void setVisible(boolean visible) {
		this.visible = visible;
	}

	public void show() {
		setVisible(true);
	};

	public void hide() {
		setVisible(false);
	}

	public String getId() {
		return id;
	}

	public void setId(String id) {
		this.id = id;
	}

	public Font getFont() {
		return font;
	}

	public Font getFont(boolean createIfNull) {
		if (font == null && createIfNull)
			font = new Font();

		return font;
	}

	public void setFont(Font font) {
		this.font = font;
	}

	public Color getColor() {
		return color;
	}

	public void setColor(Color color) {
		this.color = color;
	}

	public Color getBackgroundColor() {
		return backgroundColor;
	}

	public void setBackgroundColor(Color backgroundColor) {
		this.backgroundColor = backgroundColor;
	}

	public Map<String, Object> getExtraAttributes() {
		return extraAttributes;
	}

	public Object getExtraAttribute(String name) {
		return extraAttributes.get(name);
	}

	public void setExtraAttribute(String name, Object value) {
		extraAttributes.put(name, value);
	}

	public void removeExtraAttribute(String name) {
		extraAttributes.remove(name);
	}

	@Override
	public String toString() {

		return String.format("%-10s%-30s%-30s%-30s", isWindow(), this
				.getClass().getSimpleName(), name, caption);
	}

	public void addChild(Component comp) {
		super.addChild(comp);

		notifyEventListeners(EVENT_STATE_CHANGE);
	}

	public void addChild(Component uic, boolean autoremovePackingCase) {
		if (autoremovePackingCase && uic instanceof PackingCase) {
			for (Component c : uic.getChildrenComponents())
				addChild(c);
		} else
			addChild(uic);
	}

	public void addChildren(Component[] uics) {
		for (Component uic : uics)
			addChild(uic);
	}

	public void removeChild(Component uic) {
		super.removeChild(uic);

		notifyEventListeners(EVENT_STATE_CHANGE);
	}

	public Component getChildComponentByName(final String name) {
		Component rslt = tranverse(new Find() {
			@Override
			public boolean find(Component uic, int depth) {
				// System.out.println("in find()," + uic.name + "," + name);

				if (uic.name != null)
					return uic.name.equals(name);
				else
					return false;

			}
		}, 0);

		if (rslt != null)
			return rslt;
		else
			throw new RuntimeException(
					"getChildComponentByName not found name:" + name);
	}

	public List<Component> getChildrenComponents() {
		List<Component> rslt = new ArrayList<Component>();
		for (TreeNode tn : getChildren()) {
			Component uic = (Component) tn;

			rslt.add(uic);
		}
		return rslt;
	}

	static interface Find {
		boolean find(Component uic, int depth);
	}

	public Component getChildComponentById(final String id) {
		Component rslt = tranverse(new Find() {
			@Override
			public boolean find(Component uic, int depth) {
				/*
				 * System.out.println("");
				 * 
				 * for(int i=0;i<depth;i++) System.out.print("  ");
				 * 
				 * System.out.print("in find()," + uic.id);
				 */

				if (uic.id != null)
					return uic.id.equals(id);
				else
					return false;

			}
		}, 0);

		if (rslt != null)
			return rslt;
		else
			throw new RuntimeException("getChildComponentById not found id:"
					+ id + " under component:" + this.id);
	}

	protected Component tranverse(Find find, int depth) {
		if (find.find(this, depth))
			return this;

		for (TreeNode tn : getChildren()) {
			Component uic = (Component) tn;
			Component rslt = uic.tranverse(find, depth + 1);
			if (rslt != null)
				return rslt;
		}

		return null;
	}

	/**
	 * find the belonging window
	 * 
	 * @return
	 */
	public Window findWindow() {
		if (isWindow())
			return (Window)this;
		else {
			Component parent = (Component)getParent();
			if (parent != null)
				return parent.findWindow();
			else
				return null;
		}
	}

	public boolean isWindow() {
		return (Window.class.isInstance(this));

	}

	/**
	 * original idea is to make a "getWindow()", but, I don't want to let this
	 * class depend on the Window class.
	 * 
	 * @param clazz
	 * @return
	 */
	protected Component getComponentByClass(Class clazz) {
		System.out.println("getComponentByClass()" + this);
		if (clazz.isInstance(this)) {
			System.out.println("getComponentByClass() -- Found");
			return this;
		} else {
			System.out.println("getComponentByClass() -- Keep looking for");
			Component p = (Component) getParent();
			if (p == null) {
				System.out
						.println("getComponentByClass() -- not found until parent null");
				return null;
			} else {
				System.out
						.println("getComponentByClass() -- try to find its parent");
				return p.getComponentByClass(clazz);
			}
		}
	}

	public Window getWindow() {

		Component uic = getComponentByClass(Window.class);
		// System.out.println("getComponentByClass() ++Found");

		if (uic != null)
			return (Window) uic;
		else {

			throw new RuntimeException("the root node is not a Window.");
		}
	}

	public Object getModel() {
		return model;
	}

	public void setModel(Object model) {
		this.model = model;
	}

	public void setModel(Object model, boolean updateView) {
		setModel(model);

		if (updateView)
			updateView();
	}

	/**
	 * sync values from view to model
	 */
	public void updateModel() {
		notifyEventListeners(EVENT_STATE_CHANGE);

		for (TreeNode tn : getChildren()) {
			Component comp = (Component) tn;
			comp.updateModel();
		}
	};

	/**
	 * sync values from model to view.
	 */
	public void updateView() {
		notifyEventListeners(EVENT_BEFORE_UPDATE_VIEW);

		for (Component child : getChildrenComponents())
			child.updateView();

	};

	public abstract Object accept(Visitor visitor);

	public void addEventListener(String event, EventListener listener) {
		List<EventListener> list = eventListeners.get(event);
		if (list == null) {
			list = new ArrayList<EventListener>();
			eventListeners.put(event, list);
		}
		list.add(listener);
	}

	public void removeEventListener(String event, EventListener listener) {
		List<EventListener> list = eventListeners.get(event);
		if (list != null) {
			list.remove(listener);
		}
	}

	public boolean existsEventListener(String event, EventListener listener) {
		List<EventListener> list = eventListeners.get(event);
		return (list != null && list.contains(listener));
	}

	public void clearEventListener() {
		eventListeners.clear();
	}

	public boolean containsListener(String event) {
		List<EventListener> list = eventListeners.get(event);
		if (list != null && !list.isEmpty()) {
			return true;
		} else
			return false;

	}

	public void notifyEventListeners(String event) {
		notifyEventListeners(event, new Object[] {});
	}

	public void notifyEventListeners(String event, Object... parameters) {
		List<EventListener> list = eventListeners.get(event);
		if (list != null) {
			Iterator<EventListener> it = list.iterator();
			while (it.hasNext()) {
				// System.out.println("handle event:"+event);
				EventListener el = it.next();
				el.process(this, event, parameters);
			}

		}
	}

	/**
	 * trigger STATE_CHANGE event.
	 */
	public void forceUpdateView() {
		notifyEventListeners(EVENT_STATE_CHANGE);
	}

	/**
	 * free resource, such as file, thread, etc.
	 */
	public void destroy() {

	}
}
