package net.abstractfactory.plum.view.web.component.builder;

import java.lang.reflect.Constructor;
import java.util.List;

import org.apache.commons.codec.digest.DigestUtils;
import org.apache.log4j.Logger;
import org.jsoup.nodes.Element;

import net.abstractfactory.common.ListValueMap;
import net.abstractfactory.common.TreeNode;
import net.abstractfactory.plum.view.ComponentVisitor;
import net.abstractfactory.plum.view.client.ViewSyncMode;
import net.abstractfactory.plum.view.component.Button;
import net.abstractfactory.plum.view.component.CheckBox;
import net.abstractfactory.plum.view.component.Component;
import net.abstractfactory.plum.view.component.DateTimePicker;
import net.abstractfactory.plum.view.component.FilePicker;
import net.abstractfactory.plum.view.component.FileView;
import net.abstractfactory.plum.view.component.HyperLink;
import net.abstractfactory.plum.view.component.ImageBox;
import net.abstractfactory.plum.view.component.ImageButton;
import net.abstractfactory.plum.view.component.Label;
import net.abstractfactory.plum.view.component.ListBox;
import net.abstractfactory.plum.view.component.ListItemView;
import net.abstractfactory.plum.view.component.ListView;
import net.abstractfactory.plum.view.component.ListView.Direction;
import net.abstractfactory.plum.view.component.RadioBox;
import net.abstractfactory.plum.view.component.RadioItem;
import net.abstractfactory.plum.view.component.SpinBox;
import net.abstractfactory.plum.view.component.TextArea;
import net.abstractfactory.plum.view.component.TextBox;
import net.abstractfactory.plum.view.component.VideoBox;
import net.abstractfactory.plum.view.component.audiobox.AudioBox;
import net.abstractfactory.plum.view.component.containers.Panel;
import net.abstractfactory.plum.view.component.containers.window.Dialog;
import net.abstractfactory.plum.view.component.containers.window.Screen;
import net.abstractfactory.plum.view.component.containers.window.Window;
import net.abstractfactory.plum.view.component.grid.Cell;
import net.abstractfactory.plum.view.component.grid.Grid;
import net.abstractfactory.plum.view.component.grid.Row;
import net.abstractfactory.plum.view.component.id.IdGenerationType;
import net.abstractfactory.plum.view.component.id.IdGenerator;
import net.abstractfactory.plum.view.component.id.IdGeneratorFactory;
import net.abstractfactory.plum.view.component.listbox.DropdownList;
import net.abstractfactory.plum.view.component.menu.Menu;
import net.abstractfactory.plum.view.component.menu.MenuBar;
import net.abstractfactory.plum.view.component.menu.MenuItem;
import net.abstractfactory.plum.view.component.tab.Tab;
import net.abstractfactory.plum.view.component.tab.Tabs;
import net.abstractfactory.plum.view.component.tree.TreeNodeView;
import net.abstractfactory.plum.view.component.tree.TreeView;
import net.abstractfactory.plum.view.component.web.HtmlView;
import net.abstractfactory.plum.view.event.EventListener;
import net.abstractfactory.plum.view.layout.Box;
import net.abstractfactory.plum.view.layout.GridLayout;
import net.abstractfactory.plum.view.layout.HorizontalBox;
import net.abstractfactory.plum.view.layout.VerticalBox;
import net.abstractfactory.plum.view.web.WebPageView;
import net.abstractfactory.plum.view.web.WebTemplatePage;
import net.abstractfactory.plum.view.web.component.AbstractTemplateWebComponent;
import net.abstractfactory.plum.view.web.component.AbstractWebComponent;
import net.abstractfactory.plum.view.web.component.WebComponent;
import net.abstractfactory.plum.view.web.component.WebHyperLink;
import net.abstractfactory.plum.view.web.component.WebLabel;
import net.abstractfactory.plum.view.web.component.WebPanel;
import net.abstractfactory.plum.view.web.component.WebTab;
import net.abstractfactory.plum.view.web.component.WebTabs;
import net.abstractfactory.plum.view.web.component.container.WebScreen;
import net.abstractfactory.plum.view.web.component.container.window.WebMainWindowTitlePanel;
import net.abstractfactory.plum.view.web.component.container.window.WebWindow;
import net.abstractfactory.plum.view.web.component.container.window.dialog.WebDialog;
import net.abstractfactory.plum.view.web.component.grid.WebCell;
import net.abstractfactory.plum.view.web.component.grid.WebGrid;
import net.abstractfactory.plum.view.web.component.grid.WebRow;
import net.abstractfactory.plum.view.web.component.input.WebAudioBox;
import net.abstractfactory.plum.view.web.component.input.WebButton;
import net.abstractfactory.plum.view.web.component.input.WebCheckBox;
import net.abstractfactory.plum.view.web.component.input.WebDateTimePicker;
import net.abstractfactory.plum.view.web.component.input.WebDropdownList;
import net.abstractfactory.plum.view.web.component.input.WebFileField;
import net.abstractfactory.plum.view.web.component.input.WebFilePicker;
import net.abstractfactory.plum.view.web.component.input.WebImageBox;
import net.abstractfactory.plum.view.web.component.input.WebImageButton;
import net.abstractfactory.plum.view.web.component.input.WebListBox;
import net.abstractfactory.plum.view.web.component.input.WebRadioBox;
import net.abstractfactory.plum.view.web.component.input.WebRadioItem;
import net.abstractfactory.plum.view.web.component.input.WebSpinBox;
import net.abstractfactory.plum.view.web.component.input.WebTextArea;
import net.abstractfactory.plum.view.web.component.input.WebTextBox;
import net.abstractfactory.plum.view.web.component.input.WebVideoBox;
import net.abstractfactory.plum.view.web.component.listview.WebHListView;
import net.abstractfactory.plum.view.web.component.listview.WebHListViewItem;
import net.abstractfactory.plum.view.web.component.listview.WebVListView;
import net.abstractfactory.plum.view.web.component.listview.WebVListViewItem;
import net.abstractfactory.plum.view.web.component.menu.WebMenu;
import net.abstractfactory.plum.view.web.component.menu.WebMenuBar;
import net.abstractfactory.plum.view.web.component.menu.WebMenuItem;
import net.abstractfactory.plum.view.web.component.menu.WebSubMenu;
import net.abstractfactory.plum.view.web.component.tree.WebTreeNodeView;
import net.abstractfactory.plum.view.web.component.tree.WebTreeView;
import net.abstractfactory.plum.view.web.component.web.WebHtmlView;
import net.abstractfactory.plum.view.web.layout.WebBox;
import net.abstractfactory.plum.view.web.layout.WebGridLayout;
import net.abstractfactory.plum.view.web.layout.WebTableHBox;
import net.abstractfactory.plum.view.web.layout.WebTableVBox;

/**
 * visit abstract view DOM and generate web content.
 * 
 * @author jack
 * 
 */
public class WebViewBuilder implements ComponentVisitor<WebComponent>, ConditionalComponentBuilderCollection {

	private static Logger logger = Logger.getLogger(WebViewBuilder.class);

	private static String PAGE_FILE = "template-bootstrap.html";

	private WebTemplatePage templatePage;

	private ListValueMap<Class, ConditionalWebComponentBuilder> componentBuilderMap = new ListValueMap<Class, ConditionalWebComponentBuilder>();

	/**
	 * current web view
	 */
	private WebPageView webView;
	private EventListener nonUserCausedViewChangeEventListener;

	private static long version = 0;

	private IdGenerationType idGenerationType;

	private boolean staticResource;

	public WebViewBuilder(final String topic, EventListener nonUserCausedViewChangeEventListener,
			IdGenerationType idGenerationType) {
		this.nonUserCausedViewChangeEventListener = nonUserCausedViewChangeEventListener;
		this.idGenerationType = idGenerationType;

		WebDialog.registerBuilder(this);
		WebMainWindowTitlePanel.registerBuilder(this);

	}

	/**
	 * create a new web page view.
	 * 
	 * @param component
	 * @param pushTopic
	 * @return
	 */
	public synchronized WebPageView build(Component rootComponent, Component activeComponent, String title,
			String pushTopic, ViewSyncMode viewSyncMode, boolean increaseVersion) {
		this.staticResource = (viewSyncMode == ViewSyncMode.POST_REDIRECT_GET);

		templatePage = new WebTemplatePage(PAGE_FILE);

		webView = new WebPageView(templatePage.getDocument().clone(), pushTopic, viewSyncMode);

		WebComponent rootWebComponent = (WebComponent) rootComponent.accept(this);

		webView.setRootContentWebComponent(rootWebComponent, title);

		if (activeComponent != null) {
			webView.setActiveElementId(activeComponent.getId());
		}
		// webView.setStateChangeEventListener(stateChangeEventListener);

		String ver = increaseVersion ? getNextVersion() : getCurrentVersion();
		webView.setVersion(ver);

		if (logger.isDebugEnabled()) {
			logger.debug("build web view:" + webView + " version:" + version);
		}

		return webView;
	}

	private String getCurrentVersion() {
		return String.format("%d", version);
	}

	private String getNextVersion() {
		version++;
		return String.format("%d", version);
	}

	private void addComponentToMap(Component abstractComp, WebComponent webComp) {
		webView.getWebComponentMap().put(webComp.getId(), webComp);

		webView.getComponentToWebComponentMap().put(abstractComp, webComp);

		if (webComp instanceof WebCheckBox)
			webView.getWebCheckBoxMap().put(webComp.getId(), (WebCheckBox) webComp);
	}

	protected void visitChildren(WebComponent htmlParent, Component uicomponent) {

		for (TreeNode node : uicomponent.getChildren()) {
			Component comp = (Component) node;
			AbstractWebComponent htmlComp = (AbstractWebComponent) comp.accept(this);

			if (htmlComp == null)
				throw new RuntimeException("web component not created:" + comp.getClass().getSimpleName());

			htmlParent.addChild(htmlComp);

		}

	}

	@Override
	public WebComponent visit(Component component) {

		// 3rd party component
		return WebComponentFactory.getInstace().create(this, component);

	}

	@Override
	public WebComponent visit(TextBox component) {
		return commonVisit(WebTextBox.class, TextBox.class, component);
	}

	@Override
	public WebComponent visit(TextArea component) {
		return commonVisit(WebTextArea.class, TextArea.class, component);
	}

	@Override
	public WebComponent visit(CheckBox component) {
		return commonVisit(WebCheckBox.class, CheckBox.class, component);

	}

	@Override
	public WebComponent visit(RadioBox component) {
		return commonVisit(WebRadioBox.class, RadioBox.class, component);
	}

	@Override
	public WebComponent visit(RadioItem component) {
		return commonVisit(WebRadioItem.class, RadioItem.class, component);
	}

	@Override
	public WebComponent visit(FilePicker component) {
		return commonVisit(WebFilePicker.class, FilePicker.class, component);
	}

	@Override
	public WebComponent visit(FileView component) {
		return commonVisit(WebFileField.class, FileView.class, component);
	}

	@Override
	public WebComponent visit(ImageBox component) {
		return commonVisit(WebImageBox.class, ImageBox.class, component);
	}

	@Override
	public WebComponent visit(AudioBox component) {
		return commonVisit(WebAudioBox.class, AudioBox.class, component);
	}

	@Override
	public WebComponent visit(VideoBox component) {
		return commonVisit(WebVideoBox.class, VideoBox.class, component);
	}

	@Override
	public WebComponent visit(DateTimePicker component) {
		return commonVisit(WebDateTimePicker.class, DateTimePicker.class, component);
	}

	@Override
	public WebComponent visit(SpinBox component) {
		return commonVisit(WebSpinBox.class, SpinBox.class, component);
	}

	@Override
	public WebComponent visit(Button component) {
		return commonVisit(WebButton.class, Button.class, component);

	}

	@Override
	public WebComponent visit(ImageButton component) {
		return commonVisit(WebImageButton.class, ImageButton.class, component);
	}

	@Override
	public WebComponent visit(Label component) {
		return commonVisit(WebLabel.class, Label.class, component);

	}

	@Override
	public WebComponent visit(DropdownList component) {
		return commonVisit(WebDropdownList.class, DropdownList.class, component);
	}

	@Override
	public WebComponent visit(ListBox component) {
		return commonVisit(WebListBox.class, ListBox.class, component);
	}

	@Override
	public WebComponent visit(Screen component) {
		return commonVisit(WebScreen.class, Screen.class, component);
	}

	@Override
	public WebComponent visit(Window component) {

		return commonVisit(WebWindow.class, Window.class, component);

	}

	@Override
	public WebComponent visit(Dialog component) {
		return commonVisit(WebDialog.class, Dialog.class, component);

	}

	@Override
	public WebComponent visit(Grid component) {
		return commonVisit(WebGrid.class, Grid.class, component);
	}

	@Override
	public WebComponent visit(Row component) {
		return commonVisit(WebRow.class, Row.class, component);
	}

	@Override
	public WebComponent visit(Cell component) {
		return commonVisit(WebCell.class, Cell.class, component);
	}

	@Override
	public WebComponent visit(ListView component) {
		WebComponent wc;
		if (component.getDirection() == Direction.HORIZONTAL) {
			wc = commonVisit(WebHListView.class, ListView.class, component);
		} else {
			wc = commonVisit(WebVListView.class, ListView.class, component);
		}

		return wc;
	}

	@Override
	public WebComponent visit(ListItemView component) {
		WebComponent wc;

		ListView listView = (ListView) component.getParentComponent();
		if (listView.getDirection() == Direction.HORIZONTAL) {
			wc = commonVisit(WebHListViewItem.class, ListItemView.class, component);
		} else {
			wc = commonVisit(WebVListViewItem.class, ListItemView.class, component);
		}

		return wc;
	}

	@Override
	public WebComponent visit(GridLayout component) {
		return commonVisit(WebGridLayout.class, GridLayout.class, component);

	}

	@Override
	public WebComponent visit(Box component) {
		return commonVisit(WebBox.class, Box.class, component);
	}

	@Override
	public WebComponent visit(HorizontalBox component) {
		return commonVisit(WebTableHBox.class, HorizontalBox.class, component);

	}

	@Override
	public WebComponent visit(VerticalBox component) {
		return commonVisit(WebTableVBox.class, VerticalBox.class, component);

	}

	@Override
	public WebComponent visit(MenuBar component) {

		return commonVisit(WebMenuBar.class, MenuBar.class, component);
	}

	@Override
	public WebComponent visit(Menu component) {
		setIdIfNull(component);

		int depth = component.getMenuDepth();

		AbstractWebComponent awc;
		if (depth == 1) {
			awc = new WebMenu(component.getId(), component, templatePage.getDocument());
		} else {
			awc = new WebSubMenu(component.getId(), component, templatePage.getDocument());
		}

		awc.init();

		addComponentToMap(component, awc);

		String hashOfOwn = awc.calcHash();
		awc.setOwnHash(hashOfOwn);

		visitChildren(awc, component);

		// hash of children IDs
		String hashOfChilren = awc.calcHashOfChildrenId();
		awc.setChildrenIdsHash(DigestUtils.md5Hex(hashOfChilren));

		return awc;
	}

	@Override
	public WebComponent visit(MenuItem component) {
		return commonVisit(WebMenuItem.class, MenuItem.class, component);
	}

	@Override
	public WebComponent visit(HyperLink component) {
		return commonVisit(WebHyperLink.class, HyperLink.class, component);
	}

	@Override
	public WebComponent visit(HtmlView component) {
		return commonVisit(WebHtmlView.class, HtmlView.class, component);
	}

	@Override
	public WebComponent visit(Panel component) {
		setIdIfNull(component);
		return commonVisit(WebPanel.class, Panel.class, component);

	}

	@Override
	public WebComponent visit(Tab component) {
		return commonVisit(WebTab.class, Tab.class, component);
	}

	@Override
	public WebComponent visit(Tabs component) {
		return commonVisit(WebTabs.class, Tabs.class, component);
	}

	@Override
	public WebComponent visit(TreeNodeView component) {
		return commonVisit(WebTreeNodeView.class, TreeNodeView.class, component);
	}

	@Override
	public WebComponent visit(TreeView component) {
		return commonVisit(WebTreeView.class, TreeView.class, component);
	}

	public WebComponent commonVisitWithoutPreprocess(Class<? extends WebComponent> webComponentClass,
			Component component) {
		return commonVisit(webComponentClass, null, component, false);
	}

	protected WebComponent commonVisit(Class<? extends WebComponent> webComponentClass,
			Class<? extends Component> componentClass, Component component) {
		return commonVisit(webComponentClass, componentClass, component, true);
	}

	private void setIdIfNull(Component component) {
		if (component.getId() == null) {
			IdGenerator idGenerator = IdGeneratorFactory.getGenerator(idGenerationType);
			String id = idGenerator.generate(component);
			component.setId(id);
		}
	}

	private WebComponent commonVisit(Class<? extends WebComponent> webComponentClass,
			Class<? extends Component> componentClass, Component component, boolean preprocess) {

		setIdIfNull(component);

		if (preprocess) {
			WebComponent wc = preprocess(componentClass, component);
			if (wc != null) {
				return wc;
			}
		}

		AbstractWebComponent awc = newComponent(webComponentClass, component.getId(), component);

		awc.setStaticResource(staticResource);

		awc.init();

		// AjaxWebAgent subscribes to StateChage event of every Component
		if (!component.existsEventListener(Component.EVENT_NON_USER_CAUSED_VIEW_CHANGE,
				nonUserCausedViewChangeEventListener))
			component.addEventListener(Component.EVENT_NON_USER_CAUSED_VIEW_CHANGE,
					nonUserCausedViewChangeEventListener);

		addComponentToMap(component, awc);

		// hash of its own (without children, at the moment, the node hash no
		// child yet.)
		String hashOfOwn = awc.calcHash();
		awc.setOwnHash(hashOfOwn);

		visitChildren(awc, component);

		// hash of children IDs
		String hashOfChilren = awc.calcHashOfChildrenId();
		awc.setChildrenIdsHash(DigestUtils.md5Hex(hashOfChilren));

		// webView.getNodeHashes().put(awc.getId(), nodeHash);

		return awc;
	}

	protected AbstractWebComponent newComponent(Class clazz, String id, Component abstractComponent) {

		try {
			if (AbstractTemplateWebComponent.class.isAssignableFrom(clazz)) {
				Constructor<AbstractWebComponent> ctor = clazz.getConstructor(String.class, Component.class,
						Element.class);

				return ctor.newInstance(id, abstractComponent, templatePage.getDocument());
			} else {
				Constructor<AbstractWebComponent> ctor = clazz.getConstructor(String.class, Component.class);
				return ctor.newInstance(id, abstractComponent);
			}

		} catch (Exception e) {
			throw new RuntimeException(e);
		}
	}

	/**
	 * try to build by conditional web view builder.
	 * 
	 * @param clazz
	 * @param component
	 * @return
	 */
	WebComponent preprocess(Class<? extends Component> clazz, Component component) {
		Object clientComp = null;

		List<ConditionalWebComponentBuilder> list = componentBuilderMap.get(clazz);

		if (list == null)
			return null;

		ConditionalWebComponentBuilder candidate = null;
		int count = 0;
		for (ConditionalWebComponentBuilder componentBuilder : list) {
			if (componentBuilder.isMatch(component)) {
				candidate = componentBuilder;
				count++;
			}
		}

		if (count > 1)
			throw new RuntimeException("only one component builder should be matched");
		else if (count == 1) {
			clientComp = candidate.visit(this, component);
		}

		return (WebComponent) clientComp;
	}

	@Override
	public void add(ConditionalWebComponentBuilder componentBuilder) {
		componentBuilderMap.add(componentBuilder.getComponentClass(), componentBuilder);

	}
}
